/* hub.c - Hub communication
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <ctype.h>		/* C89 */
#include <string.h>		/* C89 */
#include <stdlib.h>		/* C89 */
#include <sys/socket.h>		/* ? */
#include <netinet/in.h>		/* ? */
#include <arpa/inet.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <inttypes.h>		/* POSIX */
#include "xvasprintf.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "common/error.h"
#include "common/intparse.h"
#include "common/strleftcmp.h"
#include "microdc.h"

struct sockaddr_in hub_addr;
static uint32_t hub_recvq_last = 0;
char *hub_name = NULL;

bool
send_my_info(void)
{
    return hub_putf("$MyINFO $ALL %s %s<%s V:%s,M:%c,H:1/0/0,S:%d>$ $%s%c$%s$%" PRId64 "$|",
	  my_nick,
	  my_description, "++", "0.401", is_active ? 'A':'P', my_ul_slots,
	  my_speed,
	  1, /* level, '1' means normal, see DCTC Documentation/Documentation/VAR */
	  my_email,
	  my_share_size);
}

bool
say_user_completion_generator(const char *base, int state, DCCompletionEntry *ce)
{
    static HMapIterator it;

    if (state == 0)
    	hmap_iterator(hub_users, &it);

    while (it.has_next(&it)) {
    	DCUserInfo *ui = it.next(&it);
	if (strleftcmp(base, ui->nick) == 0) {
	    ce->str = xstrdup(ui->nick);
	    ce->input_append_single = ':';
	    return true;
	}
    }

    return false;
}

char *
user_completion_generator(const char *base, int state)
{
    static HMapIterator it;

    if (state == 0)
    	hmap_iterator(hub_users, &it);

    while (it.has_next(&it)) {
    	DCUserInfo *ui = it.next(&it);
	if (strleftcmp(base, ui->nick) == 0)
	    return xstrdup(ui->nick);
    }

    return NULL;
}

/* Create a new user info structure, representing a user on the hub.
 */
DCUserInfo *
user_info_new(const char *nick)
{
    DCUserInfo *info;
    char *ucname;

    info = xmalloc(sizeof(DCUserInfo));
    info->nick = xstrdup(nick);
    info->description = NULL;
    info->speed = NULL;
    info->level = 0;
    info->email = NULL;
    info->share_size = 0;
    info->active_state = DC_ACTIVE_UNKNOWN;
    info->download_queue = ptrv_new();
    info->refcount = 1;
    info->is_operator = false;
    info->info_quered = false;
    info->conn_count = 0;

    /* XXX Find existing connections to this user... */
    ucname = xasprintf("%s|%s", nick, "UL");
    info->conn[info->conn_count] = hmap_get(user_conns, ucname);
    if (info->conn[info->conn_count] != NULL)
    	info->conn_count++;
    free(ucname);
    ucname = xasprintf("%s|%s", nick, "DL");
    info->conn[info->conn_count] = hmap_get(user_conns, ucname);
    if (info->conn[info->conn_count] != NULL)
    	info->conn_count++;
    free(ucname);

    return info;
}

void
free_queued_file(DCQueuedFile *qf)
{
    free(qf->filename);
    free(qf->base_path);
    free(qf);
}

void
user_info_free(DCUserInfo *ui)
{
    ui->refcount--;
    if (ui->refcount == 0) {
	free(ui->nick);
	free(ui->description);
	free(ui->speed);
	free(ui->email);
	ptrv_foreach(ui->download_queue, (PtrVForeachCallback) free_queued_file);
	ptrv_free(ui->download_queue);
	free(ui);
    }
}

/* Put some data onto the connection, printf style.
 */
bool
hub_putf(const char *format, ...)
{
    va_list args;
    uint32_t oldcur;
    int res;

    oldcur = hub_sendq->cur;
    va_start(args, format);
    res = byteq_vappendf(hub_sendq, format, args);
    va_end(args);

    if (res < 0) {
    	warn("Cannot append to hub sendq - %s\n", errstr);
	hub_disconnect();
	return false;
    }

    dump_command("-->", hub_sendq->buf+oldcur, hub_sendq->cur-oldcur);

    res = byteq_write(hub_sendq, hub_socket);
    if (res == 0 || (res < 0 && errno != EAGAIN)) {
    	warn_socket_eof(res, "send to", "hub");
	hub_disconnect();
	return false;
    }

    if (oldcur == 0 && hub_sendq->cur > 0)
    	FD_SET(hub_socket, &write_fds);

    return true;
}

void
hub_connect(struct sockaddr_in *addr)
{
    hub_socket = socket(PF_INET, SOCK_STREAM, 0);
    if (hub_socket < 0) {
        warn("Cannot create socket - %s\n", errstr);
	hub_disconnect();
        return;
    }

    /* Set non-blocking I/O on socket. */
    if (fd_set_nonblock_flag(hub_socket, true) < 0) {
        warn("Cannot set socket flags - %s\n", errstr);
	hub_disconnect();
	return;
    }

    screen_putf("Connecting to hub on %s.\n", sockaddr_in_str(addr));
    if (connect(hub_socket, (struct sockaddr *) addr, sizeof(struct sockaddr_in)) < 0
    	    && errno != EINPROGRESS) {
        warn("Cannot connect - %s\n", errstr);
	hub_disconnect();
        return;
    }

    hub_addr = *addr;
    FD_SET(hub_socket, &write_fds);
    hub_state = DC_HUB_CONNECT;
}

void
hub_disconnect(void)
{
    if (hub_socket >= 0) {
    	if (hub_state > DC_HUB_CONNECT)
    	    screen_putf("Shutting down hub connection.\n");
    	FD_CLR(hub_socket, &read_fds);
        FD_CLR(hub_socket, &write_fds);
    	if (close(hub_socket) < 0)
	    warn("Cannot close - %s\n", errstr);
    	hub_socket = -1;
    }
    if (hub_users != NULL) {
	hmap_foreach(hub_users, user_info_free);
	hmap_clear(hub_users);
    }
    if (hub_sendq != NULL)
    	byteq_clear(hub_sendq);
    if (hub_recvq != NULL)
    	byteq_clear(hub_recvq);
    if (pending_userinfo != NULL) {
    	hmap_foreach(pending_userinfo, user_info_free);
    	hmap_clear(pending_userinfo);
    }
    hub_state = DC_HUB_DISCONNECTED;
}

static bool
check_state(char *buf, DCUserState state)
{
    if (hub_state != state) {
	warn("Received %s message in wrong state.\n", strtok(buf, " "));
	hub_disconnect();
	return false;
    }
    return true;
}

static bool
parse_ip_and_port(char *source, struct sockaddr_in *addr, uint16_t defport) /* XXX: move util.c */
{
    char *port;

    port = strchr(source, ':');
    if (port == NULL) {
    	if (defport == 0)
	    return false;
	addr->sin_port = htons(defport);
    } else {
	*port = '\0';
	port++;
	if (!parse_uint16(port, &addr->sin_port))
    	    return false;
        addr->sin_port = htons(addr->sin_port);
    }

    if (!inet_aton(source, &addr->sin_addr))
    	return false;

    addr->sin_family = AF_INET;
    return true;
}

static void
hub_handle_command(char *buf, uint32_t len)
{
    if (len >= 6 && strncmp(buf, "$Lock ", 6) == 0) {
        char *key;

    	if (!check_state(buf, DC_HUB_LOCK))
	    return;

        key = memmem(buf+6, len-6, " Pk=", 4);
        if (key == NULL) {
            screen_putf("Invalid $Lock message: Missing Pk value");
	    key = buf+len;
	}
        key = decode_lock(buf+6, key-buf-6, DC_CLIENT_BASE_KEY);
        if (!hub_putf("$Key %s|", key)) {
	    free(key);
	    return;
	}
	free(key);
        if (!hub_putf("$ValidateNick %s|", my_nick))
	    return;
	hub_state = DC_HUB_HELLO;
    }
    else if (strcmp(buf, "$GetPass") == 0) {
    	if (my_password == NULL) {
    	    screen_putf("Hub requires password.\n");
	    hub_disconnect();
	    return;
	}
	screen_putf("Sending password to hub.\n");
	if (!hub_putf("$MyPass %s|", my_password))
	    return;
    }
    else if (strcmp(buf, "$BadPass") == 0) {
    	screen_putf("Password not accepted.\n");
	hub_disconnect();
    }
    else if (strcmp(buf, "$LogedIn") == 0) {
    	screen_putf("You have received operator status.\n");
    }
    else if (len >= 9 && strncmp(buf, "$HubName ", 9) == 0) {
	free(hub_name);
	hub_name = xstrdup(buf+9);
        screen_putf("Hub name is %s.\n", hub_name);
    }
    else if (strcmp(buf, "$ValidateDenide") == 0) {
    	if (!check_state(buf, DC_HUB_HELLO))
	    return;
        /* DC++ disconnects immediately if this is received.
         * But shouldn't we give the client a chance to change the nick?
         * Also what happens if we receive this when correctly logged in?
         */
    	screen_putf("Hub did not accept nick. Nick may be in use.\n");
    	hub_disconnect();
    }
    else if (len >= 7 && strncmp(buf, "$Hello ", 7) == 0) {
        DCUserInfo *ui;

    	if (hub_state == DC_HUB_HELLO) {
	    if (strcmp(buf+7, my_nick) == 0) {
    	    	screen_putf("Nick accepted. You are now logged in.\n");
	    } else {
	    	/* This probably won't happen, but better safe... */
	    	free(my_nick);
		my_nick = xstrdup(buf+7);
    	    	screen_putf("Nick accepted but modified to %s. You are now logged in.\n", my_nick);
	    }

    	    ui = user_info_new(my_nick);
	    ui->info_quered = true; /* Hub is sending this automaticly */
    	    hmap_put(hub_users, ui->nick, ui);

	    if (!hub_putf("$Version 1,0091|"))
	    	return;
	    if (!hub_putf("$GetNickList|"))
	    	return;
	    /* XXX: hm, /0/0 should be Registered/Op, calculate this value. */
	    if (!send_my_info())
	    	return;
    	    hub_state = DC_HUB_LOGGED_IN;
	} else {
	    screen_putf("User %s logged in.\n", buf+7);
	    ui = user_info_new(buf+7);
            hmap_put(hub_users, ui->nick, ui);
	    if (!hub_putf("$GetINFO %s|", buf+7))
	    	return;
	}
    }
    else if (len >= 8 && strncmp(buf, "$MyINFO ", 8) == 0) {
	DCUserInfo *ui;
    	char *token;
	uint32_t len;

    	buf += 8;
    	token = strsep(&buf, " ");
	if (strcmp(token, "$ALL") != 0) {
	    screen_putf("Invalid $MyINFO message: Missing $ALL parameter, ignoring\n");
	    return;
	}

    	token = strsep(&buf, " ");
	if (token == NULL) {
	    screen_putf("Invalid $MyINFO message: Missing nick parameter, ignoring\n");
	    return;
	}
    	ui = hmap_get(hub_users, token);
	if (ui == NULL) {
	    screen_putf("Invalid $MyINFO message: Unknown user %s, ignoring\n", token);
	    return;
	}

    	token = strsep(&buf, "$");
	if (token == NULL) {
	    screen_putf("Invalid $MyINFO message: Missing description parameter, ignoring\n");
	    return;
	}
	free(ui->description);
	ui->description = xstrdup(token);

    	token = strsep(&buf, "$");
	if (token == NULL) {
	    screen_putf("Invalid $MyINFO message: Missing description separator, ignoring\n");
	    return;
	}

    	token = strsep(&buf, "$");
	if (token == NULL) {
	    screen_putf("Invalid $MyINFO message: Missing connection speed, ignoring\n");
	    return;
	}
	len = strlen(token);
	free(ui->speed);
	if (len == 0) {
	    ui->speed = xstrdup("");
	    ui->level = 0; /* XXX: or 1? acceptable level? */
	} else {
	    ui->speed = xstrndup(token, len-1);
	    ui->level = token[len-1];
	}

    	token = strsep(&buf, "$");
	if (token == NULL) {
	    screen_putf("Invalid $MyINFO message: Missing e-mail address, ignoring\n");
	    return;
	}
	free(ui->email);
	ui->email = xstrdup(token);

    	token = strsep(&buf, "$");
	if (token == NULL) {
	    screen_putf("Invalid $MyINFO message: Missing share size, ignoring\n");
	    return;
	}
	if (!parse_uint64(token, &ui->share_size)) {
	    screen_putf("Invalid $MyINFO message: Invalid share size, ignoring\n");
	    return;
	}

    	if (ui->active_state == DC_ACTIVE_RECEIVED_PASSIVE
	    	|| ui->active_state == DC_ACTIVE_KNOWN_ACTIVE)
	    ui->active_state = DC_ACTIVE_UNKNOWN;
	/* XXX: Now that user's active_state may have changed, try queue again? */
    }
    else if (strcmp(buf, "$HubIsFull") == 0) {
        screen_putf("Hub is full.\n");
        /* DC++ does not disconnect immediately. So I guess we won't either. */
	/* Maybe we should be expecting an "hub follow" message? */
	/* hub_disconnect(); */
    }
    else if (len >= 3 && buf[0] == '<') {
        screen_putf("Public: %s\n", buf);
    }
    else if (len >= 5 && strncmp(buf, "$To: ", 5) == 0) {
    	char *msg;
	msg = strchr(buf+5, '$');
	if (msg == NULL) {
	    screen_putf("Invalid $To message: Missing text separator, ignoring\n");
	    return;
        }
        *msg = '\0';
	msg++;
        screen_putf("Private: [%s] %s\n", buf+5, msg);
    }
    else if (len >= 13 && strncmp(buf, "$ConnectToMe ", 13) == 0) {
	struct sockaddr_in addr;

    	buf += 13;
	if (strsep(&buf, " ") == NULL) {
	    screen_putf("Invalid $ConnectToMe message: Missing or invalid nick\n");
	    return;
    	}
    	if (!parse_ip_and_port(buf, &addr, 0)) {
	    screen_putf("Invalid $ConnectToMe message: Invalid address specification.\n");
	    return;
        }

	screen_putf("Connecting to user on %s\n", sockaddr_in_str(&addr));
    	user_connection_new(&addr, -1);
    }
    else if (len >= 16 && strncmp(buf, "$RevConnectToMe ", 16) == 0) {
	char *nick;
	DCUserInfo *ui;

    	nick = strtok(buf+16, " ");
	if (nick == NULL) {
	    screen_putf("Invalid $RevConnectToMe message: Missing nick parameter\n");
	    return;
        }
    	if (strcmp(nick, my_nick) == 0) {
	    screen_putf("Invalid $RevConnectToMe message: Remote nick is our nick\n");
	    return;
	}
        ui = hmap_get(hub_users, nick);
        if (ui == NULL) {
	    screen_putf("Invalid $RevConnectToMe message: Unknown user %s, ignoring\n", nick);
	    return;
        }

	if (ui->conn_count >= DC_USER_MAX_CONN) {
	    screen_putf("No more connections to user %s allowed.\n", ui->nick);
	    return;
	}

    	if (!is_active) {
    	    if (ui->active_state == DC_ACTIVE_SENT_PASSIVE) {
    		screen_putf("User %s is also passive. Cannot establish connection.\n", ui->nick);
		/* We could set this to DC_ACTIVE_UNKNOWN too. This would mean
		 * we would keep sending RevConnectToMe next time the download
		 * queue is modified or some timer expired and told us to retry
		 * download. The remote would then send back RevConnectToMe
		 * and this would happen again. This way we would try again -
		 * maybe remote has become active since last time we checked?
		 * But no - DC++ only replies with RevConnectToMe to our first
		 * RevConnectToMe. After that it ignores them all.
		 */
   		ui->active_state = DC_ACTIVE_RECEIVED_PASSIVE;
		if (hmap_remove(pending_userinfo, ui->nick) != NULL)
    	    	    ui->refcount--;
		return;
	    }
    	    if (ui->active_state != DC_ACTIVE_RECEIVED_PASSIVE) {
    	    	/* Inform remote that we are also passive. */
    		if (!hub_putf("$RevConnectToMe %s %s|", my_nick, ui->nick))
		    return;
	    }
	}
   	ui->active_state = DC_ACTIVE_RECEIVED_PASSIVE;

   	if (!hub_connect_user(ui))
	    return;
    }
    else if (len >= 10 && strncmp(buf, "$NickList ", 10) == 0) {
    	char *nick;
	char *end;

	for (nick = buf+10; (end = strstr(nick, "$$")) != NULL; nick = end+2) {
            DCUserInfo *ui;

    	    *end = '\0';
	    ui = hmap_get(hub_users, nick);
	    if (ui == NULL) {
	    	ui = user_info_new(nick);
                hmap_put(hub_users, ui->nick, ui);
	    }
	    if (!ui->info_quered) {
		if (!hub_putf("$GetINFO %s %s|", nick, my_nick))
		    return;
		ui->info_quered = true;
	    }
	}
    }
    else if (len >= 8 && strncmp(buf, "$OpList ", 8) == 0) {
    	char *nick;
	char *end;

	for (nick = buf+10; (end = strstr(nick, "$$")) != NULL; nick = end+2) {
            DCUserInfo *ui;

    	    *end = '\0';
	    ui = hmap_get(hub_users, nick);
	    if (ui == NULL) {
                ui = user_info_new(nick);
                hmap_put(hub_users, ui->nick, ui);
	    }
	    if (!ui->info_quered) {
	    	if (!hub_putf("$GetINFO %s %s|", nick, my_nick))
		    return;
		ui->info_quered = true;
	    }
    	    ui->is_operator = true;
	}
    }
    else if (len >= 6 && strncmp(buf, "$Quit ", 6) == 0) {
    	DCUserInfo *ui;
    	screen_putf("User %s quits.\n", buf+6);
	ui = hmap_remove(hub_users, buf+6);
	if (ui == NULL) {
	    screen_putf("Invalid $Quit message: Unknown user %s.\n", buf+6);
	    return;
	}
	user_info_free(ui);
    }
    else if (len >= 8 && strncmp(buf, "$Search ", 8) == 0) {
    	char *source;
	DCSearchSelection sel;

    	buf += 8;
	source = strsep(&buf, " ");
	if (source == NULL) {
	    screen_putf("Invalid $Search message: Missing source specification.\n");
	    return;
	}
	if (!parse_search_selection(buf, &sel)) {
	    screen_putf("Invalid $Search message: Invalid search specification.\n");
	    return;
	}
	if (strncmp(source, "Hub:", 4) == 0) {
	    DCUserInfo *ui = hmap_get(hub_users, source+4);
	    if (ui == NULL) {
		screen_putf("Invalid $Search message: Unknown user %s.\n", source+4);
		free(sel.patterns);
		return;
	    }
	    if (strcmp(ui->nick, my_nick) == 0)
	    	return;
	    perform_inbound_search(&sel, ui, NULL);
	} else {
            struct sockaddr_in addr;
    	    if (!parse_ip_and_port(source, &addr, DC_CLIENT_UDP_PORT)) {
		screen_putf("Invalid $Search message: Invalid address specification.\n");
		free(sel.patterns);
		return;
	    }
	    if (local_addr.sin_addr.s_addr == addr.sin_addr.s_addr
	    	    && listen_port == ntohs(addr.sin_port))
    	    	return;
	    perform_inbound_search(&sel, NULL, &addr);
	}
    }
    else if (len >= 4 && strncmp(buf, "$SR ", 4) == 0) {
    	handle_search_result(buf, len);
    }
    else if (len == 0) {
    	/* Ignore empty commands. */
    }
}

/* This function is called by handle_connection_fd_event when there is input
 * available on the connection socket.
 */
void
hub_input_available(void)
{
    int start = 0;
    int c;
    int res;

    res = byteq_read(hub_recvq, hub_socket);
    if (res == 0 || (res < 0 && errno != EAGAIN)) {
    	warn_socket_eof(res, "receive from", "hub");
	hub_disconnect();
	return;
    }

    for (c = hub_recvq_last; c < hub_recvq->cur; c++) {
        if (hub_recvq->buf[c] == '|') {
            /* Got a complete command. */
	    if (c - start > 0)
	    	dump_command("<--", hub_recvq->buf + start, c - start + 1);
            hub_recvq->buf[c] = '\0'; /* Just to be on the safe side... */
            hub_handle_command(hub_recvq->buf + start, c - start);
            start = c+1;
	    if (hub_socket < 0)
    	    	return;
        }
    }

    if (start != 0)
        byteq_remove(hub_recvq, start);

    hub_recvq_last = hub_recvq->cur;
}

void
hub_now_writable(void)
{
    if (hub_state == DC_HUB_CONNECT) {
	int error;
        int size = sizeof(error);
	socklen_t addr_len;

        if (getsockopt(hub_socket, SOL_SOCKET, SO_ERROR, &error, &size) < 0) {
            warn("Cannot get error status - %s\n", errstr);
	    hub_disconnect();
	    return;
        }
        if (error != 0) { /* The connect call on the socket failed. */
	    warn("Cannot connect - %s\n", strerror(error) /* not errno! */);
	    hub_disconnect();
	    return;
	}

	addr_len = sizeof(local_addr);
	if (getsockname(hub_socket, (struct sockaddr *) &local_addr, &addr_len) < 0) {
    	    warn("Cannot get socket address - %s\n", errstr);
	    hub_disconnect();
            return;
	}

    	screen_putf("Connected to hub from %s.\n", sockaddr_in_str(&local_addr));

    	FD_CLR(hub_socket, &write_fds);
	FD_SET(hub_socket, &read_fds);
	hub_state = DC_HUB_LOCK;
    }
    else {
    	int res;
    
	if (hub_sendq->cur > 0) {
	    res = byteq_write(hub_sendq, hub_socket);
	    if (res == 0 || (res < 0 && errno != EAGAIN)) {
    	    	warn_socket_eof(res, "send to", "hub");
		hub_disconnect();
		return;
            }
	}
	if (hub_sendq->cur == 0)
	    FD_CLR(hub_socket, &write_fds);
    }
}

/* This function tries to make a connection to a user, or ask them to
 * connect to us.
 * 
 * Before calling this function, make sure we are not connected to the
 * user already.
 *
 * This function makes sure an unanswerred (Rev)ConnectToMe hasn't been
 * sent previously.
 *
 * This function is the only place that is allowed to send $ConnectToMe.
 * $RevConnectToMe may be set by one other place (when $RevConnectToMe
 * was received and we did not previously send $RevConnectToMe).
 */
bool
hub_connect_user(DCUserInfo *ui)
{
    if (is_active) {
    	if (ui->active_state == DC_ACTIVE_SENT_ACTIVE) {
    	    warn("ConnectToMe already sent to user %s. Waiting.\n", ui->nick);
	    return true;
	}
    	if (!hub_putf("$ConnectToMe %s %s:%u|",
	    	ui->nick,
		inet_ntoa(local_addr.sin_addr),
		listen_port))
    	    return false;
	ui->active_state = DC_ACTIVE_SENT_ACTIVE;
    } else {
    	if (ui->active_state == DC_ACTIVE_SENT_PASSIVE) {
    	    screen_putf("RevConnectToMe already sent to user %s. Waiting.\n", ui->nick);
	    return true;
	}
    	if (ui->active_state == DC_ACTIVE_RECEIVED_PASSIVE) {
    	    screen_putf("User %s is also passive. Cannot communicate.\n", ui->nick);
	    return true;
	}
    	if (!hub_putf("$RevConnectToMe %s %s|", my_nick, ui->nick))
	    return false;
   	ui->active_state = DC_ACTIVE_SENT_PASSIVE;
    }
    /* hmap_put returns the old value */
    if (hmap_put(pending_userinfo, ui->nick, ui) == NULL)
    	ui->refcount++;
    return true;
}
