/* search.c - Search support
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <ctype.h>		/* C89 */
#include <assert.h>		/* ? */
#include <string.h>		/* C89 */
#include <arpa/inet.h>		/* ? */
#include <sys/socket.h>		/* ? */
#include <netinet/in.h>		/* ? */
#include <inttypes.h>		/* ? */
#include <time.h>		/* ? */
#include "xalloc.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "common/strbuf.h"
#include "common/intparse.h"
#include "common/common.h"
#include "microdc.h"

#define MAX_RESULTS_ACTIVE 10		/* Max number of search results to send to active users */
#define MAX_RESULTS_PASSIVE 5		/* Max number of search results to send to passive users */
#define SEARCH_TIME_THRESHOLD 60	/* Add no more results to searches after this many seconds elapsed */

PtrV *our_searches;

static char *extensions[] = { // NULL means match any extension
    /* ANY */ NULL, 	
    /* AUDIO */ "mp3/mp2/wav/au/rm/mid/sm",
    /* COMPRESSED */ "zip/arj/rar/lzh/gz/z/arc/pak",
    /* DOCUMENTS */ "doc/txt/wri/pdf/ps/tex",
    /* EXECUTABLES */ "pm/exe/bat/com",
    /* PICTURES */ "gif/jpg/jpeg/bmp/pcx/png/wmf/psd",
    /* VIDEO */ "mpg/mpeg/avi/asf/mov",
    /* FOLDERS */ NULL,
    /* CHECKSUM */ NULL,
};

static void
search_string_new(DCSearchString *sp, const char *p, int len)
{
    int c;

    sp->str = xstrndup(p, len);
    for (c = 0; c < len; c++)
    	sp->str[c] = tolower(sp->str[c]);
    sp->len = len;

    for (c = 0; c < 256; c++)
    	sp->delta[c] = len+1;
    for (c = 0; c < len; c++)
    	sp->delta[(uint8_t) sp->str[c]] = len-c;
}

static void
parse_search_strings(char *str, DCSearchSelection *ss)
{
    uint32_t c;
    char *t1;
    char *t2;

    ss->patterncount = 0;

    for (t1 = str; (t2 = strchr(t1, '$')) != NULL; t1 = t2+1) {
    	if (t1 != str && t1[-1] != '$')
	    ss->patterncount++;
    }
    if (*t1)
        ss->patterncount++;

    ss->patterns = xmalloc(sizeof(DCSearchString) * ss->patterncount);

    c = 0;
    for (t1 = str; (t2 = strchr(t1, '$')) != NULL; t1 = t2+1) {
    	if (t1 != str && t1[-1] != '$') {
    	    search_string_new(ss->patterns+c, t1, t2-t1);
	    c++;
	}
    }
    if (*t1)
       search_string_new(ss->patterns+c, t1, strlen(t1));
}

static bool
match_file_extension(const char *filename, DCSearchDataType type)
{
    char *ext;
    char *t1;
    char *t2;

    t1 = extensions[type];
    if (t1 == NULL)
	return true;

    ext = strrchr(filename, '.');
    if (ext == NULL)
	return false;
    ext++;

    for (; (t2 = strchr(t1, '/')) != NULL; t1 = t2+1) {
	if (strncasecmp(ext, t1, t2-t1) == 0)
	    return true;
    }
    if (strcasecmp(ext, t1) == 0)
	return true;

    return false;
}

bool
parse_search_selection(char *str, DCSearchSelection *data)
{
    char sizeres;
    char sizemin;
    char *sizestr;
    uint64_t size;
    char *datatype;

    if (str[0] != 'T' && str[0] != 'F')
    	return false;
    if (str[1] != '?')
    	return false;
    if (str[2] != 'T' && str[2] != 'F')
    	return false;
    if (str[3] != '?')
    	return false;
    sizeres = str[0];
    sizemin = str[2];

    str += 4;
    sizestr = strsep(&str, "?");
    if (sizestr == NULL)
    	return false;
    if (!parse_uint64(sizestr, &size))
    	return false;
    datatype = strsep(&str, "?");
    if (datatype == NULL || datatype[0] < '1' || datatype[0] > '9' || datatype[1] != '\0')
    	return false;
    if (*str == '\0')
    	return false;
    if (strlen(str) >= 1 << 16) /* Needed for delta match */
    	return false;

    if (sizeres) {
    	if (sizemin) {
    	    data->size_min = size;
    	    data->size_max = UINT64_MAX;
	} else {
    	    data->size_min = 0;
    	    data->size_max = size;
	}
    } else {
    	data->size_min = 0;
    	data->size_max = UINT64_MAX;
    }
    data->datatype = datatype[0]-'1';

    parse_search_strings(str, data);

    return true;
}

static bool
match_search_pattern(const char *t, DCSearchString *pattern)
{
    const char *end;
    uint32_t tlen;

    tlen = strlen(t);
    if (tlen < pattern->len)
    	return false;

    end = t + tlen - pattern->len + 1;
    while (t < end) {
    	uint32_t i = 0;
	for (; pattern->str[i] && pattern->str[i] == tolower(t[i]); i++);
	if (pattern->str[i] == '\0')
	    return true;
	t += pattern->delta[(uint8_t) tolower(t[pattern->len])];
    }
    return false;
}

static bool
match_search_patterns(const char *text, DCSearchSelection *data)
{
    uint32_t c;
    for (c = 0; c < data->patterncount; c++) {
	if (match_search_pattern(text, data->patterns+c))
	    return 1;
    }
    return 0;
}

static void
append_result(DCFileList *node, DCUserInfo *ui, struct sockaddr_in *addr)
{
    StrBuf *sb;
    char *lpath;
    char *rpath;
    int free_slots;

    sb = strbuf_new();
    lpath = filelist_get_path(node);
    rpath = translate_local_to_remote(lpath);
    free(lpath);

    free_slots = used_ul_slots > my_ul_slots ? 0 : my_ul_slots-used_ul_slots;

    strbuf_appendf(sb, "$SR %s %s", my_nick, rpath);
    if (node->type == DC_TYPE_REG)
    	strbuf_appendf(sb, "\x05%" PRId64, node->u.reg.size);
    strbuf_appendf(sb, " %d/%d\x05%s (%s)",
    	    free_slots, my_ul_slots, hub_name,
	    sockaddr_in_str(&hub_addr));
    if (ui != NULL)
    	strbuf_appendf(sb, "\x05%s", ui->nick);
    strbuf_append(sb, "|");

    if (ui != NULL) {
    	hub_putf("%s", sb->buf); /* want hub_put here */
    } else {
    	add_search_result(addr, sb->buf, strbuf_length(sb));
    }
    
    strbuf_free(sb);
}

static int
filelist_search(DCFileList *node, DCSearchSelection *data, int maxresults, DCUserInfo *ui, struct sockaddr_in *addr)
{
    assert(maxresults > 0);

    if (node->type == DC_TYPE_REG) {
    	if (data->datatype == DC_SEARCH_FOLDERS || data->datatype == DC_SEARCH_CHECKSUM) /* TTH not supported yet */
	    return 0;
    	if (node->u.reg.size < data->size_min)
	    return 0;
	if (node->u.reg.size > data->size_max)
	    return 0;
	if (!match_search_patterns(node->name, data))
	    return 0;
	if (!match_file_extension(node->name, data->datatype))
	    return 0;
	append_result(node, ui, addr);
	return 1;
    }

    if (node->type == DC_TYPE_DIR) {
    	HMapIterator it;
    	int curresults = 0;

    	if (data->datatype == DC_SEARCH_ANY || data->datatype == DC_SEARCH_FOLDERS) {
	    if (match_search_patterns(node->name, data)) {
    	    	append_result(node, ui, addr);
    	    	curresults++;
		if (curresults >= maxresults)
	    	    return curresults;
	    }
	}

	hmap_iterator(node->u.dir.children, &it);
	while (it.has_next(&it)) {
    	    DCFileList *subnode = it.next(&it);
	    curresults += filelist_search(subnode, data, maxresults-curresults, ui, addr);
	    if (curresults >= maxresults)
    	    	break;
    	}
	
	return curresults;
    }

    return 0;
}

//XXX: assumes our_filelist != NULL
bool
perform_inbound_search(DCSearchSelection *data, DCUserInfo *ui, struct sockaddr_in *addr)
{
    int maxresults;
    int curresults;

    maxresults = (ui == NULL ? MAX_RESULTS_ACTIVE : MAX_RESULTS_PASSIVE);
    curresults = filelist_search(our_filelist, data, maxresults, ui, addr);

    if (debug_level > 0) {
	if (curresults > 0) {
	    if (ui != NULL) {
    		screen_putf(_("Sent %d/%d search results to %s\n"), curresults, maxresults, ui->nick);
	    } else {
    		screen_putf(_("Sent %d/%d search results to %s\n"), curresults, maxresults, sockaddr_in_str(addr));
	    }
	} else {
    	    screen_putf(_("No search results.\n"));
	}
    }

    return false;
}

/* Parse a $SR message into a DCSearchResponse */
static DCSearchResponse *
parse_search_response(char *buf, uint32_t len)
{
    DCSearchResponse *sr;
    DCUserInfo *ui;
    char *token;
    char *filename;
    uint64_t filesize;
    DCFileType filetype;
    uint32_t slots_free;
    uint32_t slots_total;
    char *hub_name;
    struct sockaddr_in hub_addr;

    if (strncmp(buf, "$SR ", 4) != 0)
    	return NULL; /* Invalid $SR message: Not starting with $SR. */

    buf += 4;
    token = strsep(&buf, " ");
    if (token == NULL)
    	return NULL; /* Invalid $SR message: Missing user. */
    ui = hmap_get(hub_users, token);
    if (ui == NULL)
    	return NULL; /* Invalid $SR message: Unknown user. */

    filename = strsep(&buf, "/");
    if (filename == NULL)
    	return NULL; /* Invalid $SR message: Missing filename. */
    buf[-1] = '/';
    for (; filename < buf && *buf != ' '; buf--);
    if (filename == buf)
    	return NULL; /* Invalid $SR message: Missing free slots. */
    *buf = '\0';
    buf++;
    token = strchr(filename, '\x05');
    if (token == NULL) {
	filetype = DC_TYPE_DIR;
    	filesize = 0;
    } else {
    	filetype = DC_TYPE_REG;
	*token = '\0';
	if (!parse_uint64(token+1, &filesize))
	    return NULL; /* Invalid $SR message: Invalid file size */
    }

    token = strsep(&buf, "/");
    assert(token != NULL);
    if (!parse_uint32(token, &slots_free))
    	return NULL; /* Invalid $SR message: Invalid free slots. */
    token = strsep(&buf, "\x05");
    if (token == NULL)
    	return NULL; /* Invalid $SR message: Missing total slots. */
    if (!parse_uint32(token, &slots_total))
    	return NULL; /* Invalid $SR message: Invalid total slots. */

    hub_name = buf;
    token = strrchr(buf, '(');
    if (token == NULL)
    	return NULL; /* Invalid $SR message: Missing hub address. */
    *token = '\0';
    buf = token+1;
    token = strchr(buf, ')');
    if (token == NULL)
    	return NULL; /* Invalid $SR message: Missing hub address. */
    *token = '\0';

    token = strchr(buf, ':');
    if (token != NULL) {
    	*token = '\0';
	if (!parse_uint16(token+1, &hub_addr.sin_port))
    	    return NULL; /* Invalid $SR message: Invalid hub port. */
	hub_addr.sin_port = htons(hub_addr.sin_port);
    } else {
    	hub_addr.sin_port = htons(DC_HUB_TCP_PORT);
    }
    if (!inet_aton(buf, &hub_addr.sin_addr))
	return NULL; /* Invalid $SR message: Invalid hub address. */

    sr = xmalloc(sizeof(DCSearchResponse));
    sr->userinfo = ui;
    ui->refcount++;
    sr->filename = xstrdup(filename);
    sr->filetype = filetype;
    sr->filesize = filesize;
    sr->slots_free = slots_free;
    sr->slots_total = slots_total;
    sr->hub_name = xstrdup(hub_name);
    sr->hub_addr = hub_addr;
    sr->refcount = 1;

    return sr;
}

static void
free_search_response(DCSearchResponse *sr)
{
    sr->refcount--;
    if (sr->refcount == 0) {
	user_info_free(sr->userinfo);
	free(sr->filename);
	free(sr->hub_name);
	free(sr);
    }
}

static bool
match_selection_against_response(DCSearchSelection *ss, DCSearchResponse *sr)
{
    if (sr->filetype == DC_TYPE_DIR) {
    	if (ss->datatype != DC_SEARCH_ANY && ss->datatype != DC_SEARCH_FOLDERS)
    	    return false;
    	if (ss->datatype == DC_SEARCH_CHECKSUM)
    	    return false;
	if (!match_search_patterns(sr->filename, ss))
	    return false;
	return true;
    } else {
    	if (ss->datatype == DC_SEARCH_FOLDERS)
    	    return false;
	if (ss->datatype == DC_SEARCH_CHECKSUM) /* TTH not supported yet */
	    return false;
    	if (sr->filesize < ss->size_min || sr->filesize > ss->size_max)
	    return false;
	if (!match_search_patterns(sr->filename, ss))
	    return false;
	if (!match_file_extension(sr->filename, ss->datatype))
	    return false;
	return true;
    }
}

static int
compare_search_selection(DCSearchSelection *s1, DCSearchSelection *s2)
{
    uint32_t c;

    COMPARE_RETURN(s1->size_min, s2->size_min);
    COMPARE_RETURN(s1->size_max, s2->size_max);
    COMPARE_RETURN(s1->datatype, s2->datatype);
    COMPARE_RETURN(s1->patterncount, s2->patterncount);
    for (c = 0; c < s1->patterncount; c++) {
    	COMPARE_RETURN(s1->patterns[c].len, s2->patterns[c].len);
    	COMPARE_RETURN_FUNC(strncmp(s1->patterns[c].str, s2->patterns[c].str, s1->patterns[c].len));
    }

    return 0;
}

bool
add_search_request(char *args)
{
    DCSearchSelection sel;
    DCSearchRequest *sr = NULL;
    uint32_t c;
    time_t now;

    for (c = 0; args[c] != '\0'; c++) {
    	if (args[c] == '|' || args[c] == ' ')
	    args[c] = '$';
    }

    sel.size_min = 0;
    sel.size_max = UINT64_MAX;
    sel.datatype = DC_SEARCH_ANY;
    parse_search_strings(args, &sel);

    for (c = 0; c < our_searches->cur; c++) {
    	sr = our_searches->buf[c];
    	if (compare_search_selection(&sel, &sr->selection) == 0)
	    break;
    }

    if (now < 0 && time(&now) < 0) {
	warn(_("Cannot get current time - %s\n"), errstr);
	free(sel.patterns);
	return false;
    }

    if (c < our_searches->cur) {
    	screen_putf(_("Reissuing search %d.\n"), c+1);
    	free(sel.patterns);
	sr->issue_time = now;
    } else {
    	screen_putf(_("Issuing new search with index %d.\n"), c+1);
	sr = xmalloc(sizeof(DCSearchRequest));
	sr->selection = sel;
	sr->responses = ptrv_new();
	sr->issue_time = now;
	ptrv_append(our_searches, sr);
    }

    if (is_active) {
	hub_putf("$Search %s:%u F?F?0?1?%s|",
    		inet_ntoa(local_addr.sin_addr), listen_port, args);
    } else {
	hub_putf("$Search Hub:%s F?F?0?1?%s|", my_nick, args);
    }

    return true;
}

void
free_search_request(DCSearchRequest *sr)
{
    free(sr->selection.patterns);
    ptrv_foreach(sr->responses, (PtrVForeachCallback) free_search_response);
    ptrv_free(sr->responses);
}

void
handle_search_result(char *buf, uint32_t len)
{
    DCSearchResponse *sr;
    uint32_t c;
    time_t now;

    if (time(&now) < 0) {
	warn(_("Cannot get current time - %s\n"), errstr);
	return;
    }

    sr = parse_search_response(buf, len);
    if (sr == NULL)
    	return;

    for (c = 0; c < our_searches->cur; c++) {
    	DCSearchRequest *sd = our_searches->buf[c];

	if (sd->issue_time + SEARCH_TIME_THRESHOLD <= now)
	    continue;

   	if (match_selection_against_response(&sd->selection, sr)) {
	    char *fmt;

	    ptrv_append(sd->responses, sr);
	    fmt = ngettext("Added search result to search %d (now %d result).\n",
			   "Added search result to search %d (now %d results).\n",
			   sd->responses->cur);
	    screen_putf(fmt, c+1, sd->responses->cur);
	    sr->refcount++;
	}
    }

    free_search_response(sr);
}
