/* microdc.h - Header file for the project
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef MICRODC_H
#define MICRODC_H

#include <sys/types.h>
#include <stdbool.h>
#include <stdint.h>
#include <netinet/in.h>
#include <time.h>
#include "common/byteq.h"
#include "common/ptrv.h"
#include "common/error.h"
#include "common/hmap.h"
#include "ipc.h"

#define DC_CLIENT_BASE_KEY 5
#define DC_HUB_TCP_PORT 411
#define DC_CLIENT_UDP_PORT 412
#define DC_USER_MAX_CONN 2

#define SEARCH_TIME_THRESHOLD 60        /* Add no more results to searches after this many seconds elapsed */

typedef enum {
    DC_DF_DEBUG			= 1 << 0, /* Various debug messages */
    DC_DF_JOIN_PART		= 1 << 1, /* Joins and Quits messages */
    DC_DF_PUBLIC_CHAT		= 1 << 2, /* Public chat */
    DC_DF_SEARCH_RESULTS	= 1 << 3, /* Incoming search results */
    DC_DF_UPLOAD		= 1 << 4, /* "Upload complete" message (not errors) */
    DC_DF_DOWNLOAD		= 1 << 5, /* "Download complete" message (not errors) */
} DCDisplayFlag;

typedef enum {
    DC_CPL_SIMPLE,
    DC_CPL_FILE,
    DC_CPL_CUSTOM,
} DCCompletionFunctionType;

typedef enum {
    DC_TF_NORMAL,		/* Normal file transfer */
    DC_TF_LIST,			/* Transfer of MyList.DcLst */
} DCTransferFlag;

typedef enum {
    DC_DIR_UNKNOWN,
    DC_DIR_SEND,
    DC_DIR_RECEIVE,
} DCTransferDirection;

typedef enum {
    DC_ACTIVE_UNKNOWN,	    	/* unknown, the default */
    DC_ACTIVE_KNOWN_ACTIVE, 	/* we got a ConnectToMe from user */
    DC_ACTIVE_RECEIVED_PASSIVE,	/* we got a RevConnectToMe from user */
    DC_ACTIVE_SENT_PASSIVE, 	/* we sent a RevConnectToMe to user, but haven't received anything yet */
    DC_ACTIVE_SENT_ACTIVE, 	/* we sent a ConnectToMe to user, but haven't received a connection yet */
} DCActiveState;

typedef enum {
    DC_USER_DISCONNECTED,
    DC_USER_CONNECT,	    	/* Waiting for connect() to complete or socket to be writable */
    DC_USER_MYNICK = 35,	/* Waiting for $MyNick */
    DC_USER_LOCK,  	    	/* Waiting for $Lock */
    DC_USER_DIRECTION,     	/* Waiting for $Direction */
    DC_USER_KEY,                /* Waiting for $Key */
    DC_USER_GET,   	    	/* Waiting for $Get (only when sending) */
    DC_USER_SEND_GET,  	    	/* Waiting for $Send or $Get (only when sending) */
    DC_USER_FILE_LENGTH,   	/* Waiting for $FileLength (only when receiving) */
    DC_USER_DATA_RECV,  	/* Waiting for file data (only when receiving) */
    DC_USER_DATA_SEND,
} DCUserState;

typedef enum {
    DC_HUB_DISCONNECTED,
    DC_HUB_CONNECT,	    	/* Waiting for connect() to complete or socket to be writable */
    DC_HUB_LOCK,	    	/* Waiting for $Lock */
    DC_HUB_HELLO,		/* Waiting for $Hello */
    DC_HUB_LOGGED_IN,		/* Correctly logged in */
} DCHubState;

typedef enum {
    DC_TYPE_DIR,
    DC_TYPE_REG,
} DCFileType;

typedef enum {
    DC_MSG_SCREEN_PUT,
    DC_MSG_NEXT_DOWNLOAD,
    DC_MSG_WANT_DOWNLOAD,
    DC_MSG_VALIDATE_DIR,
    DC_MSG_VALIDATE_NICK,
    DC_MSG_RESOLVE_FILE, /* for download or upload */
    DC_MSG_STATUS,
    DC_MSG_DISCONNECT,
    DC_MSG_TRANSFER_STARTING,
    DC_MSG_TRANSFER_FINISHED,
} DCUserMsgId;

typedef enum {
    DC_SEARCH_ANY,
    DC_SEARCH_AUDIO,
    DC_SEARCH_COMPRESSED,
    DC_SEARCH_DOCUMENTS,
    DC_SEARCH_EXECUTABLES,
    DC_SEARCH_PICTURES,
    DC_SEARCH_VIDEO,
    DC_SEARCH_FOLDERS,
    DC_SEARCH_CHECKSUM,     /* DC++: TTH at the moment */
} DCSearchDataType;

typedef struct _DCUserConn DCUserConn;
typedef struct _DCUserInfo DCUserInfo;
typedef struct _DCFileList DCFileList;
typedef struct _DCCompletionEntry DCCompletionEntry;
typedef struct _DCSearchSelection DCSearchSelection;
typedef struct _DCSearchString DCSearchString;
typedef struct _DCSearchRequest DCSearchRequest;
typedef struct _DCUDPMessage DCUDPMessage;
typedef struct _DCSearchResponse DCSearchResponse;
typedef struct _DCQueuedFile DCQueuedFile;

struct _DCSearchResponse {
    uint32_t refcount;
    DCUserInfo *userinfo;
    char *filename; // local namespace
    DCFileType filetype;
    uint64_t filesize;
    uint32_t slots_free;
    uint32_t slots_total;
    char *hub_name;
    struct sockaddr_in hub_addr;
};

struct _DCUDPMessage {
    struct sockaddr_in addr;
    uint32_t len;
    char data[0];
};

struct _DCSearchString {
    char *str;
    uint32_t len;
    uint16_t delta[256];
};

struct _DCSearchSelection {
    uint64_t size_min;
    uint64_t size_max;
    DCSearchDataType datatype;
    uint32_t patterncount;
    DCSearchString *patterns;
};

struct _DCSearchRequest {
    DCSearchSelection selection;
    time_t issue_time;
    PtrV *responses;
};

struct _DCCompletionEntry {
    char *str;
    char input_char;
    char *input_format;
    char *display_format;
    char *input_single_format;
    char *input_single_full_format;
//  char input_char; /* character to append to input line */
//  char display_char; /* character to display */
//  char input_append_single; /* append to line buffer when this entry is the only match */
//  char input_append_single_full; /* as above, but only when the full string was entered before */
};

struct _DCFileList {
    DCFileList *parent;
    char *name;
    DCFileType type;
    union {
    	struct {
    	    uint64_t size;
	    char tth[39];
	} reg;
	struct {
    	    uint64_t totalsize;
	    HMap *children;
	} dir;
    } u;
};

struct _DCQueuedFile {
    char *filename;  /* XXX: should make this relative, not absolute */
    char *base_path; /* XXX: so that catfiles(base_path, filename) works. */
    DCTransferFlag flag;
};

struct _DCUserInfo {
    char *nick;
    char *description;
    char *speed;
    uint8_t level;
    char *email;
    uint64_t share_size;
    DCActiveState active_state;
    PtrV *download_queue;   // XXX LList?
    uint16_t slot_granted;
    uint32_t refcount;
    bool info_quered;
    bool is_operator;

    // Valid elements: conn[0..conn_count-1]
    DCUserConn *conn[DC_USER_MAX_CONN];
    // Valid range: 0 <= conn_count <= DC_USER_MAX_CONN
    int conn_count;
};

struct _DCUserConn {
    char *name;     	/* key of connection */
    // If info!=NULL then info->conn must contain this DCUserConn.
    DCUserInfo *info;
    DCTransferDirection dir;
    pid_t pid;
    int main_socket;
    IPC *ipc;
    bool occupied_slot;
    //bool we_connected;

    char *transfer_file;    /* valid iff occupied_slot */
    uint64_t transfer_pos;  /* valid iff occupied_slot */
    uint64_t transfer_len;  /* valid if(!) occupied_slot */
    time_t transfer_start;  /* valid iff occupied_slot */
};

/* We provide two completion function prototypes:
 * One for more complex completors (file completors),
 * and one for less complex completors.
 */
typedef char *(*DCCompletionFunction)(const char *base, int state);
typedef bool (*DCFileCompletionFunction)(const char *full,
	const char *base, int state, DCCompletionEntry *ce);
typedef bool (*DCCustomCompletionFunction)(const char *base, int state, DCCompletionEntry *ce);

extern printf_fn_t screen_writer;

extern DCHubState hub_state;
extern ByteQ *hub_recvq;
extern ByteQ *hub_sendq;
extern HMap *hub_users;
extern HMap *user_conns;
extern struct sockaddr_in local_addr;
extern struct in_addr force_listen_addr;
extern int hub_socket;
extern bool running;
extern HMap *pending_userinfo;
extern uint32_t display_flags;

extern uint16_t listen_port;
extern char *my_tag;
extern char *my_nick;
extern char *my_description;
extern char *my_speed;
extern char *my_email;
extern char *share_dir;
extern uint64_t my_share_size;
extern char *download_dir;
extern char *listing_dir;
extern bool is_active;
extern int my_ul_slots;
extern fd_set read_fds;
extern fd_set write_fds;
extern char *my_password;
extern PtrV *delete_files;  /* XXX: use LList? */
extern PtrV *delete_dirs;   /* XXX: use LList? */
extern int used_ul_slots;
extern int used_dl_slots;

extern DCFileList *browse_list; /* list of user we browse */
extern DCUserInfo *browse_user; /* user we browse OR WISH to browse, NULL if possibly browsing ourself */
extern bool browsing_myself;
extern char *browse_path;

/* hub.c */
void hub_input_available(void);
bool hub_putf(const char *format, ...) __attribute__ ((format (printf, 1, 2)));
bool hub_connect_user(DCUserInfo *ui);
DCUserInfo *user_info_new(const char *nick);
DCUserInfo *find_hub_user(const char *nick);
void user_info_free(DCUserInfo *ui);
void hub_connect(struct sockaddr_in *addr);
void hub_disconnect(void);
void hub_now_writable(void);
bool send_my_info(void);
char *user_completion_generator(const char *base, int state);
extern struct sockaddr_in hub_addr;
extern char *hub_name;
bool say_user_completion_generator(const char *base, int state, DCCompletionEntry *ce);
char *user_or_myself_completion_generator(const char *base, int state);

/* command.c */
void command_execute(char *line);
void *get_completor(const char *buffer, int start, int end, DCCompletionFunctionType *type);
/*bool append_download_dir(DCUserInfo *ui, DCFileList *node, char *base_path, uint64_t *bytes, uint32_t *files) XXX move transfer.c? */
void update_prompt(void);

/* screen.c */
void screen_putf(const char *format, ...) __attribute__ ((format (printf, 1, 2)));
void screen_vputf(const char *format, va_list args) __attribute__ ((format (printf, 1, 0)));
void flag_putf(DCDisplayFlag flag, const char *format, ...);
void screen_erase_and_new_line(void);
void screen_finish(void);
void screen_prepare(void);
void screen_read_input(void);
void screen_get_size(int *rows, int *cols);
char *get_argument(const char *p, int start, int count);
void set_screen_prompt(const char *prompt, ...) __attribute__ ((format (printf, 1, 2)));
//char *completion_redisplay_append(const char *input, bool only_match, void *data);
//char *completion_redisplay_prepend(const char *input, bool only_match, void *data);
//char *completion_redisplay_append_if_only_match(const char *input, bool only_match, void *data);

/* huffman.c */
uint8_t *huffman_decode(const uint8_t *data, uint32_t data_size, uint32_t *out_size);
uint8_t *huffman_encode(const uint8_t *data, uint32_t data_size, uint32_t *out_size);

/* user.c */
DCUserConn *user_connection_new(struct sockaddr_in *addr, int socket);
void user_disconnect(DCUserConn *uc);
char *user_conn_status_to_string(DCUserConn *uc, time_t now);

/* main.c */
void add_user_conn(DCUserConn *uc);
/*bool get_user_conn_status(DCUserConn *uc);*/
bool set_active(bool newactive, uint16_t newport);
bool set_share_dir(const char *dir);
bool has_user_conn(DCUserInfo *info, DCTransferDirection dir);
uint32_t get_user_conn_count(DCUserInfo *info);
bool get_package_file(const char *name, char **outname);
char *transfer_completion_generator(const char *base, int state); /* XXX: move transfer.c? */
void user_conn_cancel(DCUserConn *uc);
void warn_file_error(int res, bool write, const char *filename);
void warn_socket_error(int res, bool write, const char *subject);
void add_search_result(struct sockaddr_in *addr, char *results, uint32_t resultlen);
void free_queued_file(DCQueuedFile *qf); /* XXX: move transfer.c? */

/* fs.c */
void filelist_free(DCFileList *fl);
void filelist_list(DCFileList *fl, bool longmode);
DCFileList *filelist_lookup(DCFileList *node, const char *filename);
DCFileList *filelist_open(const char *filename);
char *filelist_get_path(DCFileList *node);
char *resolve_upload_file(DCUserInfo *ui, const char *name);
char *resolve_download_file(DCUserInfo *ui, const char *inname, const char *base_path);
bool filelist_create(const char *dir);
char *translate_local_to_remote(const char *localname);
char *translate_remote_to_local(const char *remotename);
bool local_path_completion_generator(const char *full, const char *base, int state, DCCompletionEntry *ce);
bool local_dir_completion_generator(const char *full, const char *base, int state, DCCompletionEntry *ce);
bool remote_path_completion_generator(const char *full, const char *base, int state, DCCompletionEntry *ce);
bool remote_dir_completion_generator(const char *full, const char *base, int state, DCCompletionEntry *ce);
char *apply_cwd(const char *path); /* XXX: move transfer.c? */
extern DCFileList *our_filelist;
void filelist_list_recursively(DCFileList *node, char *basepath, int skip);
    
/* connection.c */
uint8_t *decode_lock(const uint8_t *lock, int locklen, uint32_t basekey);
char *escape_message(char *str);
void dump_command(const char *header, const char *buf, int len);

/* util.c */
int ilog10(uint64_t c);
int mkdirs_for_file(char *filename, bool deletedirs);
char *catfiles(const char *p1, const char *p2);
int fd_set_nonblock_flag(int fd, int value);
char *getenv_default(const char *name, char *defvalue);
#define IS_CURRENT_DIR(x) ((x)[0] == '.' && (x)[1] == '\0')
#define IS_PARENT_DIR(x)  ((x)[0] == '.' && (x)[1] == '.' && (x)[2] == '\0')
#define COMPARE_RETURN(a,b) { if ((a) < (b)) return -1; if ((a) > (b)) return 1; }
#define COMPARE_RETURN_FUNC(f) { int _c = (f); if (_c != 0) return _c; }
char *sockaddr_in_str(struct sockaddr_in *addr);
bool direxists(const char *dir);
char *get_temp_dir(void);
bool parse_ip_and_port(char *source, struct sockaddr_in *addr, uint16_t defport);

/* search.c */
bool parse_search_selection(char *str, DCSearchSelection *data);
bool perform_inbound_search(DCSearchSelection *data, DCUserInfo *ui, struct sockaddr_in *addr);
extern PtrV *our_searches;
bool add_search_request(char *args);
void handle_search_result(char *buf, uint32_t len);
void free_search_request(DCSearchRequest *sr);

#endif
