/* user.c - User communication (in separate process)
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <assert.h>		/* ? */
#include <netinet/in.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <sys/types.h>		/* ? */
#include <sys/stat.h>		/* ? */
#include <sys/wait.h>		/* ? */
#include <fcntl.h>		/* ? */
#include <signal.h>		/* ? */
#include <sys/socket.h>		/* ? */
#include <netinet/in.h>		/* ? */
#include <arpa/inet.h>		/* ? */
#include <inttypes.h>		/* ? */
#include "dirname.h"		/* Gnulib */
#include "full-write.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "minmax.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "common/error.h"
#include "common/intutil.h"
#include "common/common.h"
#include "microdc.h"

#define LOCK_STRING "ABCABCABCABCABCABCABCABCABCABCABCA"
#define LOCK_STRING_LEN (sizeof(LOCK_STRING)-1)
#define LOCK_PK_STRING "MICRODCABCABCABCABCAB"

#define DEFAULT_RECVQ_SIZE (64*1024)
#define DEFAULT_SENDQ_SIZE (64*1024)

static char *user_nick = NULL;
static char *base_path = NULL;
static char *file_name = NULL;	/* file we wish to download, in local dirlist namespace */
// static char *file_name/*UD*/ = NULL;	/* part of DownloadState */
static char *localfile = NULL; 		/* part of DownloadState */
static DCTransferFlag file_flag;	/* part of DownloadState */
static int main_socket = -1;
static int signal_pipe[2] = { -1, -1 };
static uint32_t user_recvq_last = 0;
static ByteQ *user_recvq;
static ByteQ *user_sendq;
static int user_socket = -1;
static int transfer_file = -1;
static uint64_t data_size = 0;	/* only useful when receiving files */
static uint16_t dir_rand;
static uint64_t file_pos = 0;
static uint64_t transfer_pos = 0;
static uint64_t file_size = 0;
static bool we_connected;
static DCTransferDirection our_dir;
static DCUserState user_state = DC_USER_CONNECT;
static bool user_running = true;
static fd_set user_read_fds;
static fd_set user_write_fds;
static IPC *ipc;

/* NOTE: All the code below assumes that main never disconnects a user
 * (close on main_socket) on purpose. The user must do this first if
 * things happen gracefully.
 */

static bool
check_state(char *buf, DCUserState state)
{
    if (user_state != state) {
	warn(_("Received %s message in wrong state.\n"), strtok(buf, " "));
	user_running = false;
	return false;
    }
    return true;
}

static bool
send_user_status(uint64_t pos)
{
    return ipc_put(ipc, DC_MSG_STATUS, IPC_INT64, pos, IPC_END);
}

static bool
handle_main_msg(IPC *ipc, uint32_t id, void *userdata)
{
    switch (id) {
    case DC_MSG_DISCONNECT:
    	ipc_get(ipc, id, IPC_END);
	ipc_put(ipc, id, IPC_END); /* Ignore errors */
    	user_running = false;
	return false;
    //case DC_MSG_STATUS:
    //	ipc_get(ipc, id, IPC_END);
    //	return ipc_put(ipc, id, IPC_INT64, transfer_pos, IPC_END);
    default:
        screen_putf(_("Received message %d from main process.\n"), id);
	user_running = false;
        return false;
    }
}

static void
handle_main_ipc_error(IPC *ipc, IPCError error, int result, void *userdata)
{
    warn_writer = default_warn_writer;
    screen_writer = default_warn_writer;
    switch (error) {
    case IPC_ERROR_WRITE:
    	warn_socket_error(result, true, _("main process"));
	break;
    case IPC_ERROR_READ:
    	warn_socket_error(result, false, _("main process"));
	break;
    case IPC_ERROR_DATA:
    	screen_putf(_("Logical communication error with main process.\n"));
    	break;
    }
    user_running = false;
}

static int
user_screen_vputf(const char *format, va_list ap)
{
    char *str;
    int res;

    res = vasprintf(&str, format, ap);
    if (res < 0) {
    	user_running = false;
	return -1;
    }
    if (!ipc_put(ipc, DC_MSG_SCREEN_PUT, IPC_STR, str, IPC_END)) {
    	free(str);
    	return -1;
    }
    free(str);

    return res;
}

/* Put some data onto the connection, printf style.
 */
bool
user_putf(const char *format, ...)
{
    va_list args;
    uint32_t oldcur;
    int res;

    oldcur = user_sendq->cur;
    va_start(args, format);
    res = byteq_vappendf(user_sendq, format, args);
    va_end(args);

    if (res < 0) {
    	warn(_("Cannot append to user send queue - %s\n"), errstr);
	user_running = false;
	return false;
    }

    if (data_size/*DL*/ == 0)
        dump_command("-->", user_sendq->buf+oldcur, user_sendq->cur-oldcur);

    res = byteq_write(user_sendq, user_socket);
    if (res == 0 || (res < 0 && errno != EAGAIN)) {
	warn_socket_error(res, true, _("user"));
	user_running = false;
	return false;
    }

    if (oldcur == 0 && user_sendq->cur > 0)
    	FD_SET(user_socket, &user_write_fds);

    return true;
}

static bool
download_next_file(void)
{
    char *remote_file;
    char *dl_file;
    char *dl_base;

    if (!ipc_put(ipc, DC_MSG_NEXT_DOWNLOAD, IPC_END))
    	return false;
    if (!ipc_get(ipc, DC_MSG_NEXT_DOWNLOAD, IPC_STR, &dl_file, IPC_STR, &dl_base, IPC_INT32, &file_flag, IPC_END))
    	return false;
    if (dl_file == NULL) {
        screen_putf(_("No more files to download.\n"));
        user_running = false;
	return false;
    }

    free(file_name/*DL*/);
    free(base_path);
    file_name/*DL*/ = dl_file;
    free(localfile);
    localfile = NULL;
    base_path = dl_base;
    remote_file = translate_local_to_remote(dl_file);
    if (!user_putf("$Get %s$1|", remote_file)) {
        free(remote_file);
	return false;
    }
    free(remote_file);
    user_state = DC_USER_FILE_LENGTH;

    return true;
}

static bool
wants_to_download(bool *reply)
{
    if (!ipc_put(ipc, DC_MSG_WANT_DOWNLOAD, IPC_END))
    	return false;
    if (!ipc_get(ipc, DC_MSG_WANT_DOWNLOAD, IPC_BOOL, reply, IPC_END))
    	return false;
    return true;
}

static bool
direction_validate(DCTransferDirection dir)
{
    bool reply;

    if (!ipc_put(ipc, DC_MSG_VALIDATE_DIR, IPC_INT, dir, IPC_END))
    	return false;
    if (!ipc_get(ipc, DC_MSG_VALIDATE_DIR, IPC_BOOL, &reply, IPC_END))
    	return false;
    if (!reply) {
	warn(_("Too many connections to user, or no free slots.\n"));
	user_running = false;
	return false;
    }
    return true;
}

static bool
nick_validate(const char *str)
{
    bool reply;

    if (!ipc_put(ipc, DC_MSG_VALIDATE_NICK, IPC_STR, str, IPC_END))
    	return false;
    if (!ipc_get(ipc, DC_MSG_VALIDATE_NICK, IPC_BOOL, &reply, IPC_END))
    	return false;
    if (!reply) {
	screen_putf(_("User %s not on hub, or too many connections to user.\n"), str);
	user_running = false;
    	return false;
    }
    return true;
}

static bool
open_download_file(void)
{
    bool reply;

    if (!ipc_put(ipc, DC_MSG_RESOLVE_FILE, IPC_STR, file_name/*DL*/, IPC_STR, base_path, IPC_END))
    	return false; /* maybe free file_name here? */
    if (!ipc_get(ipc, DC_MSG_RESOLVE_FILE, IPC_STR, &localfile, IPC_END))
    	return false; /* maybe free file_name here? */
    if (localfile == NULL) {
	user_running = false;
	return false; /* maybe free file_name here? */
    }

    if (file_flag == DC_TF_NORMAL) {
        char *tmp = xasprintf(_("%s.part"), localfile);
        free(localfile);
        localfile = tmp;
    }

    transfer_file/*DL*/ = open/*64*/(localfile, O_CREAT|O_EXCL|O_WRONLY, 0644);
    if (transfer_file/*DL*/ < 0) {
	warn(_("%s: Cannot open file for writing - %s\n"), localfile, errstr);
	//free(localfile); localfile = NULL;
	//free(file_name/*DL*/); file_name/*DL*/ = NULL;
	/* FIXME: should really try to download file later instead... */
        //if (!ipc_put(ipc, DC_MSG_TRANSFER_FAILED/*DL*/, IPC_INT32, file_flag, IPC_END))
        //   return;
	return download_next_file();
    }

    if (file_size == 0) {
	warn(_("File %s is empty, not starting download.\n"), file_name/*DL*/);
	if (close(transfer_file/*DL*/) < 0)
	    warn(_("%s: Cannot close file - %s\n"), file_name/*DL*/, errstr);
	transfer_file/*DL*/ = -1;
    } else {
        if (!user_putf("$Send|"))
	    return false; /* maybe free file_name here? */
        warn(_("Starting download for file %s, %" PRIu64 " bytes\n"), file_name/*DL*/, file_size);
    }

    /* Let the main process think we are starting the download even if
     * the file is empty and we are not going to download it.
     * XXX: The code in main.c should be rewritten to handle this better.
     */
    if (!ipc_put(ipc, DC_MSG_TRANSFER_STARTING/*DL*/, IPC_INT64, file_size, IPC_STR, file_name/*DL*/, IPC_END))
	return false;
    if (!ipc_get(ipc, DC_MSG_TRANSFER_STARTING/*DL*/, IPC_BOOL, &reply, IPC_END))
    	return false;
    if (!reply) {
	user_running = false;
    	return false;
    }

    if (file_size == 0) {
        if (!ipc_put(ipc, DC_MSG_TRANSFER_FINISHED/*DL*/, IPC_INT32, file_flag, IPC_END))
           return false;
	return download_next_file();
    }
    /* Places to go from here (assuming file_size != 0):
     *   user_running becomes false
     *     User process termination. user_disconnect deals with used_*_slots.
     *   data received until file complete
     *     Will issue DC_MSG_TRANSFER_FINISHED prior to calling
     *     download_next_file.
     */
    file_pos = 0;
    transfer_pos = 0;
    data_size/*DL*/ = file_size;
    user_state = DC_USER_DATA_RECV;
    return true;
}

/* Returns user_running (!) */
static bool
open_upload_file(const char *str, uint64_t offset)
{
    struct stat st;
    char *newname;
    bool reply;
    char *uploadfile;

    newname = translate_remote_to_local(str);
    if (!ipc_put(ipc, DC_MSG_RESOLVE_FILE, IPC_STR, newname, IPC_STR, NULL, IPC_END)) {
    	free(newname);
	return false;
    }
    if (!ipc_get(ipc, DC_MSG_RESOLVE_FILE, IPC_STR, &uploadfile, IPC_END))
    	return false;

    if (uploadfile == NULL) {
        screen_putf(_("Cannot resolve upload file `%s' - invalid filename or no sharedir\n"), newname);
	free(newname);
	if (!user_putf("$Error File Not Available|"))
	    return false;
	user_running = false;
	return false;
    }
    free(newname);

    free(file_name/*UL*/);
    file_name/*UL*/ = uploadfile;

    /* Get file status and check offset. */
    if (stat(file_name/*UL*/, &st) < 0) {
        screen_putf(_("%s: Cannot get file status - %s\n"), file_name/*UL*/, errstr);
	//free(file_name/*UL*/); file_name/*UL*/ = NULL;
	if (!user_putf("$Error File Not Available|"))
	    return false;
        return true;
    }
    if (offset > st.st_size) {
	screen_putf(_("%s: Offset %" PRId64 " out of range\n"), file_name/*UL*/, offset);
	//free(file_name/*UL*/); file_name/*UL*/ = NULL;
	if (!user_putf("$Error Offset out of range|"))
	    return false;
        return true;
    }

    /* Open file and seek to desired position. */
    transfer_file/*UL*/ = open/*64*/(file_name/*UL*/, O_RDONLY);
    if (transfer_file/*UL*/ < 0) {
	screen_putf(_("%s: Cannot open file for reading - %s\n"), file_name/*UL*/, errstr);
	//free(file_name/*UL*/); file_name/*UL*/ = NULL;
	if (!user_putf("$Error File Not Available|"))
	    return false;
        return true;
    }
    if (offset != 0 && lseek/*64*/(transfer_file/*UL*/, offset, SEEK_SET) < 0) {
    	screen_putf(_("%s: Cannot seek in file - %s\n"), file_name/*UL*/, errstr);
	if (!user_putf("$Error File Not Available|"))
	    return false;
	if (close(transfer_file/*UL*/) < 0)
	    warn(_("%s: Cannot close file - %s\n"), file_name/*UL*/, errstr);
	transfer_file/*UL*/ = -1;
	//free(file_name/*UL*/); file_name/*UL*/ = NULL;
	return true;
    }

    file_size = st.st_size;
    file_pos = offset;
    transfer_pos = offset;

    /* Allocate slot. This happens after we've opened the file, because if
     * it would fail, we would have to send back DC_MSG_TRANSFER_FINISHED.
     */
    if (!ipc_put(ipc, DC_MSG_TRANSFER_STARTING/*UL*/, IPC_INT64, file_size, IPC_STR, file_name/*UL*/, IPC_END))
    	return false;
    if (!ipc_get(ipc, DC_MSG_TRANSFER_STARTING/*UL*/, IPC_BOOL, &reply, IPC_END))
    	return false;
    if (!reply) {
    	user_putf("$MaxedOut|"); /* Ignore errors, falling through */
	//transfer_file will be closed at process exit
	user_running = false;
    	return false;
    }

    /* Places to go from here:
     *   user_running becomes false
     *     User process termination. user_disconnect deals with used_*_slots.
     *   $Get received
     *     Will issue DC_MSG_TRANSFER_FINISHED prior to calling this
     *     function again.
     *   $Send received with file_size == 0
     *   $Send received with file_size != 0, download finished
     *     Will issue DC_MSG_TRANSFER_FINISHED prior to setting state
     *     to DC_USER_GET.
     */

    if (!user_putf("$FileLength %llu|", file_size))
	return false;

    user_state = DC_USER_SEND_GET;
    return true;
}

/* Handle commands and data sent by the hub.
 */
static void
user_handle_command(char *buf, uint32_t len)
{
    if (user_state == DC_USER_DATA_RECV) {
    	ssize_t res;

    	res = full_write(transfer_file/*DL*/, buf, len);
	if (res < len) {
	    warn_file_error(res, true, file_name/*DL*/);
	    if (close(transfer_file/*DL*/) < 0)
	    	warn(_("%s: Cannot close file - %s\n"), file_name/*DL*/, errstr);
	    transfer_file/*DL*/ = -1;
	    //free(localfile); localfile = NULL;
	    //free(file_name/*DL*/); file_name/*DL*/ = NULL;
	    /* We cannot expect to synchronize with remote at this point, so shut down. */
	    user_running = false; //download_next_file();
	    return;
	}
	file_pos += len;
	transfer_pos += len;
	data_size/*DL*/ -= len;
    	send_user_status(file_pos);
    	if (file_pos == file_size) {
	    flag_putf(DC_DF_DOWNLOAD, _("%s: Download complete.\n"), base_name(file_name/*DL*/));
	    if (close(transfer_file/*DL*/) < 0)
	    	warn(_("%s: Cannot close file - %s\n"), file_name/*DL*/, errstr);
            if (file_flag == DC_TF_NORMAL) {
                char *tmp;
                tmp = xstrndup(localfile, strlen(localfile)-5);
                if (rename(localfile, tmp) < 0)
                    warn(_("%s: Cannot rename file to `%s' - %s\n"), localfile, tmp, errstr);
                free(tmp);
            }
            //free(localfile); localfile = NULL;
	    //free(file_name/*DL*/); file_name/*DL*/ = NULL; base_name = NULL;
	    transfer_file/*DL*/ = -1;
	    if (!ipc_put(ipc, DC_MSG_TRANSFER_FINISHED/*DL*/, IPC_INT32, file_flag, IPC_END))
	    	return;
	    download_next_file(); /* Will change state for us. */
	    return;
	}
    }
    else if (len >= 8 && strncmp(buf, "$MyNick ", 8) == 0) {
        if (!check_state(buf, DC_USER_MYNICK))
	    return;
	if (!nick_validate(buf+8))
	    return;
	user_nick = xstrdup(buf+8);
	if (!we_connected) {
	    if (!user_putf("$MyNick %s|", my_nick))
		return;
    	    if (!user_putf("$Lock %s Pk=%s|", LOCK_STRING, LOCK_PK_STRING))
		return;
	}
	user_state = DC_USER_LOCK;
    }
    else if (len >= 6 && strncmp(buf, "$Lock ", 6) == 0) {
        char *key;
	bool download;

        if (!check_state(buf, DC_USER_LOCK))
	    return;
        key = memmem(buf+6, len-6, " Pk=", 4);
        if (key == NULL) {
            screen_putf(_("Invalid $Lock message: Missing Pk value\n"));
	    key = buf+len;
	}
        key = decode_lock(buf+6, key-buf-6, DC_CLIENT_BASE_KEY);
	if (!wants_to_download(&download))
	    return;
	if (!user_putf("$Direction %s %d|", download ? "Download" : "Upload", dir_rand)) {
	    free(key);
	    return;
	}
        if (!user_putf("$Key %s|", key)) {
	    free(key);
	    return;
	}
	free(key);
	user_state = DC_USER_DIRECTION;
    }
    else if (len >= 11 && strncmp(buf, "$Direction ", 11) == 0) {
    	char *token;
    	uint16_t remote_rand;
	bool they_download;
	bool we_download;

        if (!check_state(buf, DC_USER_DIRECTION))
	    return;

    	token = strtok(buf+11, " ");
	if (token == NULL) {
	    warn(_("Invalid $Direction message: Missing direction parameter\n"));
	    user_running = false;
	    return;
	}
    	if (strcmp(token, "Upload") == 0) {
	    they_download = false;
	} else if (strcmp(token, "Download") == 0) {
	    they_download = true;
	} else {
	    warn(_("Invalid $Direction message: Invalid direction parameter\n"));
	    user_running = false;
	    return;
	}

	token = strtok(NULL, " ");
	if (token == NULL) {
	    screen_putf(_("Invalid $Direction message: Missing challenge parameter\n"));
	    user_running = false;
	    return;
	}
	if (!parse_uint16(token, &remote_rand)) {
	    warn(_("Invalid $Direction message: Invalid challenge parameter\n"));
	    user_running = false;
	    return;
	}

	if (!wants_to_download(&we_download))
	    return;

    	if (they_download) {
	    // Remote wants to download. Do we want to download too?
	    if (we_download) {
		if (remote_rand >= dir_rand) {
		    // We lost. Let them download.
		    our_dir = DC_DIR_SEND;
		} else {
		    // We won.
		    our_dir = DC_DIR_RECEIVE;
		}
	    } else {
		// We don't want to download anything. Let remote download.
		our_dir = DC_DIR_SEND;
	    }
	} else {
	    // Remote wants to upload. Do we want to upload too?
	    if (!we_download) {
		warn(_("User does not want to download, nor do we.\n"));
		user_running = false;
		return;
	    }
	    our_dir = DC_DIR_RECEIVE;
	}
	if (!direction_validate(our_dir))
	    return;
	user_state = DC_USER_KEY;
    }
    else if (len >= 5 && strncmp(buf, "$Key ", 5) == 0) {
   	char *key;

        if (!check_state(buf, DC_USER_KEY))
	    return;
    	key = decode_lock(LOCK_STRING, LOCK_STRING_LEN, DC_CLIENT_BASE_KEY);
	if (strcmp(buf+5, key) != 0)
	    screen_putf(_("Invalid $Key message: Incorrect key, ignoring\n"));
	free(key);
    	if (our_dir == DC_DIR_SEND) {
	    user_state = DC_USER_GET;
	} else {
	    download_next_file();
	}
    }
    else if (len >= 5 && strncmp(buf, "$Get ", 5) == 0) {
	char *token;
	uint64_t offset;

	if (user_state != DC_USER_SEND_GET && user_state != DC_USER_GET) {
	    warn(_("Received %s message in wrong state.\n"), strtok(buf, " "));
	    user_running = false;
	    return;
	}
	token = strtok(buf+5, "$");
	if (token == NULL) {
	    screen_putf(_("Invalid $Get message: Missing offset, assuming start\n"));
	    offset = 0;
	} else {
	    token = strtok(NULL, ""); /* Cannot fail! */
	    if (!parse_uint64(token, &offset)) {
	    	screen_putf(_("Invalid $Get message: Offset not integer\n"));
		user_running = false;
		return;
	    }
	    if (offset > 0)
	    	offset--;
	}

	if (user_state == DC_USER_SEND_GET) {
	    /* XXX: this probably meant that the remote user
	     * decided he/she didn't want the file (somehow).
	     */
	    if (!ipc_put(ipc, DC_MSG_TRANSFER_FINISHED/*UL*/, IPC_INT32, 0, IPC_END))
	    	return;
	    if (close(transfer_file/*UL*/) < 0)
		warn(_("%s: Cannot close file - %s\n"), file_name/*UL*/, errstr);
	    transfer_file/*UL*/ = -1;
	}
    	open_upload_file(buf+5, offset); /* Changes state */
    }
    else if (len == 9 && strcmp(buf, "$MaxedOut") == 0) {
    	if (!check_state(buf, DC_USER_FILE_LENGTH))
	    return;
	screen_putf(_("User is maxed out.\n"));
	user_running = false;
    }
    else if (len >= 12 && strncmp(buf, "$FileLength ", 12) == 0) {
    	if (!check_state(buf, DC_USER_FILE_LENGTH))
	    return;
	if (!parse_uint64(buf+12, &file_size)) {
	    warn(_("Invalid $FileLength message: Invalid length integer\n"));
	    user_running = false;
	    return;
	}
    	open_download_file(); /* Will change state */
    }
    else if (len >= 7 && strncmp(buf, "$Error ", 7) == 0) {
	if (user_state == DC_USER_FILE_LENGTH) {
    	    if (strcmp(buf+7, "File Not Available") == 0) {
	    	screen_putf("%s: File not available on remote\n", file_name/*DL*/);
		if (close(transfer_file/*DL*/) < 0)
	    	    warn(_("%s: Cannot close file - %s\n"), file_name/*DL*/, errstr);
		// free(file_name/*DL*/); file_name/*UL*/ = NULL;
		transfer_file/*DL*/ = -1;
		download_next_file();
	    }
	    return;
	}
	screen_putf(_("Received error from user: %s\n"), buf+7);
	user_running = false;
    }
    else if (len >= 5 && strncmp(buf, "$Send ", 5) == 0) {
    	if (!check_state(buf, DC_USER_SEND_GET))
	    return;
	if (file_size != file_pos) { /* size is 0 or starting at end */
    	    FD_SET(user_socket, &user_write_fds);
	    assert(user_sendq->cur == 0);
	    user_state = DC_USER_DATA_SEND;
	} else {
	    /* XXX: this probably meant that the remote user
	     * decided he/she didn't want the file (somehow).
	     */
	    if (!ipc_put(ipc, DC_MSG_TRANSFER_FINISHED/*UL*/, IPC_INT32, 0, IPC_END))
	    	return;
	    user_state = DC_USER_GET;
	}
    }
}

static void
user_input_available(void)
{
    int start = 0;
    int c;
    int res;

    res = byteq_read(user_recvq, user_socket);
    if (res == 0 || (res < 0 && errno != EAGAIN)) {
 	warn_socket_error(res, false, _("user"));
	user_running = false;
	return;
    }

    for (c = user_recvq_last; c < user_recvq->cur; c++) {
	if (data_size/*DL*/ > 0) {
	    uint32_t size;

	    /* Handle what've got, but never more than data_size */
	    size = MIN(data_size/*DL*/, user_recvq->cur - start);
    	    user_handle_command(user_recvq->buf + start, size);
	    start += size;
	    if (!user_running)
	    	break;
	    c += size - 1;
	}
        else if (user_recvq->buf[c] == '|') {
            /* Got a complete command. */
	    if (c - start > 0)
	    	dump_command(_("<--"), user_recvq->buf + start, c - start + 1);
            user_recvq->buf[c] = '\0'; /* Just to be on the safe side... */
            user_handle_command(user_recvq->buf + start, c - start);
            start = c+1;
	    if (!user_running)
	    	break;
        }
    }

    if (start != 0)
        byteq_remove(user_recvq, start);

    user_recvq_last = user_recvq->cur;
}

static void
user_now_writable(void)
{
    if (user_state == DC_USER_CONNECT) {
	int error;
        int size = sizeof(error);

        if (getsockopt(user_socket, SOL_SOCKET, SO_ERROR, &error, &size) < 0) {
            warn(_("Cannot get error status - %s\n"), errstr);
	    user_running = false;
	    return;
        }
        if (error != 0) { /* The connect call on the socket failed. */
	    warn(_("Cannot connect - %s\n"), strerror(error) /* not errno! */);
	    user_running = false;
	    return;
	}

    	FD_CLR(user_socket, &user_write_fds);
    	FD_SET(user_socket, &user_read_fds);
	user_state = DC_USER_MYNICK;

    	if (we_connected) {
	    screen_putf(_("Connected to user.\n"));
	    if (!user_putf("$MyNick %s|", my_nick))
		return;
    	    if (!user_putf("$Lock %s Pk=%s|", LOCK_STRING, LOCK_PK_STRING))
		return;
	}
    }
    else if (user_state == DC_USER_DATA_SEND) {
	size_t block;
    	ssize_t res;

    	assert(file_size != 0);
    	block = MIN(DEFAULT_SENDQ_SIZE, file_size-file_pos);
	if (block > 0 && user_sendq->cur == 0) { //if (user_sendq->cur < user_sendq->max) {
	    res = byteq_full_read_upto(user_sendq, transfer_file/*UL*/, block);
	    if (res < block) {
	    	warn_file_error(res, false, file_name/*UL*/);
		if (close(transfer_file/*UL*/) < 0)
		    warn(_("%s: Cannot close file - %s\n"), file_name/*UL*/, errstr);
		transfer_file/*UL*/ = -1;
		//free(file_name/*UL*/); file_name/*UL*/ = NULL;
		user_running = false;
		return;
	    }
    	    file_pos += res;
	}

    	assert(user_sendq->cur != 0);
	res = byteq_write(user_sendq, user_socket);
	if (res == 0 || (res < 0 && errno != EAGAIN)) {
    	    warn_socket_error(res, true, _("user"));
	    user_running = false;
	    return;
	}
	transfer_pos += res;
  	send_user_status(file_pos - user_sendq->cur);

	if (file_pos == file_size && user_sendq->cur == 0) {
	    flag_putf(DC_DF_UPLOAD, _("%s: Upload complete.\n"), base_name(file_name/*UL*/));
    	    FD_CLR(user_socket, &user_write_fds);
	    if (close(transfer_file/*UL*/) < 0)
	    	warn(_("%s: Cannot close file - %s\n"), file_name/*UL*/, errstr);
	    transfer_file = -1;
    	    free(file_name/*UL*/); file_name/*UL*/ = NULL;
	    if (!ipc_put(ipc, DC_MSG_TRANSFER_FINISHED/*UL*/, IPC_INT32, 0, IPC_END))
	    	return;
	    user_state = DC_USER_GET;
	}
    }
    else {
    	int res;

	if (user_sendq->cur > 0) {
	    res = byteq_write(user_sendq, user_socket);
	    if (res == 0 || (res < 0 && errno != EAGAIN)) {
    	    	warn_socket_error(res, true, _("user"));
	    	user_running = false;
		return;
            }
	}
	if (user_sendq->cur == 0)
	    FD_CLR(user_socket, &user_write_fds);
    }
}

static void
signal_received(int signal)
{
    uint8_t signal_char;

    signal_char = signal;
    /* We don't care if this blocks - since we can't postpone this. */
    if (write(signal_pipe[1], &signal_char, sizeof(uint8_t)) < sizeof(uint8_t)) {
	/* Die only if the signal is fatal. */
	if (signal == SIGTERM || signal == SIGINT)
	    die(_("Cannot write to signal pipe - %s\n"), errstr); /* die OK */
	else
	    warn(_("Cannot write to signal pipe - %s\n"), errstr);
    }
}

static void
read_signal_input(void)
{
    uint8_t signal;

    /* This read is atomic since sizeof(int) < PIPE_BUF!
     * It also doesn't block since all data is already
     * available (otherwise select wouldn't tell us there
     * was data).
     */
    if (read(signal_pipe[0], &signal, sizeof(uint8_t)) < 0) {
	warn(_("Cannot read from signal pipe - %s\n"), errstr);
	user_running = false;
	return;
    }

    if (signal == SIGTERM) {
        user_running = false;
    }
    else if (signal == SIGUSR1) {
        /* Not implemented yet, do nothing for now. */
    }
}

static void
__attribute__((noreturn))
user_main(int main_sockets[2], struct sockaddr_in *addr, int sock)
{
    struct sigaction sigact;

    main_socket = main_sockets[1];
    ipc = ipc_new(main_socket, handle_main_msg, handle_main_ipc_error, NULL);
    screen_writer = user_screen_vputf;
    warn_writer = user_screen_vputf;

    if (close(main_sockets[0]) < 0)
    	warn(_("Cannot close socket - %s\n"), errstr);

    if (pipe(signal_pipe) < 0) {
    	warn(_("Cannot create pipe pair - %s\n"), errstr);
	goto cleanup;
    }
    sigact.sa_handler = signal_received;
    if (sigemptyset(&sigact.sa_mask) < 0) {
        warn(_("Cannot empty signal set - %s\n"), errstr);
	goto cleanup;
    }
    sigact.sa_flags = SA_RESTART;
    sigact.sa_restorer = NULL;
    if (sigaction(SIGTERM, &sigact, NULL) < 0 || sigaction(SIGUSR1, &sigact, NULL) < 0) {
        warn(_("Cannot register signal handler - %s\n"), errstr);
	goto cleanup;
    }

    //user_recvq_last = 0;
    user_recvq = byteq_new(DEFAULT_RECVQ_SIZE);
    user_sendq = byteq_new(DEFAULT_SENDQ_SIZE);
    //user_state = DC_USER_MYNICK;*/
    dir_rand = rand() % 0x8000;
    //file_pos = 0;
    //file_size = 0;
    //data_size = 0;
    if (sock < 0) {
	user_socket = socket(PF_INET, SOCK_STREAM, 0);
	if (user_socket < 0) {
            warn(_("Cannot create socket - %s\n"), errstr);
	    goto cleanup;
	}
    	we_connected = true;
    } else {
    	user_socket = sock;
    	we_connected = false;
    }

    /* Set non-blocking I/O on socket. */
    if (fd_set_nonblock_flag(user_socket, true) < 0) {
        warn(_("Cannot set socket flags - %s\n"), errstr);
        goto cleanup;
    }

    if (sock < 0) {
	/* Connect to host, non-blocking manner. Even if connect would return
	 * success, we would catch the successful connection after select -
	 * when socket_fd is writable.
	 */
	if (connect(user_socket, (struct sockaddr *) addr, sizeof(struct sockaddr_in)) < 0
	    	&& errno != EINPROGRESS) {
            warn(_("Cannot connect - %s\n"), errstr);
	    goto cleanup;
	}
    }

    FD_ZERO(&user_read_fds);
    FD_ZERO(&user_write_fds);
    FD_SET(signal_pipe[0], &user_read_fds);
    FD_SET(main_socket, &user_read_fds);
    FD_SET(user_socket, &user_write_fds); /* will set read after connect() finished */

    while (user_running) {
    	fd_set res_read_fds;
	fd_set res_write_fds;

	res_read_fds = user_read_fds;
	res_write_fds = user_write_fds;
	if (TEMP_FAILURE_RETRY(select(FD_SETSIZE, &res_read_fds, &res_write_fds, NULL, NULL)) < 0) {
	    warn(_("Cannot select - %s\n"), errstr);
	    break;
	}

    	/* Must check for user writable prior to readable, to detect
	 * connect() completion before any data is received and handled.
	 */
    	if (user_running && FD_ISSET(signal_pipe[0], &res_read_fds))
	    read_signal_input();
	if (user_running && FD_ISSET(main_socket, &res_read_fds))
	    ipc_handle(ipc);
	if (user_running && FD_ISSET(user_socket, &res_write_fds))
	    user_now_writable();
	if (user_running && FD_ISSET(user_socket, &res_read_fds))
	    user_input_available();
    }

cleanup:

    ipc_free(ipc);
    free(localfile);
    free(file_name);
    free(user_nick);
    byteq_free(user_recvq);
    byteq_free(user_sendq);

    if (transfer_file >= 0 && close(transfer_file) < 0)
    	warn(_("Cannot close transfer file - %s\n"), errstr); /* XXX: should print WHAT file - then remove " transfer" */
    if (signal_pipe[0] >= 0 && close(signal_pipe[0]) < 0)
    	warn(_("Cannot close signal pipe - %s\n"), errstr);
    if (signal_pipe[1] >= 0 && close(signal_pipe[1]) < 0)
    	warn(_("Cannot close signal pipe - %s\n"), errstr);
    if (main_socket >= 0 && close(main_socket) < 0)
    	warn(_("Cannot close main process socket - %s\n"), errstr);	/* XXX: "socket" or "connection" */
    if (user_socket >= 0 && close(user_socket) < 0)
    	warn(_("Cannot close user connection - %s\n"), errstr);

    exit(EXIT_SUCCESS);
}

DCUserConn *
user_connection_new(struct sockaddr_in *addr, int user_socket)
{
    DCUserConn *uc;
    int main_sockets[2];
    pid_t pid;

    if (socketpair(AF_LOCAL, SOCK_STREAM, 0, main_sockets) < 0) {
    	screen_putf(_("Cannot create socket pair - %s\n"), errstr);
	return NULL;
    }

    pid = fork();
    if (pid < 0) {
    	screen_putf(_("Cannot create process - %s\n"), errstr);
	if (close(main_sockets[0]) < 0)
	    screen_putf(_("Cannot close socket - %s\n"), errstr);
	if (close(main_sockets[1]) < 0)
	    screen_putf(_("Cannot close socket - %s\n"), errstr);
	return NULL;
    }
    if (pid == 0)
	user_main(main_sockets, addr, user_socket);

    if (close(main_sockets[1]) < 0)
    	screen_putf(_("Cannot close socket - %s\n"), errstr);
    if (user_socket >= 0 && close(user_socket) < 0)
    	screen_putf(_("Cannot close socket - %s\n"), errstr);

    uc = xmalloc(sizeof(DCUserConn));
    uc->pid = pid;
    uc->main_socket = main_sockets[0];
    uc->info = NULL;
    uc->occupied_slot = 0;
    uc->dir = DC_DIR_UNKNOWN;
    uc->transfer_file = NULL;
    //uc->we_connected = (user_socket < 0);
    add_user_conn(uc);
    FD_SET(uc->main_socket, &read_fds);

    return uc;
}

void
user_disconnect(DCUserConn *uc)
{
    int status;

    screen_putf(_("User %s: Shutting down user connection.\n"), uc->name);
    ipc_free(uc->ipc);
    if (uc->info != NULL) {
    	int c;
	for (c = 0; c < uc->info->conn_count; c++) {
	    if (uc->info->conn[c] == uc)
		break;
	}
	assert(c < uc->info->conn_count);
	uc->info->conn[c] = NULL;
    	uc->info->conn_count--;
	for (; c < uc->info->conn_count; c++)
	    uc->info->conn[c] = uc->info->conn[c+1];
	user_info_free(uc->info);
    }
    if (uc->occupied_slot) {
    	if (uc->dir == DC_DIR_SEND) {
	    used_ul_slots--;
	} else { /* uc->dir must be DC_DIR_RECV */
	    used_dl_slots--;
	}
    }
    FD_CLR(uc->main_socket, &read_fds);
    FD_CLR(uc->main_socket, &write_fds);
    if (close(uc->main_socket) < 0)
	warn(_("Cannot close socket - %s\n"), errstr);
    free(uc->transfer_file);
    if (waitpid(uc->pid, &status, 0) < 0)
    	warn(_("Cannot wait for process - %s\n"), errstr);
    else if (!WIFEXITED(status))
    	warn(_("User process exited with status %d\n"), status);
    free(uc);
}

char *
user_conn_status_to_string(DCUserConn *uc, time_t now)
{
    char *status;

    if (uc->occupied_slot) {
    	int percent = (uc->transfer_pos * 100)/MAX(uc->transfer_len, 1);
	int rate = (now == 0 ? 0 : uc->transfer_pos/1024/MAX(now-uc->transfer_start, 1));
	char *file = base_name(uc->transfer_file);
	status = xasprintf(
	    uc->dir == DC_DIR_RECEIVE
		? _("Downloading %3d%% (at %5d kb/s) %s")
		: _("Uploading   %3d%% (at %5d kb/s) %s"),
	    percent, rate, file);
    } else {
	status = xasprintf(_("Idle"));
    }

    return status;
}

bool
parse_ip_and_port(char *source, struct sockaddr_in *addr, uint16_t defport)
{
    char *port;

    port = strchr(source, ':');
    if (port == NULL) {
    	if (defport == 0)
	    return false;
	addr->sin_port = htons(defport);
    } else {
	*port = '\0';
	port++;
	if (!parse_uint16(port, &addr->sin_port))
    	    return false;
        addr->sin_port = htons(addr->sin_port);
    }

    if (!inet_aton(source, &addr->sin_addr))
    	return false;

    addr->sin_family = AF_INET;
    return true;
}
