/* screen.c - User interface management (Readline)
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#if defined(HAVE_READLINE_READLINE_H)
# include <readline/readline.h>
#elif defined(HAVE_READLINE_H)
# include <readline.h>
#endif
#if defined(HAVE_READLINE_HISTORY_H)
# include <readline/history.h>
#elif defined(HAVE_HISTORY_H)
# include <history.h>
#endif
#include "xalloc.h"		/* Gnulib */
#include "minmax.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "quotearg.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "microdc.h"
#include "common/error.h"
#include "common/strleftcmp.h"
#include "common/bksearch.h"
#include "common/quoting.h"

static void clear_rl();
static void user_input(char *line);
static int real_screen_vputf(const char *format, va_list args);
static char *screen_prompt;
static FILE *log_fh = NULL;
char *log_filename = NULL;

vprintf_fn_t screen_writer = real_screen_vputf;

enum {
    SCREEN_NO_HISTORY,
    SCREEN_NO_HANDLER,		/* rl_callback_handler_install not called. */
    SCREEN_NORMAL,		/* Normal non-readline status. Can print right away. */
    SCREEN_RL_DISPLAYED,	/* Readline will be displayed. Need to clear first. */
    SCREEN_RL_CLEARED,		/* Readline has been cleared. Need to redisplay after. */
} screen_state = SCREEN_NO_HISTORY;

bool
set_log_file(const char *new_filename, bool verbose)
{
    if (log_fh != NULL) {
	if (fclose(log_fh) != 0)
	    warn(_("%s: Cannot close file - %s\n"), quotearg(log_filename), errstr);
	log_fh = NULL;
	free(log_filename);
	log_filename = NULL;
    }
    if (new_filename == NULL) {
	if (verbose)
	    screen_putf(_("No longer logging to file.\n"));
	return true;
    }
    log_fh = fopen(new_filename, "a");
    if (log_fh == NULL) {
	screen_putf(_("%s: Cannot open file for appending - %s\n"), quotearg(new_filename), errstr);
	return false;
    }
    log_filename = xstrdup(new_filename);
    if (verbose)
	screen_putf(_("Logging to `%s'.\n"), quotearg(new_filename));
    return true;
}

/* Readline < 5.0 disables SA_RESTART on SIGWINCH for some reason.
 * This turns it back on.
 * This was copied from guile.
 */
static int
fix_winch(void)
{
    struct sigaction action;

    if (sigaction(SIGWINCH, NULL, &action) >= 0) {
    	action.sa_flags |= SA_RESTART;
    	sigaction(SIGWINCH, &action, NULL); /* Ignore errors */
    }

    return 0;
}

static char **
attempted_completion(const char *text, int start, int end)
{
    void *completor;
    DCCompletionFunctionType type;
    char **matches = NULL;
    char *dir_part = NULL;
    char *file_part = NULL;
    char *text_nq;
    bool quoted;

    /* Determine if current word is quoted, and dequote. */
    quoted = char_is_quoted(rl_line_buffer, start)
              && rl_line_buffer[start-1] == '"';
    text_nq = dequote_words(text, quoted, NULL);
   
    /* Do not attempt readline's default filename completion if our
     * completors fails to return any results.
     */
    rl_attempted_completion_over = 1;

    completor = default_completion_selector(rl_line_buffer, start, end, &type);
    /* XXX: Better idea than DC_CPL_CHAIN: 
     * Let the completor itself determine how to handle
     * arguments. Is the dir/file_part separation really necessary?
     * Cannot the completor itself do this itself? All that is
     * necessary is rules before, and rules after. Shouldn't the
     * completer return a complete set of results instead? And be
     * passed a LList or something where results can be added?
     * The first will always be the input result... But gcd (leading
     * character) still need to be calculated.
     */

    if (type == DC_CPL_CHAIN) {
	DCChainCompletionFunction chain_completor = completor;
	completor = chain_completor(rl_line_buffer, start, end, &type);
    }
    if (type == DC_CPL_FILE) {
	file_part = strrchr(text_nq, '/');
	if (file_part == NULL) {
	    dir_part = xstrdup("");
	    file_part = text_nq;
	} else {
	    for (; file_part > text_nq && file_part[-1] == '/'; file_part--);
	    dir_part = xstrndup(text_nq, file_part-text_nq+1);
	    for (file_part++; *file_part == '/'; file_part++);
	}
    } else if (type == DC_CPL_SIMPLE || type == DC_CPL_CUSTOM) {
	dir_part = NULL;
	file_part = text_nq;
    }

    if (type == DC_CPL_NONE || completor != NULL) {
    	PtrV *completion_entries;
    	int state;
	int lcd = INT_MAX;
	uint32_t c;

    	completion_entries = ptrv_new();
	for (state = 0; true; state++) {
	    /*DCCompletionEntry ce = { NULL, ' ', '\0', '\0', '\0' };*/
	    DCCompletionEntry ce = { NULL, ' ', "%s", "%s", NULL, NULL };

	    if (type == DC_CPL_SIMPLE) {
		DCCompletionFunction simple_completor = completor;
		ce.str = simple_completor(file_part, state);
		if (ce.str == NULL)
		    break;
	    } else if (type == DC_CPL_FILE) {
		DCFileCompletionFunction file_completor = completor;
		if (!file_completor(dir_part, file_part, state, &ce))
		    break;
	    } else if (type == DC_CPL_CUSTOM) {
		DCCustomCompletionFunction custom_completor = completor;
		if (!custom_completor(file_part, state, &ce))
		    break;
	    }
	    ptrv_append(completion_entries, xmemdup(&ce, sizeof(ce)));

    	    if (state > 0) {
    		DCCompletionEntry *oldce = completion_entries->buf[state-1];
		int c1, c2, si;

                /* Determine common leading characters for all matches. */
	    	for (si = 0; (c1 = oldce->str[si]) && (c2 = ce.str[si]); si++) {
		    if (c1 != c2)
		    	break;
    	    	}
	    	lcd = MIN(lcd, si);
	    }
	}

    	if (state == 0) { /* no matches */
	    ptrv_free(completion_entries);
	    matches = NULL;
	}
	else if (state == 1) { /* single match */
	    char *tmp;
	    DCCompletionEntry *ce;

            ce = completion_entries->buf[0];
	    matches = xmalloc(sizeof(char *) * 2);

	    if (ce->input_single_format != NULL) {
		tmp = xasprintf(ce->input_single_format, ce->str);
		free(ce->str);
		ce->str = tmp;
	    } else if (ce->input_single_full_format != NULL && strcmp(file_part, ce->str) == 0) {
		free(ce->str);
		ce->str = xasprintf(ce->input_single_full_format, file_part);
	    }

	    if (type == DC_CPL_FILE) {
		tmp = catfiles(dir_part, ce->str);
		free(ce->str);
		ce->str = xasprintf(ce->input_format, tmp);
		free(tmp);
	    } else if (type == DC_CPL_SIMPLE || type == DC_CPL_CUSTOM) {
		tmp = xasprintf(ce->input_format, ce->str);
		free(ce->str);
		ce->str = tmp;
	    }

	    matches[0] = quote_word(ce->str, quoted, true);
	    matches[1] = NULL;

	    rl_completion_append_character = ce->input_char;
	    free(ce->str);
    	    free(ce);
	    ptrv_free(completion_entries);
	}
    	else { /* multiple matches */
    	    DCCompletionEntry *ce = completion_entries->buf[0];
	    char *tmp = NULL;

    	    matches = xmalloc(sizeof(char *) * (state + 2));
	    matches[0] = xstrndup(ce->str, lcd);
	    if (type == DC_CPL_FILE) {
		char *tmp2;
		tmp2 = catfiles(dir_part, matches[0]);
		tmp = xasprintf(ce->input_format, tmp2);
		free(tmp2);
		free(matches[0]);
	    }
	    else if (type == DC_CPL_SIMPLE || type == DC_CPL_CUSTOM) {
		tmp = xasprintf(ce->input_format, matches[0]);
		free(matches[0]);
	    }
	    matches[0] = quote_word(tmp, quoted, false);
	    free(tmp);

	    for (c = 0; c < state; c++) {
	    	ce = completion_entries->buf[c];
		/*if (ce->display_format != NULL) {*/
		    /*matches[c+1] = xasprintf("%s%c", ce->str, ce->display_char);*/
		    matches[c+1] = xasprintf(ce->display_format, ce->str);
		    free(ce->str);
		/*} else {
	    	    matches[c+1] = ce->str;
		}*/
	    }
	    matches[state+1] = NULL;
	    ptrv_foreach(completion_entries, free);
	    ptrv_free(completion_entries);
	}
    }

    free(text_nq);
    free(dir_part);

    return matches;
}

/* This piece of code was snatched from lftp, and modified somewhat by me.
 * I suggest a function called rl_clear() be added to readline. The
 * function clears the prompt and everything the user has written so far on
 * the line. The cursor is positioned at the beginning of the line that
 * contained the prompt Note: This function doesn't modify the screen_state
 * variable.
 */
static void
clear_rl()
{
    extern char *rl_display_prompt;
#if HAVE__RL_MARK_MODIFIED_LINES
    extern int _rl_mark_modified_lines;
    int old_mark = _rl_mark_modified_lines;
#endif
    int old_end = rl_end;
    char *old_prompt = rl_display_prompt;

    rl_end = 0;
    rl_display_prompt = "";
    rl_expand_prompt("");
#if HAVE__RL_MARK_MODIFIED_LINES
    _rl_mark_modified_lines = 0;
#endif

    rl_redisplay();

    rl_end = old_end;
    rl_display_prompt = old_prompt;
#if HAVE__RL_MARK_MODIFIED_LINES
    _rl_mark_modified_lines = old_mark;
#endif
    if (rl_display_prompt == rl_prompt)
        rl_expand_prompt(rl_prompt);
}

static int
real_screen_vputf(const char *format, va_list args)
{
    if (screen_state == SCREEN_RL_DISPLAYED) {
        clear_rl();
        screen_state = SCREEN_RL_CLEARED;
    }
    return vprintf(format, args);
}

/* Print something on screen, printf format style.
 * __attribute__ is provided by microdc.h declaration.
 */
void
screen_putf(const char *format, ...)
{
    va_list args;

    va_start(args, format);
    screen_writer(format, args);
    if (log_fh != NULL)
	vfprintf(log_fh, format, args);
    va_end(args);
}

void
flag_putf(DCDisplayFlag flag, const char *format, ...)
{
    va_list args;

    va_start(args, format);
    if (display_flags & flag)
	screen_writer(format, args);
    if (log_fh != NULL && log_flags & flag)
	vfprintf(log_fh, format, args);
    va_end(args);
}

/* This function is called by readline whenever the user has
 * entered a full line (usually by pressing enter).
 */
static void
user_input(char *line)
{
    /* Readline has already made way for us. */
    screen_state = SCREEN_NORMAL;

    if (log_fh != NULL)
	fprintf(log_fh, "> %s\n", line == NULL ? "(null)" : line);
    
    if (line == NULL) {
        /* Ctrl+D was pressed on an empty line. */
        screen_putf("exit\n");
        running = false;
    } else if (line[0] != '\0') {
        add_history(line);
        command_execute(line);
    }

    /* The following is necessary or readline will display
     * a new prompt before we exit.
     */
    if (!running) {
        rl_callback_handler_remove();
        screen_state = SCREEN_NO_HANDLER;
    }
}

/* Move down one line (rl_on_new_line+redisplay), print a new prompt and
 * empty the input buffer. Unlike rl_clear this doesn't erase anything on
 * screen.
 *
 * This is usually called when a user presses Ctrl+C.
 * XXX: Should find a better way to do this (see lftp or bash).
 */
void
screen_erase_and_new_line(void)
{
    if (screen_state != SCREEN_NO_HANDLER) {
        rl_callback_handler_remove();
        printf("\n");
        rl_callback_handler_install(screen_prompt, user_input);
    }
}

/* Finish screen management. Usually called from main_finish.
 */
void
screen_finish(void)
{
    if (screen_state > SCREEN_NO_HANDLER) {
        rl_callback_handler_remove();
        if (screen_state == SCREEN_RL_DISPLAYED)
    	    printf("\n");
	screen_state = SCREEN_NO_HANDLER;
    }

    if (screen_state > SCREEN_NO_HISTORY) {
    	char *path;

	/* Save history */
    	get_package_file("history", &path);
    	if (mkdirs_for_file(path, false) >= 0) {
	    if (write_history(path) != 0)
		warn(_("%s: Cannot write history - %s\n"), quotearg(path), errstr);
	}
	free(path);

    	//rl_basic_word_break_characters = ?
	//rl_completer_word_break_characters = ?
    	//rl_completion_display_matches_hook = ?
	rl_attempted_completion_function = NULL;
	//rl_char_is_quoted_p = NULL;
    	rl_pre_input_hook = NULL;

	warn_writer = default_warn_writer;
        screen_state = SCREEN_NO_HISTORY;

	set_log_file(NULL, false);
    }
}

/* Prepare the screen prior to waiting for events with select/poll/epoll.
 * Redisplay the prompt if it was cleared by a call to screen_(v)put(f).
 */
void
screen_prepare(void)
{
    if (screen_state <= SCREEN_NO_HISTORY) {
    	char *path;

    	screen_state = SCREEN_NO_HANDLER;
	warn_writer = real_screen_vputf;
	screen_prompt = xasprintf("%s> ", PACKAGE);

	rl_readline_name = PACKAGE;
	rl_attempted_completion_function = attempted_completion;
	rl_completer_quote_characters = "\"";
	rl_completer_word_break_characters = " \t\n\"";
	/* rl_filename_quote_characters = " \t\n\\\"'>;|&()*?[]~!"; */
	/* rl_filename_quoting_function = quote_argument; */
	/* rl_filename_dequoting_function = dequote_argument_rl; */
	rl_char_is_quoted_p = char_is_quoted;
    	rl_pre_input_hook = fix_winch;

        using_history();
    	get_package_file("history", &path);
	if (read_history(path) != 0 && errno != ENOENT)
    	    warn(_("%s: Cannot read history - %s\n"), quotearg(path), errstr);
    }
    if (screen_state == SCREEN_NO_HANDLER) {
        rl_callback_handler_install(screen_prompt, user_input);
    }
    else if (screen_state == SCREEN_RL_CLEARED) {
        rl_redisplay();
    }
    screen_state = SCREEN_RL_DISPLAYED;
}

/* This function is called from the main loop when there's input for us to
 * read on stdin.
 */
void
screen_read_input(void)
{
    rl_callback_read_char();
}

/* Return the size of the screen.
 */
void
screen_get_size(int *rows, int *cols)
{
    int dummy;
    rl_get_screen_size(rows ? rows : &dummy, cols ? cols : &dummy);
}

void
set_screen_prompt(const char *prompt, ...)
{
    va_list args;

    if (screen_prompt != NULL)
	free(screen_prompt);
    va_start(args, prompt);
    screen_prompt = xvasprintf(prompt, args);
    va_end(args);
    rl_set_prompt(screen_prompt);
}

/* Look up completion alternatives from a sorted list using binary search.
 */
char *
sorted_list_completion_generator(const char *base, int state, void *items,
                                 size_t item_count, size_t item_size,
                                 size_t key_offset)
{
    static const void *item;
    static const void *last_item;

    if (state == 0) {
	if (!bksearchrange(base, items, item_count, item_size,
	                   key_offset, (comparison_fn_t) strleftcmp,
	                   &item, &last_item))
	    return NULL;
    }

    if (item <= last_item) {
	char *key = xstrdup(*(const char **) (((const char *) item) + key_offset));
	item = (const void *) (((const char *) item) + item_size);
	return key;
    }

    return NULL;
}
