/* util.c - Utility functions that didn't make it elsewhere
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <stdint.h>		/* Gnulib/POSIX/C99 */
#include <sys/types.h>		/* ? */
#include <sys/stat.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <fcntl.h>		/* ? */
#include "xalloc.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "quotearg.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "common/comparison.h"
#include "microdc.h"

/* Move to lib !? */
int
ilog10(uint64_t c)
{
    int r;
    for (r = 0; c > 0; r++)
        c /= 10;
    return r;
}

/* XXX: try to get rid of screen_putf from this function. */
int
mkdirs_for_file(char *filename, bool deletedirs)
{
    char *t;

    for (t = filename; *t == '/'; t++);
    while ((t = strchr(t, '/')) != NULL) {
	struct stat st;

        *t = '\0';
	if (stat(filename, &st) < 0) {
	    if (errno != ENOENT) {
	    	screen_putf(_("%s: Cannot get file status - %s\n"), quotearg(filename), errstr);
		return -1;
	    } else {
        	if (mkdir(filename, 0777) < 0) {
    	    	    screen_putf(_("%s: Cannot create directory - %s\n"), quotearg(filename), errstr); 
		    return -1;
		} else if (deletedirs) {
	    	    if (ptrv_find(delete_dirs, filename, (comparison_fn_t) strcmp) < 0)
	    		ptrv_append(delete_dirs, xstrdup(filename));
		}
	    }
	}
        *t = '/';
	for (; *t == '/'; t++);
    }

    return 0;
}

/* Can never return NULL */
char *
catfiles(const char *p1, const char *p2)
{
    char *sep;
    char *out;

    if (*p2 == '\0')
    	return xstrdup(p1);
    if (*p1 == '\0')
    	return xstrdup(p2);

    sep = (p1[strlen(p1)-1] == '/' ? "" : "/");
    out = xasprintf("%s%s%s", p1, sep, p2);
    return out;
}

/* Get environment variable with default value if it is not set.
 * Can be moved info lib directory: Finalize name.
 */
char *
getenv_default(const char *name, char *defvalue)
{
    char *value;
    value = getenv(name);
    return (value == NULL ? defvalue : value);
}

/* Set or remove the O_NONBLOCK flag on the specified file descriptor.
 * Can be moved into lib directory: Finalize name, expand interface to any
 * flag?
 */
int
fd_set_nonblock_flag(int fd, int value)
{
    int flags;

    flags = fcntl(fd, F_GETFL, 0);
    if (flags < 0)
    	return -1;

    if (value != 0)
    	flags |= flags;
    else
    	flags &= ~flags;

    return fcntl(fd, F_SETFL, flags);
}

char *
sockaddr_in_str(struct sockaddr_in *addr)
{
    static char buffer[22];
    unsigned char *bytes;

    bytes = (unsigned char *) &addr->sin_addr;
    snprintf(buffer, sizeof(buffer), "%d.%d.%d.%d:%u",
    	    bytes[0], bytes[1], bytes[2], bytes[3],
	    ntohs(addr->sin_port));

    return buffer;
}

PtrV *
wordwrap(char *str, size_t len, size_t first_width, size_t other_width)
{
    size_t width;
    PtrV *out;

    out = ptrv_new();
    width = first_width;
    while (len > width) {
        size_t c, d;

        if (str[width] == ' ') {
            ptrv_append(out, xstrndup(str, width));
            for (c = width+1; c < len && str[c] == ' '; c++);
            str += c;
            len -= c;
        } else {

            for (c = width; c > 0 && str[c-1] != ' '; c--);
            for (d = width+1; d < len && str[d] != ' '; d++);
            if (d-c <= width) {
                ptrv_append(out, xstrndup(str, c-1));
                str += c;
                len -= c;
            } else {
                ptrv_append(out, xstrndup(str, width));
                str += width;
                len -= width;
            }
        }
        width = other_width;
    }
    if (len > 0)
        ptrv_append(out, xstrndup(str, len));

    return out;
}
