/* variables.c - Setting variables
 *
 * Copyright (C) 2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "xalloc.h"		/* Gnulib */
#include "quotearg.h"		/* Gnulib */
#include "minmax.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "common/strleftcmp.h"
#include "common/intutil.h"
#include "common/strbuf.h"
#include "common/quoting.h"
#include "common/comparison.h"
#include "common/bksearch.h"
#include "common/quoting.h"
#include "microdc.h"

static void var_set_nick(DCVariable *var, const char *new_value);
static void var_set_password(DCVariable *var, const char *new_value);
static void var_set_description(DCVariable *var, const char *new_value);
static void var_set_tag(DCVariable *var, const char *new_value);
static void var_set_speed(DCVariable *var, const char *new_value);
static void var_set_email(DCVariable *var, const char *new_value);
static void var_set_share_dir(DCVariable *var, const char *new_value);
static void var_set_download_dir(DCVariable *var, const char *new_value);
static void var_set_listing_dir(DCVariable *var, const char *new_value);
static void var_set_slots(DCVariable *var, const char *new_value);
static char *var_get_slots(DCVariable *var);
static char *var_get_string(DCVariable *var);
static bool display_completion_generator(const char *base, int state, DCCompletionEntry *ce);
static char *speed_completion_generator(const char *base, int state);
static char *var_get_bool(DCVariable *var);
static void var_set_active(DCVariable *var, const char *new_value);
static char *var_get_listen_addr(DCVariable *var);
static void var_set_listen_addr(DCVariable *var, const char *new_value);
static char *var_get_listen_port(DCVariable *var);
static void var_set_listen_port(DCVariable *var, const char *new_value);
static char *var_get_display_flags(DCVariable *var);
static void var_set_display_flags(DCVariable *var, const char *new_value);
static void var_set_log_file(DCVariable *var, const char *new_value);
static char *speed_completion_generator(const char *base, int state);
static char *variable_completion_generator(const char *base, int state);
static bool display_completion_generator(const char *base, int state, DCCompletionEntry *ce);

typedef struct {
    DCDisplayFlag flag;
    const char *name;
} DCDisplayFlagDetails;

/* This list must be sorted according to strcmp. */
DCDisplayFlagDetails display_flag_details[] =  {
    { DC_DF_CONNECTIONS, 	"connections" },
    { DC_DF_DEBUG, 		"debug" },
    { DC_DF_DOWNLOAD, 		"download" },
    { DC_DF_JOIN_PART, 		"joinpart" },
    { DC_DF_PUBLIC_CHAT, 	"publicchat" },
    { DC_DF_SEARCH_RESULTS, 	"searchresults" },
    { DC_DF_UPLOAD, 		"upload" },
};
static const int display_flag_count = sizeof(display_flag_details)/sizeof(*display_flag_details);

static char *speeds[] = {
    "Modem",
    "28.8Kbps",
    "33.6Kbps",
    "56Kbps",
    "Satellite",
    "ISDN",
    "DSL",
    "Cable",
    "LAN(T1)",
    "LAN(T3)",
};
static const int speeds_count = sizeof(speeds)/sizeof(*speeds);

/* This structure must be sorted by variable name, according to strcmp. */
static DCVariable variables[] = {
    {
      "active",
      var_get_bool, var_set_active, &is_active,
      DC_CPL_NONE, NULL,
      NULL,
      "Enable if listening for remote connections"
    },
    {
      "description",
      var_get_string, var_set_description, &my_description,
      DC_CPL_NONE, NULL,
      NULL,
      "This is the description which is visible to other users of the hub."
    },
    {
      "display",
      var_get_display_flags, var_set_display_flags, &display_flags,
      DC_CPL_CUSTOM, display_completion_generator,
      NULL,
      "Types of messages to display on screen"
    },
    {
      "downloaddir",
      var_get_string, var_set_download_dir, &download_dir,
      DC_CPL_FILE, local_dir_completion_generator,
      NULL,
      "Directory which files are downloaded to"
    },
    {
      "email",
      var_get_string, var_set_email, &my_email,
      DC_CPL_NONE, NULL,
      NULL,
      "The e-mail visible to other users of the hub"
    },
    {
      "listenaddr",
      var_get_listen_addr, var_set_listen_addr, &force_listen_addr,
      DC_CPL_NONE, NULL,
      NULL,
      "Address to send to clients"
    },
    {
      "listenport",
      var_get_listen_port, var_set_listen_port, &listen_port,
      DC_CPL_NONE, NULL,
      NULL,
      "Port to listen on for connections"
    },
    {
      "listingdir",
      var_get_string, var_set_listing_dir, &download_dir,
      DC_CPL_FILE, local_dir_completion_generator,
      NULL,
      "Directory where file listings are kept"
    },
    {
      "log",
      var_get_display_flags, var_set_display_flags, &log_flags,
      DC_CPL_CUSTOM, display_completion_generator,
      NULL,
      "Types of messages to log (if logfile set)"
    },
    {
      "logfile",
      var_get_string, var_set_log_file, &log_filename,
      DC_CPL_FILE, local_path_completion_generator,
      NULL,
      "File to log screen messages to (will be appeneded)"
    },
    {
      "nick",
      var_get_string, var_set_nick, &my_nick,
      DC_CPL_NONE, NULL,
      NULL,
      "This is the desired (but not necessarily the current) nick name."
    },
    {
      "password",
      var_get_string, var_set_password, &my_password,
      DC_CPL_NONE, NULL,
      NULL,
      "The optional password to pass to the hub."
    },
    {
      "sharedir",
      var_get_string, var_set_share_dir, &share_dir,
      DC_CPL_FILE, local_path_completion_generator,
      NULL,
      "Directory containing files to share (or a single file to share)"
    },
    {
      "slots",
      var_get_slots, var_set_slots, &my_ul_slots,
      DC_CPL_NONE, NULL,
      NULL,
      "Number of open upload slots"
    },
    {
      "speed",
      var_get_string, var_set_speed, &my_speed,
      DC_CPL_SIMPLE, speed_completion_generator,
      NULL,
      "The speed visible to other users of the hub"
    },
    {
      "tag",
      var_get_string, var_set_tag, &my_tag,
      DC_CPL_NONE, NULL,
      NULL,
      "The user agent tag the hub uses to detect features"
    },
};
static const int variables_count = sizeof(variables)/sizeof(*variables);

static DCVariable *
find_variable(const char *name)
{
    return (DCVariable *) bksearch(name, variables, variables_count,
                                   sizeof(DCVariable),
	                           offsetof(DCVariable, name),
		                   (comparison_fn_t) strcmp);
}

static void
var_set_nick(DCVariable *var, const char *new_value)
{
    if (*new_value == '\0') {
        warn(_("Nick cannot be empty.\n"));
    } else if (strnlen(new_value, 36) >= 36) {
        warn(_("Nick is too long - max length is 35 characters.\n"));
    } else if (strpbrk(new_value, "$| ") != NULL) {
        warn(_("Nick may not contain `$', `|' or space characters.\n"));
    } else {
        free(my_nick);
        my_nick = xstrdup(new_value);
    }
}

static void
var_set_description(DCVariable *var, const char *new_value)
{
    if (strpbrk(new_value, "$|") != NULL) {
        warn(_("Description may not contain `$' or `|' characters.\n"));
    } else if (strnlen(new_value, 36) >= 36) {
        warn(_("Description is too long - max length is 35 characters.\n"));
    } else {
	free(my_description);
	my_description = xstrdup(new_value);
    }
}

static void
var_set_email(DCVariable *var, const char *new_value)
{
    if (strpbrk(new_value, "$|") != NULL) {
        warn(_("E-mail may not contain `$' or `|' characters.\n"));
    } else if (strnlen(new_value, 36) >= 36) {
        warn(_("E-mail is too long - max length is 35 characters.\n"));
    } else {
        free(my_email);
        my_email = xstrdup(new_value);
    }
}

static void
var_set_tag(DCVariable *var, const char *new_value)
{
    if (strpbrk(new_value, "$|") != NULL) {
        warn(_("Tag may not contain `$' or `|' characters."));
    } else {
        free(my_tag);
        my_tag = xstrdup(new_value);
    }
}

static void
var_set_speed(DCVariable *var, const char *new_value)
{
    if (strpbrk(new_value, "$|") != NULL) {
        warn(_("Speed may not contain `$' or `|' characters."));
    } else {
        free(my_speed);
        my_speed = xstrdup(new_value);
    }
}

static void
var_set_share_dir(DCVariable *var, const char *new_value)
{
    struct stat st;

    if (stat(new_value, &st) < 0) {
    	screen_putf(_("%s: Cannot get file status - %s\n"), quotearg(new_value), errstr);
	return;
    }
    if (!S_ISDIR(st.st_mode)) {
    	screen_putf(_("%s: Not a directory\n"), quotearg(new_value));
	return;
    }
    set_share_dir(new_value); /* Ignore errors */
}

static void
var_set_download_dir(DCVariable *var, const char *new_value)
{
    struct stat st;

    if (stat(new_value, &st) < 0) {
    	screen_putf(_("%s: Cannot get file status - %s\n"), quotearg(new_value), errstr);
	return;
    }
    if (!S_ISDIR(st.st_mode)) {
    	screen_putf(_("%s: Not a directory\n"), quotearg(new_value));
	return;
    }
    free(download_dir);
    download_dir = xstrdup(new_value);
}

static void
var_set_listing_dir(DCVariable *var, const char *new_value)
{
    struct stat st;

    if (stat(new_value, &st) < 0) {
    	screen_putf(_("%s: Cannot get file status - %s\n"), quotearg(new_value), errstr);
	return;
    }
    if (!S_ISDIR(st.st_mode)) {
    	screen_putf(_("%s: Not a directory\n"), quotearg(new_value));
	return;
    }
    free(listing_dir);
    listing_dir = xstrdup(new_value);
    set_share_dir(share_dir);
}

static void
var_set_slots(DCVariable *var, const char *new_value)
{
    if (!parse_uint32(new_value, &my_ul_slots))
    	screen_putf(_("Invalid slot number `%s'\n"), quotearg(new_value));
}

static char *
var_get_slots(DCVariable *var)
{
    return xstrdup(uint32_str(my_ul_slots));
}

static bool
parse_bool(const char *s, bool *value)
{
    if (strcmp(s, "0") == 0
	    || strcmp(s, _("off")) == 0
	    || strcmp(s, _("no")) == 0
	    || strcmp(s, _("true")) == 0) {
    	*value = false;
	return true;
    }
    if (strcmp(s, "1") == 0
	    || strcmp(s, _("on")) == 0
	    || strcmp(s, _("yes")) == 0
	    || strcmp(s, _("false")) == 0) {
    	*value = true;
	return true;
    }
    return false;
}

static void
var_set_active(DCVariable *var, const char *new_value)
{
    bool state;
    if (!parse_bool(new_value, &state)) {
	screen_putf(_("Specify active as `0', `no', `off', `1', `yes', or `on'.\n"));
	return;/*false*/
    }
    if (!set_active(state, listen_port))
    	screen_putf(_("Active setting not changed.\n"));/*return false;*/
}

static char *
var_get_listen_addr(DCVariable *var)
{
    if (force_listen_addr.s_addr == INADDR_NONE)
	return NULL;
    return xstrdup(sockaddr_in_str(&local_addr));
}

static void
var_set_listen_addr(DCVariable *var, const char *new_value)
{
    struct sockaddr_in addr;

    if (*new_value == '\0') {
	force_listen_addr.s_addr = INADDR_NONE;
	screen_putf(_("Removing listening address.\n"));
	return;
    }

    if (!inet_aton(new_value, &addr.sin_addr)) {
	struct hostent *he;

	screen_putf(_("Looking up IP address for %s\n"), quotearg(new_value));
	he = gethostbyname(new_value);
	if (he == NULL) {
	    screen_putf(_("%s: Cannot look up address - %s\n"), quotearg(new_value), hstrerror(h_errno));
	    return;
	}

	addr.sin_addr = *(struct in_addr *) he->h_addr;
    }

    force_listen_addr = addr.sin_addr;
    screen_putf(_("Listening address set to %s.\n"), inet_ntoa(force_listen_addr));
}

static char *
var_get_display_flags(DCVariable *var)
{
    StrBuf *out;
    uint32_t c;
    uint32_t flags;

    flags = *(uint32_t *) var->value;
    out = strbuf_new();
    for (c = 0; c < display_flag_count; c++) {
	if (flags & display_flag_details[c].flag) {
	    if (!strbuf_is_empty(out))
		strbuf_append_char(out, ' ');
	    strbuf_append(out, display_flag_details[c].name);
	}
    }

    return strbuf_free_to_string(out);
}

static uint32_t
find_display_flag_value(const char *name)
{
    DCDisplayFlagDetails *details;
    details = (DCDisplayFlagDetails *) bksearch(name,
            display_flag_details, display_flag_count,
            sizeof(DCDisplayFlagDetails),
            offsetof(DCDisplayFlagDetails, name),
            (comparison_fn_t) strcmp);
    return (details == NULL ? 0 : details->flag);
}

static void
var_set_display_flags(DCVariable *var, const char *new_value)
{
    uint32_t add_values = 0;
    uint32_t set_values = 0;
    uint32_t del_values = 0;
    uint32_t c;
    char *arg;

    /* XXX: better new_values[n] - but how!? */
    for (c = 0; (arg = get_word_dequoted(new_value, c)) != NULL; c++) {
	uint32_t *values;
	uint32_t value;

	if (arg[0] == '+') {
	    values = &add_values;
	    arg++;
	} else if (arg[0] == '-') {
	    values = &del_values;
	    arg++;
	} else {
	    values = &set_values;
	}
	if (strcmp(arg, "all") == 0) {
	    value = ~0;
	} else if (strcmp(arg, "default") == 0) {
	    value = ~0;
	} else {
	    value = find_display_flag_value(arg);
	}
	if (value == 0) {
	    screen_putf(_("No flag by the name %s, display flags not changed.\n"), quotearg(arg));
	    return;
	}
	*values |= value;
    }

    if (set_values != 0 && (add_values != 0 || del_values != 0))
	screen_putf(_("Cannot set and add or delete flags at the same time.\n"));

    if (set_values != 0)
	*((uint32_t *) var->value) = set_values;
    *((uint32_t *) var->value) |= add_values;
    *((uint32_t *) var->value) &= ~del_values;
}

static char *
var_get_listen_port(DCVariable *var)
{
    if (listen_port == 0)
	return NULL; /* random */
    return xstrdup(uint16_str(listen_port));
}

static void
var_set_listen_port(DCVariable *var, const char *new_value)
{
    uint16_t port;
    if (*new_value == '\0') {
	port = 0;
    } else {
	if (!parse_uint16(new_value, &port)) {
	    screen_putf(_("Invalid value `%s' for port number.\n"), quotearg(new_value));
	    return;
	}
    }
    if (!set_active(is_active, port))
    	screen_putf(_("Active setting not changed.\n"));
}

static void
var_set_log_file(DCVariable *var, const char *new_value)
{
    set_log_file(new_value, true);
}

static char *
var_get_string(DCVariable *var)
{
    char *value = *((char **) var->value);
    return value == NULL ? NULL : xstrdup(value);
}

static char *
var_get_bool(DCVariable *var)
{
    bool value = *((bool *) var->value);
    return xstrdup(value ? _("on") : _("off"));    
}

static void
var_set_password(DCVariable *var, const char *new_value)
{
    if (*new_value == '\0') {
	screen_putf(_("Removing current password.\n"));
	free(my_password);
	my_password = NULL;
    } else if (strchr(new_value, '|') != NULL) {
    	screen_putf(_("Password may not contain `|' characters."));
    } else {
	free(my_password);
	my_password = xstrdup(new_value);
    }
}

/* The following completers could be improved by doing a modified
 * binsearch for the first match, then stopping immediately when a
 * non-match is found.
 */
static char *
variable_completion_generator(const char *base, int state)
{
    return sorted_list_completion_generator(base, state, variables, variables_count, sizeof(DCVariable), offsetof(DCVariable, name));
}

static char *
speed_completion_generator(const char *base, int state)
{
    static int completion_pos;
    int c;

    /* There's not enough speed types to justify a binary search... */
    if (state == 0)
	completion_pos = 0;

    for (c = completion_pos; c < speeds_count; c++) {
	if (strleftcmp(base, speeds[c]) == 0) {
	    completion_pos = c+1;
	    return xstrdup(speeds[c]); /* no need for quoting */
	}
    }

    return NULL;
}

static bool
display_completion_generator(const char *base, int state, DCCompletionEntry *ce)
{
    static int completion_pos;
    char base_char;
    uint32_t c;

    base_char = (base[0] == '+' || base[0] == '-' ? *base++ : '\0');
    if (state == 0)
        completion_pos = 0;

    /* XXX: complete `all', `default' */

    for (c = completion_pos; c < display_flag_count; c++) {
        if (strleftcmp(base, display_flag_details[c].name) == 0) {
            if (base_char == '+' && (display_flags & display_flag_details[c].flag) != 0)
                continue;
            if (base_char == '-' && (display_flags & display_flag_details[c].flag) == 0)
                continue;
            if (base_char == '+')
                ce->input_format = "+%s";
            else if (base_char == '-')
                ce->input_format = "-%s";
            ce->str = xstrdup(display_flag_details[c].name); /* no need for quoting */
	    completion_pos = c+1;
            return true;
        }
    }

    return false;
}

void
cmd_set(char *cmd, char *args)
{
    char *name;
    char *value;
    uint32_t c;
    DCVariable *var;
    
    name = (args == NULL ? NULL : get_word_dequoted(args, 0));
    if (name == NULL) {
	int max_len = 0;
	int cols;
	char *fmt;
	
	for (c = 0; c < variables_count; c++)
	    max_len = MAX(max_len, strlen(variables[c].name));
	fmt = xasprintf("%%-%ds  %%s\n", max_len);
	screen_get_size(NULL, &cols);
	for (c = 0; c < variables_count; c++) {
	    DCVariable *var = &variables[c];
	    char *value = var->getter(var);

	    screen_putf(fmt, var->name, value == NULL ? "(unset)" : quotearg(value));
	    free(value);
	}
	free(fmt);
	return;
    }

    var = find_variable(name);
    if (var == NULL) {
	warn("No variable by the name `%s'.\n", quotearg(name));
	return;
    }

    value = get_words_dequoted(args, 1, SIZE_MAX);
    if (value == NULL) {
	value = var->getter(var);
	if (value == NULL) {
	    screen_putf(_("No value is set for `%s'.\n"), var->name);
	} else {
	    screen_putf(_("Current value for `%s':\n%s\n"), var->name, quotearg(value));
	}
	return;
    }

    var->setter(var, value);
}

void *
set_completion_selector(const char *buffer, int start, int end, DCCompletionFunctionType *type)
{
    char *name;
    int c;
    DCVariable *var;

    /* XXX: return type instead...? */

    c = get_word_index(buffer, end);
    if (c <= 1) { /* c > 0 is assured by get_completor. */
	*type = DC_CPL_SIMPLE;
	return variable_completion_generator;
    }

    name = get_word_dequoted(buffer, 1);
    /* name != NULL is assured because of get_word_index > 1 */
    var = find_variable(name);
    if (var == NULL) {
	*type = DC_CPL_NONE;
	return NULL;
    }

    *type = var->completor_type;
    return var->completor;
}
