/* command.c - Command input and basic commands
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <ctype.h>		/* C89 */
#include <string.h>		/* C89 */
#include <sys/stat.h>		/* POSIX.1: stat */
#include <unistd.h>		/* POSIX.1: fork */
#include <netdb.h>		/* POSIX.1: gethostbyname, h_errno, ?hstrerror? */
#include <arpa/inet.h>		/* BSD 4.3: inet_aton */
#include <dirent.h>
#include <inttypes.h>		/* POSIX.1 (CX): PRI* */
#include "xvasprintf.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#include "quotearg.h"		/* Gnulib */
#include "quote.h"		/* Gnulib */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "stpcpy.h"		/* Gnulib */
#include "common/tmap.h"
#include "common/error.h"
#include "common/strleftcmp.h"
#include "common/intutil.h"
#include "common/minmaxonce.h"
#include "common/strbuf.h"
#include "common/range.h"
#include "common/quoting.h"
#include "common/comparison.h"
#include "common/bksearch.h"
#include "common/swap.h"
#include "common/optparser.h"
#include "microdc.h"

static void cmd_exit(int argc, char **argv);
static void cmd_say(int argc, char **argv);
static void cmd_msg(int argc, char **argv);
static void cmd_raw(int argc, char **argv);
static void cmd_disconnect(int argc, char **argv);
static void cmd_connect(int argc, char **argv);
static void cmd_grantslot(int argc, char **argv);
static void cmd_browse(int argc, char **argv);
static void cmd_pwd(int argc, char **argv);
static void cmd_find(int argc, char **argv);
static void cmd_ls(int argc, char **argv);
static void cmd_cd(int argc, char **argv);
static void cmd_get(int argc, char **argv);
static void cmd_queue(int argc, char **argv);
static void cmd_unqueue(int argc, char **argv);
static void cmd_who(int argc, char **argv);
static void cmd_transfers(int argc, char **argv);
static void cmd_cancel(int argc, char **argv);
static void cmd_search(int argc, char **argv);
static void cmd_status(int argc, char **argv);
static void cmd_results(int argc, char **argv);
static void cmd_unsearch(int argc, char **argv);
static void cmd_alias(int argc, char **argv);
static void cmd_unalias(int argc, char **argv);
static void cmd_shell(int argc, char **argv);
static void shell_completion_selector(const char *line,int ws,int we,char *word,PtrV *results);
static void executable_completion_generator(const char *line,int ws,int we,char *word,PtrV *results);
static void alias_completion_generator(const char *line,int ws,int we,char *word,PtrV *results);
static void command_completion_generator(const char *line,int ws,int we,char *word,PtrV *results);

typedef enum {
    DC_CMD_BUILTIN,
    DC_CMD_ALIAS,
} DCCommandType;

typedef struct _DCCommand DCCommand;
typedef struct _DCAliasCommand DCAliasCommand;
typedef struct _DCBuiltinCommand DCBuiltinCommand;

struct _DCCommand {
    char *name;
    DCCommandType type;
};

struct _DCAliasCommand {
    DCCommand cmd;
    char *alias_spec;
};

struct _DCBuiltinCommand {
    DCCommand cmd;
    DCBuiltinCommandHandler handler;
    DCCompletorFunction completor;
};

static TMap *commands; /* initialized in command_init */

static void
add_builtin_command(const char *name, DCBuiltinCommandHandler handler, DCCompletorFunction completor)
{
    DCBuiltinCommand *builtin = xmalloc(sizeof(DCBuiltinCommand));
    builtin->cmd.name = xstrdup(name); /* XXX: this should be avoidable (pass char *name instead) */
    builtin->cmd.type = DC_CMD_BUILTIN;
    builtin->handler = handler;
    builtin->completor = completor;
    tmap_put(commands, builtin->cmd.name, builtin);
}

static void
add_alias_command(const char *name, const char *alias_spec)
{
    DCAliasCommand *alias = xmalloc(sizeof(DCAliasCommand));
    alias->cmd.name = xstrdup(name);
    alias->cmd.type = DC_CMD_ALIAS;
    alias->alias_spec = xstrdup(alias_spec);
    tmap_put(commands, alias->cmd.name, alias);
}

static void
free_command(DCCommand *cmd)
{
    if (cmd->type == DC_CMD_ALIAS) {
	DCAliasCommand *alias = (DCAliasCommand *) cmd;
	free(alias->alias_spec);
    }
    free(cmd->name);
    free(cmd);
}

void
command_finish(void)
{
    tmap_foreach_value(commands, free_command);
    tmap_free(commands);
}

void
command_init(void)
{
    commands = tmap_new();
    tmap_set_compare_fn(commands, (comparison_fn_t) strcmp);
    add_builtin_command("browse", cmd_browse, user_or_myself_completion_generator);
    add_builtin_command("cancel", cmd_cancel, transfer_completion_generator);
    add_builtin_command("cd", cmd_cd, remote_dir_completion_generator);
    add_builtin_command("connect", cmd_connect, NULL);
    add_builtin_command("disconnect", cmd_disconnect, NULL);
    add_builtin_command("exit", cmd_exit, NULL);
    add_builtin_command("find", cmd_find, remote_path_completion_generator);
    add_builtin_command("get", cmd_get, remote_path_completion_generator);
    add_builtin_command("grantslot", cmd_grantslot, user_completion_generator);
    add_builtin_command("ls", cmd_ls, remote_path_completion_generator);
    add_builtin_command("msg", cmd_msg, user_completion_generator);
    add_builtin_command("pwd", cmd_pwd, NULL);
    add_builtin_command("queue", cmd_queue, user_with_queue_completion_generator);
    add_builtin_command("raw", cmd_raw, NULL);
    add_builtin_command("results", cmd_results, NULL);
    add_builtin_command("say", cmd_say, say_user_completion_generator);
    add_builtin_command("search", cmd_search, NULL);
    add_builtin_command("set", cmd_set, set_completion_selector);
    add_builtin_command("status", cmd_status, NULL);
    add_builtin_command("transfers", cmd_transfers, NULL);
    add_builtin_command("unqueue", cmd_unqueue, user_completion_generator);
    add_builtin_command("unsearch", cmd_unsearch, NULL);
    add_builtin_command("who", cmd_who, user_completion_generator);
    add_builtin_command("alias", cmd_alias, NULL);
    add_builtin_command("unalias", cmd_unalias, alias_completion_generator);
    add_builtin_command("shell", cmd_shell, shell_completion_selector);
    add_alias_command("ll", "ls -l");
}

void
default_completion_selector(const char *line,int ws,int we,char *word,PtrV *results)
{
    const char *newline;

    if (find_unquoted_leading_char(line, line+we, '#') != NULL)
	return; /* completing in commented out text */

    newline = find_last_unquoted_char(line, line+we, ';');
    if (newline != NULL) {
	newline = find_word_start(newline+1, line+we);
	ws -= newline-line;
	we -= newline-line;
	line = newline;
    }
    if (get_word_index(line, ws) == 0) {
        command_completion_generator(line,ws,we,word,results);
    } else {
        DCCommand *cmd;
        char *name;

        name = get_word_dequoted(line, 0);
        cmd = tmap_get(commands, name);
        free(name);
        if (cmd != NULL) {
            if (cmd->type == DC_CMD_BUILTIN) {
                DCBuiltinCommand *builtin = (DCBuiltinCommand *) cmd;
		if (builtin->completor != NULL)
		    builtin->completor(line,ws,we,word,results);
            } else if (cmd->type == DC_CMD_ALIAS) {
                DCAliasCommand *alias = (DCAliasCommand *) cmd;
                int sizemod;
                char *newline;

                sizemod = strlen(alias->alias_spec) - strlen(cmd->name);
                newline = xasprintf("%s%s", alias->alias_spec, find_word_end(line, NULL));
                default_completion_selector(newline,ws+sizemod,we+sizemod,word,results);
                free(newline);
            }
        }
    }
}

static void
alias_completion_generator(const char *line,int ws,int we,char *word,PtrV *results)
{
    TMapIterator it;

    tmap_iterator_partial(commands, &it, word, (comparison_fn_t) strleftcmp);
    while (it.has_next(&it)) {
        DCCommand *cmd = it.next(&it);
        if (cmd->type == DC_CMD_ALIAS)
            ptrv_append(results, new_completion_entry(cmd->name, NULL));
    }
}

static void
command_completion_generator(const char *line,int ws,int we,char *word,PtrV *results)
{
    TMapIterator it;

    tmap_iterator_partial(commands, &it, word, (comparison_fn_t) strleftcmp);
    while (it.has_next(&it)) {
        DCCommand *cmd = it.next(&it);
        ptrv_append(results, new_completion_entry(cmd->name, NULL));
    }
}

void
command_execute(const char *line)
{
    for (;;) {
        PtrV *args;
	DCCommand *cmd;
	char *name;

	line = find_word_start(line, NULL);
	if (*line == '\0' || *line == '#')
	    break;
	if (*line == ';') {
	    line++;
	    continue;
	}

	name = get_word_dequoted_termchar(line, 0, ';'); /* won't return NULL */
	cmd = tmap_get(commands, name);
	if (cmd != NULL && cmd->type == DC_CMD_ALIAS) {
	    DCAliasCommand *alias = (DCAliasCommand *) cmd;
	    char *newline;

	    newline = xasprintf("%s%s", alias->alias_spec, find_word_end(line, NULL));
	    free(name);
	    command_execute(newline);
	    free(newline);
	    return;
	}

	/* Process arguments until ";" or "#" or end of line. */
        args = ptrv_new();
	ptrv_append(args, name);
	for (;;) {
	    line = find_word_start(find_word_end_termchar(line, NULL, ';'), NULL);
	    if (*line == '\0' || *line == '#' || *line == ';')
		break;
	    ptrv_append(args, get_word_dequoted_termchar(line, 0, ';'));
	}
	ptrv_append(args, NULL);

	if (cmd == NULL) {
	    warn(_("%s: Unknown command.\n"), quotearg(name));
	} else {
	    DCBuiltinCommand *builtin = (DCBuiltinCommand *) cmd;
	    /*int c;
	    screen_putf("Execute: <%s>", cmd->name);
	    for (c = 0; c < args->cur-1; c++)
		screen_putf(" [%s]", (char *) args->buf[c]);
	    screen_putf("\n");*/
	    builtin->handler(args->cur-1, (char **) args->buf);
	}

	ptrv_foreach(args, free); /* will free(name) as that is args->buf[0] */
	ptrv_free(args);
    }
}

/* This is not exactly the same as bash, I consider it better:
 *
 *  [1] Allow completion of dirs relative to CWD always (bash: only in
 *      last resort, when there are no exefile matches).
 *  [2] When a dir has been completed (dirpath!=NULL, line contains at
 *      least one /), then allow completion of any dirs+exefiles relative
 *      to that dir. 
 *  [3] Only allow completion of exefiles relative to CWD if PATH
 *      contains `.'.
 *
 * Bash completes like [2] and [3]. But bash deals with things differently
 * when it comes to [1]: When PATH does not contain ., then completion of
 * directories in CWD is only used as a last resort if there are no
 * executable matches in other dirs in PATH.
 */
static void
executable_completion_generator(const char *line,int ws,int we,char *word,PtrV *results)
{
    char *path;
    char *p1;
    char *p;
    const char *file_part;
    char *dir_part;
    bool path_has_cwd = false;

    get_file_dir_part(word, &dir_part, &file_part);

    path = getenv("PATH");
    if (path == NULL) {
	/*path = xconfstr(_CS_PATH);
	if (path == NULL || *path == '\0')*/
	    return;
    } else {
	path = xstrdup(path);
    }

    for (p1 = path; (p = strsep(&p1, ":")) != NULL; ) {
        DIR *dh;

        if (strcmp(p, ".") == 0)
            path_has_cwd = true;
        dh = opendir(p); /* ignore errors */
        if (dh != NULL) {
            struct dirent *ent;
            while ((ent = readdir(dh)) != NULL) { /* ignore errors */
                if (strleftcmp(word, ent->d_name) == 0) {
                    char *full;
                    full = catfiles(p, ent->d_name);
                    if (access(full, X_OK) == 0) { /* ignore errors */
                        struct stat sb;
                        if (stat(full, &sb) >= 0 && S_ISREG(sb.st_mode))
                            ptrv_append(results, new_completion_entry(ent->d_name, ent->d_name));
                    }
                    /* XXX: use is_directory instead of that stat above */
                    free(full);
                }
            }
            closedir(dh);
        }
    }

    if (*dir_part != '\0' || path_has_cwd) {
        local_fs_completion_generator(line,ws,we,word,results,DC_CPL_DIR|DC_CPL_EXE|DC_CPL_DOT);
    } else if (results->cur == 0) {
        local_fs_completion_generator(line,ws,we,word,results,DC_CPL_DIR|DC_CPL_DOT);
    }

    free(dir_part);
    free(path);
}

static void
shell_completion_selector(const char *line,int ws,int we,char *word,PtrV *results)
{
    if (get_word_index(line, ws) == 1) {
        executable_completion_generator(line,ws,we,word,results);
    } else {
        local_path_completion_generator(line,ws,we,word,results);
    }
}

static void
cmd_shell(int argc, char **argv)
{
    screen_suspend();
    shell_child = fork();
    if (shell_child < 0) {
	warn(_("Cannot create child process - %s\n"), errstr);
	return;
    }
    if (shell_child == 0) {
        if (execvp(argv[1], argv+1) < 0)
	    die(_("Cannot execute - %s\n"), errstr); /* die OK */
        exit(EXIT_FAILURE); /* shouldn't get here */
    }
}

static void
cmd_status(int argc, char **argv)
{
    char *fmt1;
    HMapIterator it;
    uint32_t c;

    switch (hub_state) {
    case DC_HUB_DISCONNECTED:
	screen_putf(_("Hub state: %s\n"), _("Not connected"));
	break;
    case DC_HUB_CONNECT:
	screen_putf(_("Hub state %s\n"), _("Waiting for complete connection"));
	break;
    case DC_HUB_LOCK:
	screen_putf(_("Hub state: %s\n"), _("Waiting for $Lock"));
	break;
    case DC_HUB_HELLO:
	screen_putf(_("Hub state: %s\n"), _("Waiting for $Hello"));
	break;
    case DC_HUB_LOGGED_IN:
	screen_putf(_("Hub state: %s\n"), _("Logged in"));
	break;
    }

    if (hub_state >= DC_HUB_LOGGED_IN) {
        screen_putf(_("Hub users: %s\n"), uint32_str(hmap_size(hub_users)));
    } else {
        screen_putf(_("Hub users: %s\n"), _("(not logged in)"));
    }

    screen_putf(_("Pending user connections:\n"));
    hmap_iterator(pending_userinfo, &it);
    for (c = 0; it.has_next(&it); c++)
        screen_putf("  %s\n", quotearg(it.next(&it)));
    if (c == 0)
        screen_putf(_("  (none)\n"));

    fmt1 = ngettext("byte", "bytes", my_share_size);
    screen_putf(_("Share size: %" PRId64" %s (%" PRId64 " MB).\n"),
                my_share_size, fmt1, my_share_size/(1024*1024));
}

static void
cmd_exit(int argc, char **argv)
{
    /*if (argc > 1) {
	warn(_("%s: too many arguments\n"), argv[0]);
	return;
    }*/
    running = false;
}

static void
cmd_say(int argc, char **argv)
{
    char *msg;
    char *tmp;

    if (argc <= 1) {
    	screen_putf(_("Usage: %s MESSAGE..\n"), argv[0]);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    tmp = join_strings(argv+1, argc-1, ' ');
    msg = escape_message(tmp);
    free(tmp);
    hub_putf("<%s> %s|", my_nick, msg); /* Ignore error */
    free(msg);
}

static void
cmd_msg(int argc, char **argv)
{
    char *msg;
    char *tmp;
    DCUserInfo *ui;

    if (argc <= 2) {
    	screen_putf(_("Usage: %s USER MESSAGE..\n"), argv[0]);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    ui = hmap_get(hub_users, argv[1]);
    if (ui == NULL) {
    	screen_putf(_("%s: No such user on this hub\n"), quotearg(argv[1]));
	return;
    }

    tmp = join_strings(argv+2, argc-2, ' ');
    msg = escape_message(tmp);
    free(tmp);
    hub_putf("$To: %s From: %s $<%s> %s|", ui->nick, my_nick, my_nick, tmp); /* Ignore error */
    free(msg);
}

static void
cmd_raw(int argc, char **argv)
{
    char *msg;
    char *tmp;
    
    if (argc <= 1) {
    	screen_putf(_("Usage: %s DATA...\n"), argv[0]);
	return;
    }
    if (hub_state < DC_HUB_LOCK) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    tmp = join_strings(argv+1, argc-1, ' ');
    msg = escape_message(tmp);
    free(tmp);
    hub_putf("%s", msg); /* Ignore error */
    free(msg);
}

static void
cmd_connect(int argc, char **argv)
{
    char *portstr;
    struct sockaddr_in addr;

    if (argc == 1) {
    	screen_putf(_("Usage: %s HOST[:PORT]\n"), argv[0]);
	return;
    }
    if (hub_state != DC_HUB_DISCONNECTED) {
    	screen_putf(_("Already connected or connecting, disconnect first.\n"));
	return;
    }

    portstr = strchr(argv[1], ':');
    if (portstr != NULL) {
    	*portstr = '\0';
    	if (!parse_uint16(portstr+1, &addr.sin_port)) {
	    screen_putf(_("Invalid port number %s\n"), quote(portstr));
	    return;
	}
	addr.sin_port = htons(addr.sin_port);
    } else {
    	addr.sin_port = htons(DC_HUB_TCP_PORT);
    }

    if (!inet_aton(argv[1], &addr.sin_addr)) {
    	struct hostent *he;

	screen_putf(_("Looking up IP address for %s\n"), quotearg(argv[1]));
	he = gethostbyname(argv[1]);
	if (he == NULL) {
    	    screen_putf(_("%s: Cannot look up address - %s\n"), quotearg(argv[1]), hstrerror(h_errno));
	    return;
	}
    	addr.sin_addr = *(struct in_addr *) he->h_addr;
    }

    addr.sin_family = AF_INET;

    hub_connect(&addr); /* Ignore errors */
}

static void
cmd_disconnect(int argc, char **argv)
{
    if (hub_state == DC_HUB_DISCONNECTED) {
    	warn(_("Not connected.\n"));
    } else {
    	warn(_("Disconnecting from hub.\n"));
	hub_disconnect();
    }
}

static void
cmd_grantslot(int argc, char **argv)
{
    uint32_t c;

    if (argc == 1) {
    	screen_putf(_("Usage: %s USER...\n"), argv[0]);
	return;
    }
    if (hub_state != DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    for (c = 1; c < argc; c++) {
        DCUserInfo *ui;

        ui = hmap_get(hub_users, argv[c]);
        if (ui == NULL) {
    	    screen_putf(_("%s: No such user on this hub\n"), quotearg(argv[c]));
    	    return;
        }
        ui->slot_granted = true;
    }
}

/* XXX: move queue.c/browse.c? */
void
browse_none(void)
{	
    /* Clean up previous browse. */
    if (browse_list != NULL) {
	filelist_free(browse_list);
	browse_list = NULL;
	free(browse_path);
	browse_path = NULL;
	free(browse_path_previous);
	browse_path_previous = NULL;
    }
    if (browse_user != NULL) {
	user_info_free(browse_user);
	browse_user = NULL;
    }
    browsing_myself = false;
}

/* XXX: move to queue.c */
static int
queued_file_cmp(const char *filename, DCQueuedFile *qf)
{
    return strcmp(filename, qf->filename);
}

/* XXX: move queue.c/browse.c? */
static void
cmd_browse(int argc, char **argv)
{
    DCUserInfo *ui;
    char *filename;
    struct stat st;

    if (argc == 1) {
    	if (!browsing_myself && browse_user == NULL) {
	    screen_putf(_("Not browsing any user.\n"));
	    return;
	}
	browse_none();
	update_prompt();
	return;
    }

    if (strcmp(my_nick, argv[1]) == 0) {
	browse_none();
	browse_list = our_filelist;
	browse_path = xstrdup("/");
	browse_path_previous = NULL;
	browse_user = NULL;
	browsing_myself = true;
	update_prompt();
	return;
    }

    ui = hmap_get(hub_users, argv[1]);
    if (ui == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), quotearg(argv[1]));
        return;
    }

    filename = xasprintf("%s/%s.MyList.DcLst", listing_dir, ui->nick);
    if (stat(filename, &st) < 0) {
	if (errno != ENOENT) {
            screen_putf(_("%s: Cannot get file status - %s\n"), quotearg(filename), errstr);
            free(filename);
            return;
	}

        free(filename);
	if (ptrv_find(ui->download_queue, "/MyList.DcLst", (comparison_fn_t) queued_file_cmp) < 0) {
	    DCQueuedFile *queued = xmalloc(sizeof(DCQueuedFile));

	    queued->filename = xstrdup("/MyList.DcLst");
	    queued->base_path = xstrdup("/");
	    queued->flag = DC_TF_LIST;
	    ptrv_prepend(ui->download_queue, queued);
	}
    	if (!has_user_conn(ui, DC_DIR_RECEIVE) && ui->conn_count < DC_USER_MAX_CONN)
	    hub_connect_user(ui); /* Ignore errors */
	else
	    screen_putf(_("No free connections. Queued file for download.\n"));

	browse_none();
	browse_user = ui;
	browsing_myself = false;
	ui->refcount++;
	update_prompt();
	return;
    }

    browse_none();
    browse_list = filelist_open(filename);
    browse_path = xstrdup("/");
    browse_path_previous = NULL;
    browse_user = ui;
    browsing_myself = false;
    ui->refcount++;
    free(filename);
    update_prompt();
}

/* XXX: move queue.c/browse.c? */
static void
cmd_pwd(int argc, char **argv)
{
    if (browse_list == NULL) {
	if (browse_user == NULL) {
	    screen_putf(_("Not browsing any user.\n"));
	} else {
	    screen_putf(_("(%s) Waiting for file list.\n"), quotearg(browse_user->nick));
	}
    } else {
	screen_putf(_("(%s) %s\n"),
		quotearg_n(0, browsing_myself ? my_nick : browse_user->nick),
		quotearg_n(1, browse_path)); /* XXX: print as %s:%s instead */
    }
}

/* XXX: move queue.c/browse.c? */
static void
cmd_cd(int argc, char **argv)
{
    DCFileList *node;
    char *path;

    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }

    if (argc == 1) {
	path = xstrdup("/");
    } else if (strcmp(argv[1], "-") == 0) {
	if (browse_path_previous == NULL) {
	    warn(_("No previous path.\n"));
	} else {
	    swap(browse_path, browse_path_previous);
	    update_prompt();
	}
	return;
    } else {
	path = apply_cwd(argv[1]);
    }

    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
    	screen_putf(_("%s: No such file or directory\n"), quotearg(argv[1]));
	free(path);
	return;
    }
    if (node->type != DC_TYPE_DIR) {
    	screen_putf(_("%s: Not a directory\n"), quotearg(argv[1]));
	free(path);
	return;
    }
    free(browse_path_previous);
    browse_path_previous = browse_path;
    browse_path = filelist_get_path(node);
    update_prompt();
}

/* XXX: move queue.c/browse.c? */
static void
cmd_find(int argc, char **argv)
{
    char *path;
    DCFileList *node;

    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    if (argc == 1) {
        path = xstrdup(browse_path);
    } else {
	path = apply_cwd(argv[1]);
    }

    node = filelist_lookup(browse_list, path);
    free(path);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), quotearg(argv[1]));
	return;
    }
    filelist_list_recursively(node, "", 1);
}

/* XXX: move queue.c/browse.c? */
static void
list_directory(const char *path, bool long_mode)
{
    char *abspath;
    DCFileList *node;

    abspath = apply_cwd(path);
    node = filelist_lookup(browse_list, abspath);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), quotearg(path));
    } else {
	filelist_list(node, long_mode);
    }
    free(abspath);
}

/* XXX: move queue.c/browse.c? */
static void
cmd_ls(int argc, char **argv)
{
    OptDetail long_opts[] = {
	{ "l", OPTP_NO_ARG, 'l' },
	{ "help", OPTP_NO_ARG, 'h' },
	{ NULL },
    };
    OptParser *p;
    bool long_mode = false;

    p = optparser_new(long_opts, -1, 0);
    optparser_parse(p, argc, argv);
    while (optparser_has_next(p)) {
	switch (optparser_next(p)) {
	case 'l':
	    long_mode = true;
	    break;
	case 'h':
	    screen_putf("Usage: ls [OPTION]... [FILE]...\n");
	    screen_putf("  -l          use a long listing format\n");
	    screen_putf("      --help  display this help and exit\n");
	    optparser_free(p);
	    return;
	}
    }
    if (optparser_error(p) != NULL) {
	screen_putf("%s: %s\n", argv[0], optparser_error(p));
	return;
    }
    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    if (!optparser_has_next_arg(p))
	list_directory(browse_path, long_mode);
    while (optparser_has_next_arg(p)) {
	char *arg = optparser_next_arg(p);
	list_directory(arg, long_mode);
    }
    optparser_free(p);
}

/* XXX: move to download.c/queue.c/browse.c? */
static bool
append_download_dir(DCUserInfo *ui, DCFileList *node, char *base_path, uint64_t *bytes, uint32_t *files)
{
    if (node->type == DC_TYPE_REG) {
	DCQueuedFile *queued;
    	char *path = filelist_get_path(node);

	if (ptrv_find(ui->download_queue, path, (comparison_fn_t) queued_file_cmp) >= 0) {
	    screen_putf(_("Queue already contains this file, ignoring\n"));
	    free(path);
	    return false;
	}

	queued = xmalloc(sizeof(DCQueuedFile));
	queued->filename = path;
	queued->base_path = xstrdup(base_path);
	queued->flag = DC_TF_NORMAL;
	ptrv_append(ui->download_queue, queued);

	(*bytes) += node->u.reg.size;
	(*files) ++;
	return true;
    }
    if (node->type == DC_TYPE_DIR) {
    	HMapIterator it;
	bool s = false;

	hmap_iterator(node->u.dir.children, &it);
	while (it.has_next(&it))
	    s = append_download_dir(ui, it.next(&it), base_path, bytes, files) || s;
	return s;
    }
    return false;
}

/* XXX: move to download.c/queue.c/browse.c? */
static void
cmd_get(int argc, char **argv)
{
    DCFileList *node;
    char *path;
    const char *file_name;
    uint64_t bytes = 0;
    uint32_t files = 0;

    /* Note: This command assumes that we are browsing some user.
     * If we in the future would allow browse-less getting,
     * i.e. get NICK FILE, then we must here add additional checks
     * found in cmd_browse, such as strcmp(my_nick, nick)==0 etc.
     */

    file_name = argv[1];
    if (file_name == NULL) {
    	screen_putf(_("Usage: %s FILE\n"), argv[0]);
	return;
    }
    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    if (browsing_myself) {
	screen_putf(_("Cannot download files from myself.\n"));
	return;
    }

    path = apply_cwd(file_name);
    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), quotearg(argv[1]));
	free(path);
	return;
    }
    free(path);

    if (append_download_dir(browse_user, node, browse_path, &bytes, &files)) {
        char *fmt1 = ngettext("byte", "bytes", bytes);
        char *fmt2 = ngettext("file", "files", files);
    	path = filelist_get_path(node);
	screen_putf(_("Downloading %s (%" PRId64 " %s in %d %s)\n"), quotearg(path), bytes, fmt1, files, fmt2);
	free(path);
    } else {
        /* XXX: hmm, shouldn't we return even if queue is non-empty? */
    	if (browse_user->download_queue->cur == 0) {
	    screen_putf(_("No files to download (empty directories).\n"));
	    return;
	}
    }

    if (!has_user_conn(browse_user, DC_DIR_RECEIVE) && browse_user->conn_count < DC_USER_MAX_CONN) {
    	hub_connect_user(browse_user); /* Ignore errors */
    } else {
	screen_putf(
          ngettext("No free connections. Queued file for download.\n",
	  	   "No free connections. Queued files for download.\n", files));
    }
}

/* XXX: move to download.c/queue.c/browse.c? */
static void
cmd_queue(int argc, char **argv)
{
    uint32_t c;
    DCUserInfo *ui;

    if (argc == 1) {
    	screen_putf(_("Usage: %s USER\n"), argv[0]);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    ui = hmap_get(hub_users, argv[1]);
    if (ui == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), quotearg(argv[1]));
	return;
    }
    for (c = 0; c < ui->download_queue->cur; c++) {
	DCQueuedFile *queued = ui->download_queue->buf[c];
	screen_putf(_("%d. [ %s ] %s\n"),
		c+1,
		quotearg_n(0, queued->base_path),
		quotearg_n(1, queued->filename + strlen(queued->base_path)));
    }
}

/* XXX: move to download.c/queue.c/browse.c? */
static void
removed_queued_by_range(uint32_t sp, uint32_t ep, void *userdata)
{
    DCUserInfo *user = userdata;
    uint32_t c;

    for (c = sp-1; c < ep; c++) {
	if (user->download_queue->buf[c] != NULL) {
	    free_queued_file(user->download_queue->buf[c]);
	    user->download_queue->buf[c] = NULL;
	}
    }
}

/* XXX: move to download.c/queue.c/browse.c? */
static void
compact_queue_ptrv(DCUserInfo *user)
{
    int32_t first_free = -1;
    uint32_t c;

    for (c = 0; c < user->download_queue->cur; c++) {
	if (user->download_queue->buf[c] == NULL) {
	    if (first_free == -1)
		first_free = c;
	} else {
	    if (first_free != -1) {
		user->download_queue->buf[first_free] = user->download_queue->buf[c];
		user->download_queue->buf[c] = NULL;
		for (first_free++; first_free < c; first_free++) {
		    if (user->download_queue->buf[first_free] == NULL)
			break;
		}
	    }
	}
    }

    if (first_free != -1)
	user->download_queue->cur = first_free;
}    

/* XXX: move to download.c/queue.c/browse.c? */
static void
cmd_unqueue(int argc, char **argv)
{
    DCUserInfo *user;
    const char *range;

    if (argc == 1) {
    	screen_putf(_("Usage: %s USER [RANGE]\n"), argv[0]);
	return;
    }

    /* XXX: parse each argument, allow RANGE RANGE2 .. */
    if (argc > 2) {
        range = argv[2];
    } else {
	range = "1-";
    }

    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    user = hmap_get(hub_users, argv[1]);
    if (user == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), quotearg(argv[1]));
        return;
    }
    if (!foreach_in_range(range, 1, user->download_queue->cur, NULL, NULL)) {
	screen_putf(_("%s: Invalid range, or index out of range (1-%d)\n"), quotearg(range), user->download_queue->cur);
	return;
    }
    foreach_in_range(range, 1, user->download_queue->cur, removed_queued_by_range, user);
    compact_queue_ptrv(user);
}

static int
user_info_compare(const void *i1, const void *i2)
{
    const DCUserInfo *f1 = *(const DCUserInfo **) i1;
    const DCUserInfo *f2 = *(const DCUserInfo **) i2;

    return strcmp(f1->nick, f2->nick);
}

#define IFNULL(a,x) ((a) == NULL ? (x) : (a))

static void
cmd_who(int argc, char **argv)
{
    uint32_t maxlen;
    HMapIterator it;
    uint32_t c;
    DCUserInfo **items;
    uint32_t count;
    uint32_t cols;
    StrBuf *out;

    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }

    if (argc > 1) {
    	DCUserInfo *ui;

	ui = hmap_get(hub_users, argv[1]);
	if (ui == NULL) {
    	    screen_putf(_("%s: No such user on this hub\n"), quotearg(argv[1]));
	} else {
	    char *fmt1 = ngettext("byte", "bytes", ui->share_size);
    	    screen_putf(_("Nick: %s\n"), quotearg(ui->nick));
    	    screen_putf(_("Description: %s\n"), quotearg(IFNULL(ui->description, "")));
    	    screen_putf(_("Speed: %s\n"), quotearg(IFNULL(ui->speed, "")));
    	    screen_putf(_("Level: %d\n"), ui->level);
    	    screen_putf(_("E-mail: %s\n"), quotearg(IFNULL(ui->email, "")));
    	    screen_putf(_("Operator: %d\n"), ui->is_operator);
    	    screen_putf(_("Share Size: %" PRId64 " %s (%" PRId64 " MB)\n"),
    	        ui->share_size, fmt1, ui->share_size/(1024*1024));
	}
    	return;
    }

    maxlen = 0;
    for (hmap_iterator(hub_users, &it); it.has_next(&it); ) {
    	DCUserInfo *ui = it.next(&it);
	maxlen = max(maxlen, strlen(quotearg(ui->nick)));
    }

    count = hmap_size(hub_users);
    items = xmalloc(count * sizeof(DCUserInfo *));
    hmap_iterator(hub_users, &it);
    for (c = 0; c < count; c++)
	items[c] = it.next(&it);
    qsort(items, count, sizeof(DCUserInfo *), user_info_compare);

    screen_get_size(NULL, &cols);

    out = strbuf_new();
    for (c = 0; c < count; c++) {
    	DCUserInfo *ui = items[c];
	char *nick = quotearg(ui->nick);

	strbuf_clear(out);
	strbuf_append(out, nick);
	strbuf_append_char_n(out, maxlen+1-strlen(nick), ' ');
	strbuf_appendf(out, "  %7" PRId64 "M", ui->share_size / (1024*1024));
	strbuf_append(out, ui->is_operator ? " op" : "   ");
	if (ui->download_queue->cur > 0)
	    strbuf_appendf(out, " (%3d)", ui->download_queue->cur);
	else
	    strbuf_append(out, "      ");
	strbuf_appendf(out, " %s", quotearg(ui->description ? ui->description : ""));
	if (strbuf_length(out) > cols)
	    strbuf_set_length(out, cols);
	screen_putf("%s\n", strbuf_buffer(out));
    }
    free(items);
    strbuf_free(out);
}

static void
cmd_transfers(int argc, char **argv)
{
    char *format;
    HMapIterator it;
    uint32_t maxlen = 0;
    time_t now = 0;

    hmap_iterator(user_conns, &it);
    while (it.has_next(&it)) {
    	DCUserConn *uc = it.next(&it);
	maxlen = max(maxlen, strlen(uc->name));
    }

    format = xasprintf("%%-%ds  %%s\n", maxlen);

    if (time(&now) == (time_t) -1) {
    	warn(_("Cannot get current time - %s\n"), errstr);
	return;
    }

    hmap_iterator(user_conns, &it);
    while (it.has_next(&it)) {
    	DCUserConn *uc = it.next(&it);
	char *status;

    	/*if (!get_user_conn_status(uc))
	    continue;*/
	status = user_conn_status_to_string(uc, now);
	screen_putf(format, quotearg(uc->name), status);
	free(status);
    }

    screen_putf(_("Upload slots: %d/%d  Download slots: %d/unlimited\n"), used_ul_slots, my_ul_slots, used_dl_slots);

    free(format);
}

static void
cmd_cancel(int argc, char **argv)
{
    DCUserConn *uc;

    if (argc == 1) {
    	screen_putf(_("Usage: %s CONNECTION\n"), argv[0]);
	return;
    }

    uc = hmap_get(user_conns, argv[1]);
    if (uc == NULL) {
    	screen_putf(_("%s: No such user connection.\n"), quotearg(argv[1]));
	return;
    }

    user_conn_cancel(uc);
}

static void
cmd_search(int argc, char **argv)
{
    char *tmp;
    
    if (argc == 1) {
    	screen_putf(_("Usage: %s STRING...\n"), argv[0]);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    
    tmp = join_strings(argv+1, argc-1, ' ');
    add_search_request(tmp); /* Ignore errors */
    free(tmp);
}

static void
cmd_results(int argc, char **argv)
{
    uint32_t c;

    if (argc == 1) {
        time_t now;
    
	if (time(&now) == (time_t) -1) {
            warn(_("Cannot get current time - %s\n"), errstr);
            return;
        }

    	for (c = 0; c < our_searches->cur; c++) {
    	    DCSearchRequest *sd = our_searches->buf[c];
    	    char *status;
    	    char *spec;

            spec = search_selection_to_string(&sd->selection);
            status = sd->issue_time + SEARCH_TIME_THRESHOLD <= now
                   ? _("Closed") : _("Open");
	    screen_putf(_("%d. %s (%s) Results: %d\n"), c+1, quotearg(spec),
                   status, sd->responses->cur);
	}
    } else {
    	DCSearchRequest *sd;
	if (!parse_uint32(argv[1], &c) || c == 0 || c-1 >= our_searches->cur) {
    	    screen_putf(_("Invalid search index.\n"));
	    return;
	}
	sd = our_searches->buf[c-1];
	for (c = 0; c < sd->responses->cur; c++) {
	    DCSearchResponse *sr = sd->responses->buf[c];
	    char *n;
	    char *t;

	    n = translate_remote_to_local(sr->filename);
	    if (sr->filetype == DC_TYPE_DIR) /* XXX: put into some function */
	    	t = "/";
	    else
	    	t = "";
	    screen_putf("%d. %s %s%s\n", c+1, quotearg(sr->userinfo->nick), n, t);
	    free(n);
	}
    }
}

static void
cmd_unsearch(int argc, char **argv)
{
    DCSearchRequest *sd;
    uint32_t index;

    if (argc == 1) {
    	screen_putf(_("Usage: %s INDEX\n"), argv[0]);
	return;
    }
    if (!parse_uint32(argv[1], &index) || index == 0 || index-1 >= our_searches->cur) {
    	screen_putf(_("Invalid search index %s.\n"), quote(argv[1]));
	return;
    }

    sd = our_searches->buf[index-1];
    ptrv_remove_range(our_searches, index-1, index);
    free_search_request(sd);    
}

static void
cmd_alias(int argc, char **argv)
{
    if (argc == 1) {
	TMapIterator it;

	for (tmap_iterator(commands, &it); it.has_next(&it); ) {
	    DCCommand *cmd = it.next(&it);
	    if (cmd->type == DC_CMD_ALIAS) {
		DCAliasCommand *alias = (DCAliasCommand *) cmd;
		screen_putf("alias %s \"%s\"\n" /*no translation */, cmd->name, quotearg(alias->alias_spec));
	    }
	}
    } else if (argc == 2) {
	DCCommand *cmd = tmap_get(commands, argv[1]);
	if (cmd != NULL && cmd->type == DC_CMD_ALIAS) {
	    DCAliasCommand *alias = (DCAliasCommand *) cmd;
	    screen_putf("alias %s \"%s\"\n" /*no translation */, argv[1], quotearg(alias->alias_spec));
	} else {
	    warn(_("%s: No such alias.\n"), quotearg(argv[1]));
	}
    } else if (argc == 3) {
	DCCommand *cmd = tmap_get(commands, argv[1]);
	if (cmd == NULL) {
	    DCAliasCommand *alias = xmalloc(sizeof(DCAliasCommand));
	    alias->cmd.name = xstrdup(argv[1]);
	    alias->cmd.type = DC_CMD_ALIAS;
	    alias->alias_spec = xstrdup(argv[2]);
	    tmap_put(commands, alias->cmd.name, alias);
	} else if (cmd->type == DC_CMD_ALIAS) {
	    DCAliasCommand *alias = (DCAliasCommand *) cmd;
	    free(alias->alias_spec);
	    alias->alias_spec = xstrdup(argv[2]);
	} else {
	    warn(_("%s: Cannot override built-in command.\n"), quotearg(cmd->name));
	}
    } else {
        warn(_("%s: Too many arguments.\n"), argv[0]);
    }
}

static void
cmd_unalias(int argc, char **argv)
{
    if (argc == 1) {
    	screen_putf(_("Usage: %s ALIAS\n"), argv[0]);
    } else {
	DCCommand *cmd = tmap_get(commands, argv[1]);
	if (cmd == NULL || cmd->type != DC_CMD_ALIAS) {
	    warn(_("%s: No such alias.\n"), quotearg(argv[1]));
	} else {
	    DCAliasCommand *alias = (DCAliasCommand *) cmd;
	    tmap_remove(commands, cmd->name);
	    free(alias->cmd.name);
	    free(alias->alias_spec);
	    free(alias);
	}
    }
}

void
update_prompt(void)
{
    if (browsing_myself || browse_user != NULL) {
	char *nick = browsing_myself ? my_nick : browse_user->nick;
	
	if (browse_list == NULL) {
	    set_screen_prompt("%s:(%s)> ", PACKAGE, quotearg(nick));
	} else {
	    set_screen_prompt("%s:%s:%s> ", PACKAGE, quotearg_n(0, nick), quotearg_n(1, browse_path));
	}
    } else {
	set_screen_prompt("%s> ", PACKAGE);
    }
}
