/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>

#include "srecord-uploader.h"
#include "misc.h"
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>
#include <stdint.h>
#include <string.h>
#include <assert.h>

#include "sep.h"

enum
{
    SREC_BAD_PREFIX = 2,
    SREC_BAD_CHECKSUM,
    SREC_BAD_LENGTH,
    SREC_UNKNOWN_TYPE,
};


struct srecord
{
  uint8_t type;
  uint32_t address;

  /* The length of the data (ie, the number of bytes it encodes, NOT
     the number of bytes used to encode it). */
  uint8_t data_length;

  /* This is a pointer into a string outside of this struct.  See the
     comment on the srec function below. */
  const char *data;
};


static uint32_t
string_to_int (const char *in, int len)
{
    char str[10];
    assert (len > 0);
    assert (len <= 4);

    int i;
    for (i = 0; i < len * 2; ++i)
        str[i] = in[i];
    str[i] = '\0';

    return strtol (str, NULL, 16);
}

/* Reads a line REC of length LEN,  which should be a valid
   Motorola S-Record, and initialises SR accordingly.
   Returns zero if successful.  Non-zero otherwise.

   The "data" member of SR is a pointer into REC, and therefore
   it is the caller's responsibility to ensure that REC survives
   SR. */
static int
srec (struct srecord *sr, const char *rec, int len)
{
    if (rec[0] != 'S')
        return SREC_BAD_PREFIX;

    /* Find the last real character of the line.
       This implementation hopes to be robust against line ending
       issues etc. */
    do
    {
        char last = rec[len - 1];
        if (last >= '0' && last <= '9')
            break;

        if (last >= 'A' && last <= 'F')
            break;

        if (last >= 'a' && last <= 'f')
            break;
    }
    while (len--);
    len -= 2;

    const uint8_t checksum = string_to_int (rec + len, 1);

    uint8_t sum = 0;
    int idx = 2;
    for (idx = 2; idx < len; idx +=2)
    {
        uint8_t byte = string_to_int (rec + idx, 1);
        sum += byte;
    }

    sum += checksum;
    if (sum != 0xFF)
    {
        return SREC_BAD_CHECKSUM;
    }

    const uint8_t count = string_to_int (rec + 2, 1);

    if (count * 2 != idx - 2)
    {
        return SREC_BAD_LENGTH;
    }

    /* Length of the "address" field in octets */
    int addrlen = -1;
    switch (rec[1])
    {
    case '0':
    case '1':
    case '5':
    case '9':
        addrlen = 2;
        break;

    case '2':
    case '6':
    case '8':
        addrlen = 3;
        break;

    case '3':
    case '7':
        addrlen = 4;
        break;

    default:
        return SREC_UNKNOWN_TYPE;
    };

    sr->type   = rec[1] - '0';
    sr->address = string_to_int (rec + 4, addrlen);
    sr->data_length = count - addrlen - 1;
    sr->data = rec + 4 + addrlen * 2;

    return 0;
}


struct srecord_upload
{
  struct upload_format parent;
  FILE *fp;
  char *lineptr;
  size_t linelen;
  int line;
};

static inline uint8_t
hex_to_byte (char x)
{
  if (x >= '0' && x <= '9')
    return x - '0';

  if (x >= 'A' && x <= 'F')
    return x - 'A' + 10;

  if (x >= 'a' && x <= 'f')
    return x - 'a' + 10;

  return 0xFF;
}

/* Stores a serialization of SR into *BUFFER.  *BLEN should be the length
   of *BUFFER.

   Returns the length of the serialization on success, -1 otherwise. */
int
serialize (const struct srecord *sr, uint8_t *buffer, size_t blen)
{
  size_t required_length = 4 + 1 + sr->data_length;
  if (blen < required_length)
    {
      return -1;
    }

  int i;
  for (i = 0; i < sr->data_length * 2; i+=2)
    {
      uint8_t high = hex_to_byte (sr->data[i]);
      if (high == 0xFF)
	return -1;
      uint8_t low = hex_to_byte (sr->data[i+1]);
      if (low == 0xFF)
	return -1;
      *buffer++ = (high << 4) | low;
    }

  return required_length;
}

/* Return true iff the string S contains only whitespace */
static bool
all_white (const char *s, int maxlen)
{
  int x;
  for (x = 0; x < maxlen; ++x)
    {
      if (s[x] == 0)
        return true;
      if (s[x] != ' ' && s[x] != '\n' && s[x] != '\r')
        return false;
    }

  return true;
}

static bool
getSrecord (char **lineptr, size_t *linelen, FILE *fp, int *line, struct srecord *sr)
{
  do
    {
      (*line)++;
      int n = getline (lineptr, linelen, fp);
      if (n < 0)
        return false;

      if (all_white (*lineptr, *linelen))
        continue;

      int err = srec (sr, *lineptr, n);
      if (err == 0)
        return true;
      if (err != 0)
        {
          upad_msg (MSG_ERROR, "Error reading S-record at line %d\n", *line);
          return false;
        }
    }
  while (!feof (fp));
  return true;
}

static bool
srec_get_data_fragment (struct upload_format *uf, uint32_t *address,
			uint8_t *data, size_t bufsize, size_t *n_data)
{
  struct srecord_upload *sru = (struct srecord_upload *) uf;
  do
    {
      struct srecord sr;

      if (feof (sru->fp))
        return false;

      if (!getSrecord (&sru->lineptr, &sru->linelen, sru->fp, &sru->line, &sr))
        return false;

      assert (sr.data_length <= bufsize);

      switch (sr.type)
        {
        case 1:
        case 2:
        case 3:
          *address = sr.address;
          *n_data = sr.data_length;
          serialize (&sr, data, MAX_PAYLOAD_LENGTH - 2);

          return true;
          break;
        case 0:
          upad_msg (MSG_INFORMATION, "Header record: ");
          int i;
          for (i = 0; i < sr.data_length * 2; i += 2)
            {
              uint8_t c0 = hex_to_byte (sr.data[i]) << 4;
              c0 += hex_to_byte (sr.data[i+1]);
              putchar (c0);
            }
          putchar ('\n');
          break;
        }
    }
  while (1);

  return true;
}

static void
destroy_srec_uploader (struct upload_format *u)
{
  struct srecord_upload *sru = (struct srecord_upload *) u;
  free (sru->lineptr);
  free (u);
}

struct upload_format *
create_srec_uploader (FILE *fp)
{
  struct srecord_upload *sru = safe_realloc (NULL, sizeof (*sru));
  memset (sru, 0, sizeof (*sru));
  sru->fp = fp;
  ((struct upload_format *)sru)->get_data_fragment = srec_get_data_fragment;
  ((struct upload_format *)sru)->destroy = destroy_srec_uploader;
  return (struct upload_format *) sru;
}

