/*
    upad - A program for debugging, and uploading code to embedded devices.
    Copyright (C) 2016, 2019 John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/
#include <config.h>
#include <stdint.h>
#include <stdbool.h>
#include "flash.h"
#include "target/s12z/flash.h"

#include "bdc.h"

const struct flash_command flash_commands[] =
  {
    {Z12FL_ERASE_VERIFY_ALL,            false, 0, "ERASE_VERIFY_ALL"},
    {Z12FL_ERASE_VERIFY_BLOCK,          true,  1, "ERASE_VERIFY_BLOCK"},
    {Z12FL_ERASE_VERIFY_PFLASH,         true,  2, "ERASE_VERIFY_PFLASH"},
    {Z12FL_READ_ONCE,                   false, 5, "READ_ONCE"},
    {Z12FL_PROGRAM_PFLASH,              true,  5, "PROGRAM_PFLASH"},
    {Z12FL_PROGRAM_ONCE,                false, 5, "PROGRAM_ONCE"},
    {Z12FL_ERASE_ALL_BLOCKS,            false, 0, "ERASE_ALL_BLOCKS"},
    {Z12FL_ERASE_FLASH_BLOCK,           true,  1, "ERASE_FLASH_BLOCK"},
    {Z12FL_ERASE_PFLASH_SECTOR,         true,  1, "ERASE_PFLASH_SECTOR"},
    {Z12FL_UNSECURE_FLASH,              false, 0, "UNSECURE_FLASH"},
    {Z12FL_VERIFY_BACKDOOR_ACCESS_KEY,  false, 4, "VERIFY_BACKDOOR_ACCESS_KEY"},
    {Z12FL_SET_USER_MARGIN_LEVEL,       true,  2, "SET_USER_MARGIN_LEVEL"},
    {Z12FL_SET_FIELD_MARGIN_LEVEL,      true,  2, "SET_FIELD_MARGIN_LEVEL"},
    {Z12FL_ERASE_VERIFY_EEPROM_SECTION, true,  2, "ERASE_VERIFY_EEPROM_SECTION"},
    {Z12FL_PROGRAM_EEPROM,              true,  5, "PROGRAM_EEPROM"},
    {Z12FL_ERASE_EEPROM_SECTOR,         true,  1, "ERASE_EEPROM_SECTOR"},
    {Z12FL_PROTECTION_OVERRIDE,         true,  3, "PROTECTION_OVERRIDE"},
  };

void
flash_setup (void)
{
  /* Set the clock to (hopefully!) a suitable rate */
  bdc_write_octet ((uint32_t) &FTMRZ->FCLKDIV, 0x05);
  bdc_delay ();
}

bool
flash_command (uint8_t code, uint8_t arg_aux, int n_args,
	       const uint16_t *args, struct cache *cache)
{
  /* Clear any previous error indications */
  if (cache->fstat & 0x30)
    bdc_write_octet ((uint32_t) &FTMRZ->FSTAT, 0x30);

  /* Set the number of arguments */
  bdc_write_octet ((uint32_t) &FTMRZ->FCCOBIX, n_args);

  /* Fill the first dword of FCCOB */
  bdc_write_dword ((uint32_t) &FTMRZ->FCCOB[0],
		   code << 24 | arg_aux << 16 | args[0]);

  /* Fill the remaining dwords of FCCOB */
  for (int i = 1; i < n_args; i += 2)
    {
      bdc_fill_dword (args[i] << 16 | args[i + 1]);
    }

  /* Pull the trigger */
  bdc_write_octet ((uint32_t) &FTMRZ->FSTAT, 0x80);

  uint8_t fstat;
  do
    {
      bdc_read_byte ((uint32_t) &FTMRZ->FSTAT, &fstat);
    }
  while (!(fstat & 0x80));
  cache->fstat = fstat;

  return !(fstat & 0x30);
}

/* Local Variables:  */
/* mode: c           */
/* c-style: "gnu"    */
/* End:              */
