/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016, 2019 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <config.h>
#include <stdlib.h>

#include "cfm.h"
#include "bdccsr.h"
#include "transaction.h"

/* These errors can be cleared by writing them with 1s  */
#define BDCCSR_ERR_MASK							\
  (BDCCSR_RAMWF | BDCCSR_NORESP | BDCCSR_RDINV | BDCCSR_ILLACC | BDCCSR_ILLCMD)

#define MAX_BDC_ATTEMPTS 3


/* A reliable version function to read memory addresses from the
   target.
   DEV is the descriptor on which the device is connected.
   ADDRESS is the adddress to read.
   LENGTH is the number of data to read.
   X is the address where to store the data.  It is the caller's
   responsibility to ensure that X points to a buffer sufficient
   to hold the data.
   Returns true on success.  False otherwise.  */
bool
cfm_read_memory (int dev, tgt_addr address, int length, uint8_t *x)
{
  int tries = 0;
  /* Clear the error flags if any are set.  */
  uint16_t bdccsr = execute_read_bdccsr (dev);
  if (bdccsr & BDCCSR_ERR_MASK)
    {
      bdccsr |= BDCCSR_ERR_MASK;
      execute_write_bdccsr (dev, bdccsr);
    }

  do
    {
      execute_read_memory (dev, address, length, x);
      bdccsr = execute_read_bdccsr (dev);
      if (bdccsr & BDCCSR_OVRUN)
        execute_sync (dev);
      if (tries++ > MAX_BDC_ATTEMPTS)
        {
          fprintf (stderr, "BDC overrun after %d sync attempts\n", tries);
          abort ();
        }
    }
  while (bdccsr & BDCCSR_OVRUN);

  /* Return success if no error flags are set  */
  return  !(bdccsr & BDCCSR_ERR_MASK);
}

