/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016, 2019 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>

#include <stdint.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include "breakpoints.h"
#include "user-interface.h"
#include "gdb-interface.h"
#include "options.h"
#include "misc.h"
#include "transaction.h"
#include <signal.h>

#include "bdccsr.h"

#include "main-loop.h"

extern bool running;

/* Fork and start a gdb server.  */
static int
start_server (const struct user_options *opts)
{
  FILE *fp = fopen (opts->control_file, "w");
  if (fp == NULL)
    {
      upad_errno (MSG_ERROR, "Cannot open %s", opts->control_file);
      return -1;
    }
  pid_t pid = fork ();
  if (pid < 0)
    {
      upad_errno (MSG_ERROR, "cannot fork");
      fclose (fp);
      return -1;
    }

  if (pid == 0)
    {
      /* Child */
      fclose (fp);
      if (-1 == setsid ())
	{
	  upad_errno (MSG_ERROR, "Cannot set session id");
	  return -1;
	}
    }
  else
    {
      /* Parent */
      fprintf (fp, "pid: %d\n", pid);
      fprintf (fp, "port: %s\n", opts->gdb_port);
      fclose (fp);

      exit (0);
    }

  return 0;
}

static bool exiting = false;

static void
my_handler (int sig)
{
  exiting = true;
}

#define POLL_TIME_MICROSEC 50000

/* Check the microcontroller's BDCCSR and the current state of
   the uPad state machine, and deal with anything that has to
   be done.  */
static bool
process_csr_state (int conn, int des)
{
  uint16_t bdccsr = execute_read_bdccsr (des);
  if (0 == (bdccsr & BDCCSR_ENABLE))
    {
      /* If we land here, then the processor has been reset without
	 us commanding it to.   */
      gdb_send_broken (conn, des);
      return false;
    }

  /* If there are breakpoints, and we're in a running state, but the
     processor is not running, the presumably a breakpoint has been
     hit.  */
  if (breakpoints_active ())
    {
      if (running && (bdccsr & BDCCSR_ACTIVE))
	{
	  running = false;
	  uint8_t x = breakpoint_fired (des);
	  enum bp_type t = breakpoint_type_in_set (des, x);
	  switch (t)
	    {
	    case BP_BREAK:
	      /* Stopped on breakpoint */
	      gdb_send_interrupt (conn, des);
	      break;
	    case BP_WATCH:
	      /* Stopped on watchpoint */
	      gdb_positive_ack (NULL, conn, des);
	      break;
	    default:
	      /* Stopped for unknown reason */
	      gdb_send_stopped (conn, des);
	      break;
	    }
	}
    }
  return true;
}


int
main_loop (int des, const struct user_options *opts)
{
  struct gdb_svr *svr = NULL;
  /* Start the gdb server if requested */
  if (opts->gdb_server)
    {
      svr = create_gdb_server (opts->gdb_port);
      if (svr == NULL)
        upad_msg (MSG_ERROR, "Cannot start gdb server\n");
    }

  if (opts->start_server)
    {
      start_server (opts);
    }

  {
    struct user_interface *ui = NULL;

    fd_set listen_fds;
    FD_ZERO (&listen_fds);

    /* Set up the interactive user interface if appropriate */
    if (opts->interactive)
      {
        ui = user_interface_init (opts);
        if (ui)
          {
            ui->xxdes = des;
            FD_SET (ui->ui_des, &listen_fds);
          }
      }

    if (svr)
      FD_SET (gdb_svr_conn_fd (svr), &listen_fds);

    int conn = -1;
    bool ran_successfully = false;
    struct timeval poll_time;
    poll_time.tv_sec = 0;
    poll_time.tv_usec = POLL_TIME_MICROSEC;

    /* Cleanly exit on SIGTERM */
    struct sigaction act;
    memset (&act, 0, sizeof (act));
    act.sa_handler = my_handler;
    sigaction (SIGTERM, &act, 0);

    /* Run the main loop itself */
    while ((ui && ui->running) ||
	   (svr && gdb_svr_conn_fd (svr) >= 0 && !opts->interactive))
      {
        if (exiting)
          break;
        /* Wait for an event */
        fd_set out_fds = listen_fds;
        int r = select (FD_SETSIZE, &out_fds, NULL, NULL, &poll_time);
        if (r < 0 && errno != EINTR)
          {
            upad_errno (MSG_ERROR, "select");
            user_interface_shutdown (ui);
            break;
          }
        ran_successfully = true;
        if (r < 0)
          continue;

        /* Deal with any requests from GDB */
        if (svr)
          {
            /* Accept any connection requests */
            if (FD_ISSET (gdb_svr_conn_fd(svr), &out_fds))
              {
                conn = accept_gdb_connection (svr);
                if (conn >= 0)
                  {
                    FD_SET (conn, &listen_fds);
                    FD_CLR (gdb_svr_conn_fd(svr), &listen_fds);
                    gdb_connection_initialize (conn, des);
                    continue;
                  }
              }

            if (conn != -1)
	      {
		ran_successfully = process_csr_state (conn, des);
		if (! ran_successfully)
		  goto end;
	      }

            /* Read any commands from GDB and deal with accordingly */
            if (conn != -1 && FD_ISSET (conn, &out_fds))
              {
                char bcmd;
                int n = read (conn, &bcmd, 1);
                if  (n < 0)
                  {
                    upad_errno (MSG_ERROR, "ERROR reading from gdb client");
                  }

                if (n > 0)
                  {
                    process_gdb_message (conn, bcmd, des);
                  }
                else
                  {
                    FD_CLR (conn, &listen_fds);
                    close_gdb_connection (conn);
                    conn = -1;
                    FD_SET (gdb_svr_conn_fd (svr), &listen_fds);
                    /* Currently we perform a "once" behaviour.  So exit the loop here.  */
                    break;
                  }
              }
          }

        /*  Deal with any user requests */
        if (ui && ui->running && FD_ISSET (ui->ui_des, &out_fds))
          {
            user_interface_process (ui);
          }
      }

  end:
    /* Shut things down */
    if (conn > 0)
      close (conn);
    destroy_gdb_server (svr);

    user_interface_shutdown (ui);

    return ran_successfully ? 0 : -1;
  }
}

/* Local Variables:  */
/* mode: c           */
/* c-style: "gnu"    */
/* End:              */
