/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016, 2019 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>

#include <stdint.h>
#include <misc.h>
#include <getopt.h>
#include <string.h>
#include <stdio.h>

#include "options.h"

static void
usage (const char *progname)
{
  fprintf (stderr, "Usage: %s [-p port] [-e] [-n] [file]\n", progname);
  exit (1);
}

/*-
@menu
* @samp{--version} or @samp{-v}::
* @samp{--port} or @samp{-p}::
* @samp{--erase-only} or @samp{-e}::
* @samp{--gdb-server}::
* @samp{--no-gdb-server}::
* @samp{--gdb-port} or @samp{-g}::
* @samp{--no-reset} or @samp{-n}::
* @samp{--interactive} or @samp{-i}::
* @samp{--no-interactive} or @samp{-q}::
* @samp{--script} or @samp{-T}::
* @samp{--binary} or @samp{-b}::
* @samp{--bounce}::
* @samp{--no-bounce}::
* @samp{--start-server}::
* @samp{--stop-server}::
* @samp{--no-splash}::
* @samp{--verbose} or @samp{-V}::
@end menu
*/

static int long_flag;

#define FLAG_NO_SPLASH     0
#define FLAG_BOUNCE        1
#define FLAG_NO_BOUNCE     2
#define FLAG_GDB_SERVER    3
#define FLAG_NO_GDB_SERVER 4
#define FLAG_START_SERVER  5
#define FLAG_STOP_SERVER   6


static const struct option long_options[] =
  {
    /*-
      @node @samp{--version} or @samp{-v}
      @section @samp{--version} or @samp{-v}

      Displays the version of @upad{}.
    */
    {"version",     0,                 0,  'v' },

    /*-
      @node @samp{--port} or @samp{-p}
      @section @samp{--port} or @samp{-p}

      Specifies the serial port to use for communication between
      the local computer and the controller.
      Normally is is not necessary to use this option because it
      is automatically detected.
    */
    {"port",        required_argument, 0,  'p' },

    /*-
      @node @samp{--erase-only} or @samp{-e}
      @section @samp{--erase-only} or @samp{-e}

      This option species that the controller should perform a bulk
      erase of the target's non-volatile memory and not program
      anything.
      If you specify this option, then you may not also specify a
      filename.
    */
    {"erase-only",  0,                 0,  'e' },

    /*-
      @node @samp{--gdb-server}
      @section @samp{--gdb-server}

      Runs @upad{} as a gdb server.  This is the default.
    */
    {"gdb-server", 0,                 &long_flag,  FLAG_GDB_SERVER},

    /*-
      @node @samp{--no-gdb-server}
      @section @samp{--no-gdb-server}

      Tells @upad{} not to run as a gdb server.
    */
    {"no-gdb-server", 0,              &long_flag,  FLAG_NO_GDB_SERVER},

    /*-
      @node @samp{--gdb-port} or @samp{-g}
      @section @samp{--gdb-port} or @samp{-g}

      Sets the TCP port number on which @upad{} will listen for connections
      from a gdb client.
      If the first character of the argument is not a digit, then instead
      of a TCP port number it is the name of a socket on the local filesystem.
      If not specified the default is port 2159.
      This setting is only meaningful if a gdb server is to be run.
      @strong{For security reasons you should not specify a port number
      less than 1024, or any port on which connections from untrusted
      sources are possible.}
    */
    {"gdb-port",    required_argument,                 0,  'g' },

    /*-
      @node @samp{--no-reset} or @samp{-n}
      @section @samp{--no-reset} or @samp{-n}

      By default, @upad{} will reset the target device immediately after
      the programming operation has completed.
      This option requests that no reset should happen.
    */
    {"no-reset",    0,                 0,  'n' },

    /*-
      @node @samp{--interactive} or @samp{-i}
      @section @samp{--interactive} or @samp{-i}

      Start @upad{} with an interactive user interface.
      This is the default if neither @samp{--erase-only} nor a filename is provided.
      The user interface is described in @ref{Low Level Debugging}.

    */
    {"interactive", 0,                 0,  'i' },

    /*-
      @node @samp{--no-interactive} or @samp{-q}
      @section @samp{--no-interactive} or @samp{-q}

      Negates the effect of @samp{--interactive}.  This will be the default
      if the standard input is not a terminal.
    */
    {"no-interactive", 0,                 0,  'q' },

    /*-
      @node @samp{--script} or @samp{-T}
      @section @samp{--script} or @samp{-T}

      The @samp{--script} specifies that @upad{} should go into
      scripting mode.
      This option requires an argument which is the name of the
      script file to run.
    */
    {"script",      required_argument, 0,  'T' },

    /*-
      @node @samp{--binary} or @samp{-b}
      @section @samp{--binary} or @samp{-b}

      The file to be uploaded should be considered a raw binary file.
      Its contents should be uploaded verbatim.   This option requires
      an argument which is the address where to load the contents.
    */
    {"binary",      required_argument, 0,  'b' },

    /*-
      @node @samp{--bounce}
      @section @samp{--bounce}
      @cindex bounce

      The @samp{--bounce} option specifies an alternative bounce code
      file to use, instead of the default one.   It requires an argument
      which is the name of the file containing the bounce code.  The file
      should be a raw binary file.  This option is mutually exclusive with
      the @samp{--no-bounce} option.
    */
    {"bounce",      required_argument, &long_flag, FLAG_BOUNCE },

    /*-
      @node @samp{--no-bounce}
      @section @samp{--no-bounce}

      The @samp{--no-bounce} option specifies that programming operations
      should not be bounced (@pxref{Bouncing}) through the target device's RAM.  Using this
      option will mean that programming will be somewhat slower.  This option
      may be useful if you suspect that the bounce algorithm is faulty.
    */
    {"no-bounce",   0,                 &long_flag, FLAG_NO_BOUNCE },

    /*-
      @node @samp{--start-server}
      @section @samp{--start-server}
      @cindex start-server

      Start a server and run it in the background.    This option will
      cause a gdb server to start running and will then return control
      to user.  One argument must be provided which is the name of a
      file where @upad{} can write information identifying the running
      server.

      As a special case, if the argument is the name of an existing directory,
      then the control file will be created in that directory.  Furthermore,
      if no gdb-port has been specified, then it'll default to a local socket
      created in that directory.

      The @samp{--start-server} option implies @samp{--gdb-server}
      and @samp{--no-interactive}.
    */
    {"start-server",      required_argument, &long_flag, FLAG_START_SERVER },

    /*-
      @node @samp{--stop-server}
      @section @samp{--stop-server}
      @cindex stop-server

      The @samp{--stop-server} option stops a server which was started using
      @samp{--start-server}.  It takes a single argument which must be the
      name of the file which was specified when the server was started.
    */
    {"stop-server",      required_argument, &long_flag, FLAG_STOP_SERVER },


    /*-
      @node @samp{--no-splash}
      @section @samp{--no-splash}
      @cindex no-splash

      The @samp{--no-splash} option avoids the text on startup being displayed.
      It is not meaningful except in interactive mode.
    */
    {"no-splash",      0, &long_flag, FLAG_NO_SPLASH },

    /*-
      @node @samp{--verbose} or @samp{-V}
      @section @samp{--verbose} or @samp{-V}

      The @samp{--verbose} option increases the number of user visible messages
      which are displayed when @upad{} is running.
      It may be specified more than once.
    */
    {"verbose",   0,                 0, 'V'},

    {0,             0,                 0,   0  }
  };

struct opt_workspace
{
  bool no_bounce;
  bool interactive_on;
  bool interactive_off;
  bool gdb_server_on;
  bool gdb_server_off;
  bool start_server;
  bool stop_server;
};

static void
opt_start_server (struct opt_workspace *ow, struct user_options *opts)
{
  ow->start_server = true;
  free (opts->control_file);
  opts->control_file = strdup (optarg);
}

static void
opt_stop_server (struct opt_workspace *ow, struct user_options *opts)
{
  ow->stop_server = true;
  opts->stop_server = true;
  free (opts->control_file);
  opts->control_file = strdup (optarg);
}

static void
opt_no_bounce (struct opt_workspace *ow, struct user_options *opts)
{
  if (opts->bounce_code)
    {
      fprintf (stderr, "--bounce and --no-bounce are mutually exclusive\n");
      exit (1);
    }
  ow->no_bounce = true;
}

static void
opt_bounce (struct opt_workspace *ow, struct user_options *opts)
{
  if (ow->no_bounce)
    {
      fprintf (stderr, "--bounce and --no-bounce are mutually exclusive\n");
      exit (1);
    }
  opts->bounce_code = optarg;
}

static void
opt_no_splash (struct opt_workspace *ow, struct user_options *opts)
{
  opts->no_splash = true;
}


static void
opt_no_gdb_server (struct opt_workspace *ow, struct user_options *opts)
{
  ow->gdb_server_off = true;
}

static void
opt_gdb_server (struct opt_workspace *ow, struct user_options *opts)
{
  ow->gdb_server_on = true;
}

typedef void (*opt_despatch) (struct opt_workspace *, struct user_options *);


static const opt_despatch despatch_table[] =
{
  opt_no_splash,
  opt_bounce,
  opt_no_bounce,
  opt_gdb_server,
  opt_no_gdb_server,
  opt_start_server,
  opt_stop_server,
};

extern int verbosity;

/* Parses the command line options and populates OPTS accordingly. */
void
parse_options (int argc, char **argv, struct user_options *opts)
{
  struct opt_workspace ow;
  memset (&ow, 0, sizeof (ow));

  opts->gdb_port = NULL;
  for (;;)
    {
      int option_index = 0;
      int c = getopt_long (argc, argv, "b:g:vp:einqT:V", long_options, &option_index);

      if (c == -1)
        break;

      switch (c)
	{
	case 'p':
          opts->port = optarg;
          break;
	case 'e':
          opts->erase_only = true;
          break;
	case 'i':
          ow.interactive_on = true;
          break;
	case 'q':
          ow.interactive_off = true;
          break;
	case 'b':
	  opts->binary = true;
	  opts->load_address = strtol (optarg, NULL, 0);
          break;
	case 'g':
          free (opts->gdb_port);
          opts->gdb_port = strdup (optarg);
          break;
	case 'n':
          opts->no_reset = true;
          break;
	case 'v':
          printf ("%s - %s\n", PACKAGE_NAME, PACKAGE_VERSION);
          exit (0);
          break;
	case 'V':
	  verbosity++;
          break;
        case 'T':
          opts->script = optarg;
          break;
        case 0:
        {
            opt_despatch x = despatch_table[long_flag];
            x (&ow, opts);
        }
          break;
	default:
          usage (argv[0]);
          break;
	};
    }

  if (ow.start_server && ow.stop_server)
    {
      fprintf (stderr, "The options --start-server and --stop-server are mutually exclusive\n");
      exit (1);
    }

  if (ow.start_server && ow.gdb_server_off)
    {
      fprintf (stderr, "The options --start-server and --no-gdb-server are mutually exclusive\n");
      exit (1);
    }

  if (ow.interactive_on && ow.interactive_off)
    {
      fprintf (stderr, "The options --interactive and --no-interactive are mutually exclusive\n");
      exit (1);
    }

  if (ow.gdb_server_on && ow.gdb_server_off)
    {
      fprintf (stderr, "The options --gdb-server and --no-gdb-server are mutually exclusive\n");
      exit (1);
    }

  opts->gdb_server = true;
  if (ow.gdb_server_off)
    opts->gdb_server = false;

  if (ow.no_bounce)
    {
      opts->bounce_code = NULL;
    }
  else if (opts->bounce_code == NULL)
    {
      opts->bounce_code = TARGETDIR "/bounce.bin";
    }

  opts->file = (optind < argc) ? argv[optind] : NULL;

  if (opts->erase_only && opts->file)
    {
      fprintf (stderr,
	       "A filename should not be specified together with -e .\n");
      exit (1);
    }

  if (!opts->erase_only && !opts->file && !ow.interactive_off)
    opts->interactive = true;

  if (ow.start_server)
    {
      opts->gdb_server = true;
      opts->start_server = true;
      opts->interactive = false;
      canonicalise_control_file (opts);
    }
  if (opts->gdb_port == NULL)
    {
      opts->gdb_port = strdup ("2159");
    }
}

/* Local Variables:  */
/* mode: c           */
/* c-style: "gnu"    */
/* End:              */
