#!/usr/bin/env bash

# Copyright (C) 2007 Juan Manuel Borges Caño

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

MP="$(basename $0)"
MP_AUTHOR="[@]pkgauthor[@]"
MP_DESCRIPTION="[@]pkgdescription[@]"
MP_BUGREPORT="[@]pkgbugreport[@]"
MP_DATADIR="[@]pkgdatadir[@]"

function mp_is_project
{
	b1=$(basename $1)
	[ -d $1 ] && [[ $b1 != "project" ]] && [[ $b1 != "project-lib" ]]
}

function mp_usage
{
	printf "Usage: $MP [OPTION]...\n"
	printf "$MP_DESCRIPTION\n"
	printf "\n"
	printf "Mandatory arguments for long options are mandatory for short options too.\n"
	printf "  -l, --list\t\t\t\tlist available project skeletons\n"
	printf "  -s, --skeleton=SKELETON\t\tset project skeleton type\n"
	printf "  -t, --tarname=TARNAME\t\t\tset tar name\n"
	printf "  -p, --package=PACKAGE\t\t\tset full package name\n"
	printf "  -d, --description=DESCRIPTION\t\tset short description\n"
	printf "  -D, --dependencies=DEPENDENCIES\tset pkg-config dependencies modules\n"
	printf "  -a, --author=AUTHOR\t\t\tset full author name\n"
	printf "  -b, --bugreport=BUGREPORT\t\tset bug report adress\n"
	printf "  -h, --help\t\t\t\tshows a help message\n"
	printf "  -v, --version\t\t\t\tshows the program version\n"
	printf "\n"
	printf "Report bugs to <$MP_BUGREPORT>.\n"
	exit 0
}

function mp_version
{
	printf "[@]pkgversion[@]\n"
	exit 0
}

function mp_ls_projects
{
	for d in "${MP_DATADIR}"/*
	do
		if mp_is_project $d
			then printf "$(basename $d)\n"
		fi
	done
	exit 0
}

function mp_error
{
	echo "$MP: $@"
	echo "Try \`$MP --help' for more information."
	exit 1
}

TEMP=$(getopt -o "ls:t:p:d:D:a:b:hv" \
    -l "list,skeleton:,tarname:,package:,description:,dependencies:,author:,bugreport:,help,version" \
    -n "$MP" -- "$@")
eval set -- "$TEMP"
while [[ -n "$1" ]]
do
	case "$1" in
		"-l" | "--list") mp_ls_projects; shift;;
		"-s" | "--skeleton") SKELETON="$2"; shift 2;;
		"-t" | "--tarname") TARNAME="$2"; shift 2;;
		"-p" | "--package") PACKAGE="$2"; shift 2;;
		"-d" | "--description") DESCRIPTION="$2"; shift 2;;
		"-D" | "--dependencies") DEPENDENCIES="$2"; shift 2;;
		"-a" | "--author") AUTHOR="$2"; shift 2;;
		"-b" | "--bugreport") BUGREPORT="$2"; shift 2;;
		"-h" | "--help") mp_usage; shift;;
		"-v" | "--version") mp_version; shift;;
		--) shift; break;;
		*) printf "Internal error!\n"; exit 1;;
	esac
done

if [[ -z $SKELETON ]]
	then mp_error "you must specify a skeleton"
fi
if [[ -z $TARNAME ]]
	then mp_error "you must specify a tar name"
fi
if [[ -z $PACKAGE ]]
	then PACKAGE=$TARNAME
fi
if [[ -z $DESCRIPTION ]]
	then DESCRIPTION="a project created with mkproject."
fi
if [[ -z $AUTHOR ]]
	then AUTHOR=$(grep $(whoami) /etc/passwd | cut -d':' -f5 | cut -d',' -f1)
fi
if [[ -z $AUTHOR ]]
	then AUTHOR="unknown"
fi
if [[ -z $BUGREPORT ]]
	then BUGREPORT="unknown"
fi

source "${MP_DATADIR}/mp.sh" "$SKELETON"
source "${MP_DATADIR}/$SKELETON/mkproject.sh"
