/**
 * Ce programme montre comment il est possible de faire jouer de la musique.
 * 
 * Cette démonstration il utilise les fonctions suivantes : 
 *
 *------------------------------------------------------------------------------
 * MLV_init_sound : Cette fonction initialise la librairie MLV pour pouvoir
 *                  jouer de la musique.
 *                  Cette fonction doit être exécutée avant toutes les autes
 *                  fonctions. L'initialisation permet de dire à la librairie
 *                  quel est le format et la fréquence d'échatillonage des 
 *                  pistes sonores qui seront envoyées à la carte audio.
 *
 *                  La fréquence d'échantillonnage est la fréquence à laquelle
 *                  un échantillon à été enregistré. Cela veux dire que c'est à
 *                  cette fréquence qu'un échantillon doit être envoyé à la
 *                  carte son.
 *                  Le format correspond à la taille mémoire d'un échantillon.
 *
 *                  Lorsque vous enregistrez un morceau de musique, vous devez
 *                  gnéralement configurer ces deux informations.
 *                  Sous linux, vous pouvez utiliser le logiciel rezound qui
 *                  permet assez facilement de faire ce genre de manipulations.
 *
 * void MLV_init_sound(
 *    int        freq,     Fréquence d'échantillonnage.
 *                         Valeure commune :
 *                             11025, 22050 et 44100.
 *    Uint16     format    Format de la piste audio. 
 *                         Les valeurs possibles sont :
 *                     AUDIO_U8 (Unsigned 8-bit samples)
 *                     AUDIO_S8 (Signed 8-bit samples)
 *                     Les valeurs suivantes ne sont pas toujours supportées par
 *                     tous les périphériques audios :
 *                     AUDIO_U16 or AUDIO_U16LSB (unsigned 16-bit little-endian)
 *                     AUDIO_S16 or AUDIO_S16LSB (signed 16-bit little-endian)
 *                     AUDIO_U16MSB (unsigned 16-bit big-endian)
 *                     AUDIO_S16MSB (signed 16-bit big-endian)
 *                     AUDIO_U16SYS = 
 *                           AUDIO_U16LSB or AUDIO_U16MSB
 *                           depending on hardware CPU endianness
 *                     AUDIO_S16SYS = 
 *                          AUDIO_S16LSB or AUDIO_S16MSB
 *                          depending on hardware CPU endianness
 *
 *                     Voir la documentation de la SDL :
 *                          http://www.libsdl.org/cgi/docwiki.cgi/SDL_AudioSpec
 * );
 *
 *------------------------------------------------------------------------------
 * MLV_close_sound : Ferme proprement les différents périphériques audios.
 * 
 * void MLV_close_sound();
 *
 *------------------------------------------------------------------------------
 * MLV_load_music : Charge un ficher contenant de la musique en mémoire.
 *                  Cette fonction prend en paramètre le chemin d'accès du 
 *                  fichier contenant la musique et renvoie un pointeur vers 
 *                  l'espace  mémoire où a été chargé la musique.
 *
 *                  Le format de fichier accepté est le .wav
 * 
 * MLV_Sheet_music* MLV_load_music(
 *    const char*     file_music     Chemin d'accès vers un fichier contenant de
 *                                   la musique encodée en format .wav.
 * );
 *
 *------------------------------------------------------------------------------
 * MLV_close_music : Ferme un morceau de musique chargé en mémoire.
 * 
 * void MLV_close_music(
 *    MLV_Sheet_music*    music    Le morceau de musique à fermer
 * );
 *
 *------------------------------------------------------------------------------
 * MLV_play_music : Joue une musique qui est chargé en mémoire
 * 
 * void MLV_play_music( 
 *    MLV_Sheet_music*     music,    Le morceau de musique à jouer 
 *    float                volume    Le volume sonore
 * );
 *
 *------------------------------------------------------------------------------
 * MLV_stop_all_music : Arrête toutes les musiques.
 * 
 * void MLV_stop_all_music();
 *
 *------------------------------------------------------------------------------
 */


#include "MLV_all.h"

//
// Programme principal
//
// Attention ! 
// Pour pouvoir compiler ce programme sous windows et sous macintosh,
// il faut, pour la déclaration du main, respecter strictement la syntaxe
// suivante :
//
int main( int argc, char *argv[] ){
	//
	// Créé et affiche la fenêtre
	//
	MLV_create_window( "Nom_fenetre", "nom_icone", 640, 480 );

	//
	// Initialise l'infrastructure son de la librairie MLV.
	//
	// Le format Wav est extrèmement gourmand en place mémoire.
	// Les  musiques qui ont été choisis pour cette démonstration ont 
	// volontairement été échantillonés en 11025 Hz afin d'avoir des 
	// pettits fichiers de musique. La qualité a par contre été sacrifiée.
	// Normalement, les fichiers musicaux sont échantillonés à 44100 Hz au
	// au lieu de 11025 Hz.
	//
	MLV_init_sound( 11025, AUDIO_S16 );

	//
	// Charge en mémoire un fichier contenant un morceau de musique.
	//
	MLV_Sheet_music* music = MLV_load_music( "./fugue.wav" );

	//
	// Joue la musique chargée en mémoire.
	//
	MLV_play_music( music, 1.0 );


	//
	// Attend 20 seconde avant la fin du programme.
	//
	MLV_wait_seconds( 20 );

	//
	// Arrête toutes les musiques en cours d'exécution.
	//
	MLV_stop_all_music();

	//
	// Ferme les morceaux de musiques qui ont été ouverts.
	//
	MLV_close_music( music );

	//
	// Arrête l'infrastructure son de la librairie MLV.
	//
	MLV_close_sound();

	//
	// Ferme la fenêtre
	//
	MLV_close_window();
	return 0;
}
