/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \file MLV_evenement.h
 *
 * \author Adrien Boussicault
 * \author Marc Zipstein
 * 
 * \brief Ce fichier définit les prototypes des fonctions permettant d'utiliser les èvenements dans la librairie MLV.
 */

#ifndef __MLV_EVENEMENT_H__
#define __MLV_EVENEMENT_H__

#include "MLV_prompt.h"

/**
 * \brief Enumère les différents types d'évènement de la librairie MLV.
 */
typedef enum {
	MLV_NONE = 0, /**< Aucun évènement */
	MLV_KEY = 1, /**< Evnènement provenant du clavier */
	MLV_PROMPT = 2, /**< Evènement provenant de la validation d'un texte dans un prompt */
	MLV_MOUSE_BUTTON = 3, /**< Evenement provenant d'un clique de souris. */
	MLV_MOUSE_MOTION = 4 /**< Evènement provenant du déplacement de la souris. */
} MLV_Event;


/**
 * \brief Enumère les différents états possibles d'un bouton clavier ou souris.
 */
typedef enum {
	MLV_PRESSED, /**< Le bouton est appuyé */
	MLV_RELEASED /**< Le bouton est relaché */
} MLV_State;

 /**
 * /brief Cette fonction récupère un évènement dans la file d'attente, remplit en fonction de l'évènement récupéré une partie des paramètres de la fonction et renvoie un entier codant le type de l'évènement qui a été récupéré.
 *
 * Le type des évènements est défini par :
 *     MLV_Event
 * et peut valoir les valeurs suivantes:
 *     Valeur              Signification
 *     ------              --------------
 *     MLV_NONE            Aucun évènement récupéré
 *     MLV_KEY             Une touche du clavier a été appuyée ou
 *                         relachée.
 *     MLV_PROMPT          L'utilisateur à écrit une chaîne de
 *                         caractère et validé cette chaîne par
 *                         la touche entrée.
 *     MLV_MOUSE_BUTTON    Un des boutons de la souris a été
 *                         appuyé.
 *     MLV_MOUS_MOTION     La souris a été déplacée.
 *
 *  Voici un tableau qui donne les champs modifiés en fonction 
 *  de l'évènement qui a été récupéré.
 *
 *     Type Evenement      Paramètres modifiés
 *     --------------      -------------------
 *     MLV_NONE            AUCUN
 *     MLV_KEY             key_sym, key_mod, state
 *     MLV_PROMPT          texte, prompt
 *     MLV_MOUSE_BUTTON    mouse_x, mouse_y, mouse_button, state
 *     MLV_MOUSE_MOTION    mouse_x, mouse_y
 *
 *  L'utilisateur n'est pas obligé d'utiliser un champs si il
 *  n'en a pas l'utilité. Dans ce cas, il peut mettre la valeur
 *  NULL. Le champs est alors ignoré.
 *
 * \param key_sym Code de la touche qui a été appuyée ou relachée. La valeur de ce champs est définie par l'énumération : MLV_Keyboard.
 * \param key_mod Mode dans lequel se trouve le clavier lorsqu'une touche est appuyée ou relachée. La valeur de ce champs est définie par l'énumération : MLV_Keyboard_mod.
 * \param texte Texte récupéré par le prompt
 * \param prompt Prompt qui a récupéré le texte
 * \param mouse_x Coordonnée en X de la position de la souris
 * \param mouse_y Coordonnée en Y de la position de la souris
 * \param mouse_button Bouton de la souris qui a été appuyé ou relaché. Ce champs est défini par l'énumération MLV_Button.
 * \param state Etat de la touche concernée par l'évènement. La valeur de ce champs est défini par l'énumération: MLV_State.
 */
MLV_Event MLV_get_event( 
	int* key_sym, int* key_mod, 
	char** texte, MLV_Prompt** prompt,
	int* mouse_x, int* mouse_y, int* mouse_button,
	int* state
);

#endif
