/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "MLV_prompt.h"
#include "MLV_evenement.h"
#include "prompt.h"

#ifndef MEMORY_DEBUG
#include <SDL/SDL.h>
#include <SDL/SDL_gfxPrimitives.h>
#else
#include "memory_debug.h"
#endif

#include "sdlkeyboardtochar.h"
#include "glib.h"

#include <string.h>
#include "warning_error.h"

#include "memory_management.h"

#define SIZE_FONT 8
#define SIZE_BORD 1
#define TIME_PROMPT 1000 //Periode d'oscillation du prompt (ms)

//
// Les fonction avec le suffixe _NST veux dire : No Safe Thread.
// Cela veux dire qu'anvant d'utiliser de telle fonctions, il faut 
// s'assurer la propriété entiere des objet qu'il va modifier.
//

#include "data_structure.h"

extern DataMLV* MLV_data;

struct _MLV_Prompt {
    //Géométrie du prompt
    int sommetHautGauche[2];
    int sommetBasDroit[2];
    int width;
    int height;
    SDL_Rect rectangle;
    //Taille de la font du texte
    int sizeFont;
    //Positiononnement du texte
    int bord[2];
    int widthMessage;
    int widthAnswer;
    //taille visible du text
    int lengthVisibleAnswer;
    //Couleurs du prompt
    Uint32 borderColor;
    Uint32 textColor;
    Uint32 backgroundColor;
    //Rendu du prompt
    SDL_Surface* apparence;
    SDL_Surface* answer;
    SDL_Rect answerRectangle;
    //Message d'information
    char* informativeMessage;
    //position du prompt dans la réponse courante
    int positionPrompt;
    int lastTime;
    int promptIsVisible;  //le Prompt (qui clignote ) est dans son etat visible
    //Le prompt est actif 
    int isListenning;
    //Histroique - le premier élément de l'historique correspond à la reponse courante du prompt
    GList* history;
    GList* positionHistory;
    //Gestion concurentiel de l'acces au prompt
    SDL_sem* semaphore;
};


/*
 *               bord[0]   withMessage          withAnswer            bord[0]
 *                  <---><-------------><---------------------------><-->
 *               _   ____________________________________________________
 *    bord[1]   |   |                                                   |
 *              |_  |   _____________________________________________   |
 *              |   |   |               |                           |   |
 *   sizeFont   |   |   | I. Message    |    An|swer                |   |
 *              |_  |   |_______________|___________________________|   |
 *              |   |                                                   |
 *    bord[1]   |_  |___________________________________________________|
 *
 *
 *  lengthVisibleAnswer : nombre de caractere visible de la réponse
 *
 * */
inline void recalculateTextPositions_NTS( MLV_Prompt* prompt ){
    prompt->widthMessage = strlen( prompt->informativeMessage ) * prompt->sizeFont;
    prompt->widthAnswer = prompt->width - prompt->widthMessage - 2 * prompt->bord[0];
    prompt->bord[1] = (prompt->height -  prompt->sizeFont)/2;
    if( prompt->widthAnswer <= 0 ){
        prompt->lengthVisibleAnswer = 0;
    }else{
        prompt->lengthVisibleAnswer = prompt->widthAnswer / prompt->sizeFont;
    }
    if(prompt->answer){
//		DEBUG("Free answer");
        SDL_FreeSurface( prompt->answer );
    }
    if(  prompt->widthAnswer > 0 ){
//		DEBUG("Create answer");
        prompt->answer = SDL_CreateRGBSurface(SDL_HWSURFACE|SDL_SRCALPHA, prompt->widthAnswer, prompt->sizeFont, 32, 0, 0, 0, 0);
        prompt->answerRectangle.x = prompt->bord[0] + prompt->widthMessage;
        prompt->answerRectangle.y = prompt->bord[1];
        prompt->answerRectangle.w = prompt->widthAnswer;
        prompt->answerRectangle.h = prompt->sizeFont;
    }else{
        prompt->answer = NULL;
    }
}

void lockPrompt( MLV_Prompt* prompt ){
    if( SDL_SemWait( prompt->semaphore )  ){
        fprintf(stderr,"lockPrompt - Probleme de semaphore\n");
    }
}

void unlockPrompt( MLV_Prompt* prompt ){
    if( SDL_SemPost(  prompt->semaphore ) ){
        fprintf(stderr,"unlockPrompt - Probleme de semaphore\n");
    }
}

typedef struct _PromptInformation {
    GSList* promptList;
    MLV_Prompt* promptActivated;
    SDL_sem* semaphore;
} PromptInformation;

PromptInformation promptInformation;

void lockPromptInformation( ){
    if( SDL_SemWait( promptInformation.semaphore )  ){
        fprintf(stderr,"lockPromptInformation - Probleme de semaphore\n");
    }
}

void unlockPromptInformation( ){
    if( SDL_SemPost(  promptInformation.semaphore ) ){
        fprintf(stderr,"unlockPromptInformation - Probleme de semaphore\n");
    }
}

/*
prompt->history->data
  0                   m                          l-1
---------------------------------------------------------
|   |   |   |   | X | Y |   |   |   |   |   |   |   | \0 |
---------------------------------------------------------
                    |
                 Prompt
new

  0                   m                      l-1  l
---------------------------------------------------------
|   |   |   |   | X | c | Y |   |   |   |   |   |   | \0 |
---------------------------------------------------------
                        |
                      Prompt
 */
inline void addCaracPrompt_NTS( MLV_Prompt* prompt, char c ){
    int l = strlen(prompt->history->data);
    int m = prompt->positionPrompt ;
//	DEBUG("Add History Entry");
    char* new = MLV_MALLOC( ( l + 2 ), char );
    strncpy( new, prompt->history->data , m );
    new[prompt->positionPrompt] = c;
    strncpy( new + m + 1 , prompt->history->data + m , l - m );
    new[l + 1] = '\0';
//	DEBUG("Free History Entry");
	MLV_FREE( prompt->history->data, char );
    prompt->history->data = new;
    prompt->positionPrompt++;
}

/*
prompt->history->data
  0                   m                          l-1
---------------------------------------------------------
|   |   |   |   | X | Y | Z |   |   |   |   |   |   | \0 |
---------------------------------------------------------
                    |
                 Prompt
resultat

  0                   m                           l-1
------------------------------------------------------
|   |   |   |   | X | Z |   |   |   |   |   |   | \0 |
------------------------------------------------------
                    |
                 Prompt
 */
inline void suppressCaracPrompt_NTS( MLV_Prompt* prompt ){
    int l = strlen(prompt->history->data);
    int m = prompt->positionPrompt ;
    if( l == m ) return;
    memmove( prompt->history->data + m, prompt->history->data + m + 1, l - m );
}

/*
prompt->history->data
  0                   m                          l-1
---------------------------------------------------------
|   |   |   | X | Y | Z |   |   |   |   |   |   |   | \0 |
---------------------------------------------------------
                    |
                 Prompt
resulat

  0              m-1  m                           l-1
------------------------------------------------------
|   |   |   | X | Z |   |   |   |   |   |   |   | \0 |
------------------------------------------------------
                |
              Prompt
 */
inline void returnDeletionCaracPrompt_NTS( MLV_Prompt* prompt ){
    int l = strlen(prompt->history->data);
    int m = prompt->positionPrompt ;
    if( m != 0 ){
        memmove( prompt->history->data + m-1, prompt->history->data + m , l - m + 1 );
        prompt->positionPrompt--;
    }
}


inline void recalculateSommetBas_NTS( MLV_Prompt* prompt ){
    prompt->sommetBasDroit[0] = prompt->sommetHautGauche[0] + prompt->width;
    prompt->sommetBasDroit[1] = prompt->sommetHautGauche[1] + prompt->height;
}

inline void changePositionPrompt_NTS(MLV_Prompt* prompt, int sommetHautGaucheX, int sommetHautGaucheY){
    prompt->sommetHautGauche[0] = sommetHautGaucheX;
    prompt->sommetHautGauche[1] = sommetHautGaucheY;
    prompt->rectangle.x = sommetHautGaucheX;
    prompt->rectangle.y = sommetHautGaucheY;
    recalculateSommetBas_NTS( prompt );
    recalculateTextPositions_NTS( prompt );
}

void MLV_change_position_prompt(MLV_Prompt* prompt, int sommetHautGaucheX, int sommetHautGaucheY){
    lockPrompt( prompt);
    changePositionPrompt_NTS( prompt, sommetHautGaucheX, sommetHautGaucheY);
    unlockPrompt( prompt);
}

inline void changeSizePrompt_NTS(MLV_Prompt* prompt, int width, int height){
    if( (width < 0) || (height < 0)){
        fprintf(stderr,"Cree Prompt - les tailles ne sont pas correctes.");
        exit(1);
    }
    prompt->width = width;
    prompt->height =  height;
    prompt->rectangle.w = width;
    prompt->rectangle.h = height;
    if( prompt->apparence ){
//		DEBUG("Free apparence");
        SDL_FreeSurface(prompt->apparence);
    }
//	DEBUG("Create apparence");
    prompt->apparence = SDL_CreateRGBSurface(SDL_HWSURFACE|SDL_SRCALPHA, prompt->width, prompt->height, 32, 0, 0, 0, 0);
    recalculateSommetBas_NTS( prompt );
    recalculateTextPositions_NTS( prompt );
}

void MLV_change_size_prompt(MLV_Prompt* prompt, int width, int height){
    lockPrompt( prompt);
    changeSizePrompt_NTS( prompt, width, height);
    unlockPrompt( prompt );
}

inline void changeGeometryPrompt_NTS(MLV_Prompt* prompt,  int sommetHautGaucheX, int sommetHautGaucheY, int width, int height){
    changeSizePrompt_NTS( prompt, width, height);
    changePositionPrompt_NTS( prompt, sommetHautGaucheX, sommetHautGaucheY);
};

void MLV_change_geometry_prompt(MLV_Prompt* prompt,  int sommetHautGaucheX, int sommetHautGaucheY, int width, int height){
    lockPrompt( prompt);
    changeGeometryPrompt_NTS( prompt, sommetHautGaucheX, sommetHautGaucheY, width, height);
    unlockPrompt( prompt );
};

inline void changeColorsPrompt_NTS( MLV_Prompt* prompt,  Uint32 borderColor, Uint32 textColor, Uint32 backgroundColor ){
    prompt->borderColor = borderColor;
    prompt->textColor = textColor;
    prompt->backgroundColor = backgroundColor;
};

inline void MLV_change_colors_prompt( MLV_Prompt* prompt,  Uint32 borderColor, Uint32 textColor, Uint32 backgroundColor ){
    lockPrompt( prompt );
    changeColorsPrompt_NTS( prompt, borderColor, textColor, backgroundColor );
    unlockPrompt( prompt );
}


inline int isInPrompt_NTS( MLV_Prompt* prompt, int x, int y ){
    int result = 0;
    if( (x >= prompt->sommetHautGauche[0]) &&
        (x <= prompt->sommetBasDroit[0]) &&
        (y >= prompt->sommetHautGauche[1]) &&
        (y <= prompt->sommetBasDroit[1]) 
    ) result = 1;
    return result;
}

inline MLV_Prompt* isInAPrompt_NTS( int x, int y ){
    MLV_Prompt* result = NULL;
    GSList* list = promptInformation.promptList;
    while( list ){
        MLV_Prompt* prompt = (MLV_Prompt*) list->data;
        if( isInPrompt_NTS( prompt,  x, y ) ){
            result = prompt;
            break;
        }
        list = list->next;
    }
    return result;
}

inline void constructPrompt_NTS( MLV_Prompt* prompt ){
    boxColor(prompt->apparence, 0, 0, prompt->width-1, prompt->height-1, prompt->backgroundColor);

    stringColor( prompt->apparence, prompt->bord[0] , prompt->bord[1], prompt->informativeMessage, prompt->textColor );

    if( prompt->widthAnswer > 0 ){
        boxColor(prompt->answer, 0, 0, prompt->widthAnswer-1, prompt->sizeFont -1, prompt->backgroundColor);
        char * message = (char*) prompt->history->data ;
        int decalage = 0;
        if( (prompt->sizeFont * prompt->positionPrompt) > (prompt->widthAnswer) ){
            decalage = prompt->widthAnswer - prompt->sizeFont * prompt->positionPrompt -1;
        }
        stringColor( prompt->answer, decalage , 0, message, prompt->textColor );
        SDL_BlitSurface( prompt->answer, NULL, prompt->apparence, &(prompt->answerRectangle));
    }

    int time = SDL_GetTicks();
    if( ( time - prompt->lastTime > TIME_PROMPT ) ){
        prompt->lastTime = time;
        prompt->promptIsVisible = ! prompt->promptIsVisible;
    };

    if( (! prompt->isListenning) || prompt->promptIsVisible ){
        int abscisse;
        if( (prompt->sizeFont * prompt->positionPrompt) > (prompt->widthAnswer) ){
            abscisse = prompt->bord[0] + prompt->widthMessage + prompt->widthAnswer -2;
        }else{
            abscisse = prompt->bord[0] + prompt->widthMessage + prompt->positionPrompt * prompt->sizeFont;
        }
        lineColor(prompt->apparence, abscisse  , prompt->bord[1] , abscisse  , prompt->bord[1] + prompt->sizeFont -1 , prompt->textColor );
    }

    rectangleColor(prompt->apparence, 0, 0, prompt->width-1, prompt->height-1, prompt->borderColor);
}

inline void promptRegister_NTS(MLV_Prompt* prompt){
    promptInformation.promptList = g_slist_prepend( promptInformation.promptList , prompt );
}

inline void promptUnregister_NTS(MLV_Prompt* prompt){
    if( promptInformation.promptActivated == prompt ) promptInformation.promptActivated = NULL;
    promptInformation.promptList = g_slist_remove( promptInformation.promptList, prompt );
}

MLV_Prompt* MLV_create_prompt( 
	int sommetHautGaucheX, int sommetHautGaucheY,
	int width, int height,
	Uint32 borderColor, Uint32 textColor,
	Uint32 backgroundColor,
	const char* informativeMessage
){
//	DEBUG("Add prompt");
    MLV_Prompt* prompt = MLV_MALLOC( 1, MLV_Prompt );

//	DEBUG("Create semaphore");
    prompt->semaphore = SDL_CreateSemaphore(1);

    prompt->sizeFont = SIZE_FONT;
    prompt->bord[0] = SIZE_BORD;
//	DEBUG("Add Informative message");
    prompt->informativeMessage = MLV_MALLOC( (strlen(informativeMessage)+1), char );
    strcpy( prompt->informativeMessage, informativeMessage );

    prompt->apparence = NULL;
    prompt->answer = NULL;
    changeGeometryPrompt_NTS(prompt, sommetHautGaucheX, sommetHautGaucheY, width, height);

    MLV_change_colors_prompt(prompt, borderColor, textColor, backgroundColor );

//	DEBUG("Add History Entry");
    prompt->history = g_list_prepend(NULL, MLV_MALLOC( 1, char ) );
    ((char*) prompt->history->data)[0] = '\0';
    prompt->positionHistory = prompt->history;
    prompt->positionPrompt = 0;

    prompt->lastTime = SDL_GetTicks();
    prompt->promptIsVisible = 1;
    prompt->isListenning = 0;

    lockPromptInformation();
    promptRegister_NTS( prompt );
    unlockPromptInformation();
    return prompt;
}

inline void gfree_NTS( void* data,void* useless ){
//	DEBUG("Free Entry Hystory");
	MLV_FREE( data, char );
}

inline void suppressHistory_NTS( MLV_Prompt* prompt ){
    g_list_foreach( prompt->history, gfree_NTS, NULL );
//	DEBUG("Free history");
    g_list_free( prompt->history );
}

void MLV_suppress_history( MLV_Prompt* prompt ){
    lockPrompt( prompt );
    suppressHistory_NTS( prompt );
    unlockPrompt( prompt );
}

void MLV_close_prompt( MLV_Prompt* prompt ){
    lockPromptInformation( );
    lockPrompt( prompt );
    promptUnregister_NTS( prompt );
//	DEBUG("Free apparence");
    SDL_FreeSurface(prompt->apparence);
//	DEBUG("Free answer");
    SDL_FreeSurface( prompt->answer );
//	DEBUG("Free Informative message");
	MLV_FREE( prompt->informativeMessage, char );
//    MLV_FREE( prompt->history->data, char );
    suppressHistory_NTS( prompt );
//	DEBUG("Free semaphore");
    SDL_DestroySemaphore( prompt->semaphore );
    unlockPromptInformation( );
//	DEBUG("Free prompt");
	MLV_FREE( prompt, MLV_Prompt );
}

void MLV_change_informative_message( MLV_Prompt* prompt, const char* message ){
    lockPrompt(prompt);
    if( prompt->informativeMessage ){
//		DEBUG("Free Informative message");
		MLV_FREE( prompt->informativeMessage, char );
    }
//	DEBUG("Add Informative message");
    prompt->informativeMessage = MLV_MALLOC( (strlen(message)+1), char );
    strcpy( prompt->informativeMessage, message );
    recalculateTextPositions_NTS( prompt );
    unlockPrompt(prompt);
}


void MLV_draw_prompt(MLV_Prompt* prompt){
    lockPrompt(prompt);
    constructPrompt_NTS( prompt );
    SDL_BlitSurface( prompt->apparence, NULL, MLV_data->screen, &(prompt->rectangle));
    unlockPrompt(prompt);
}

void MLV_draw_all_prompt(){
    lockPromptInformation( );
    g_slist_foreach( promptInformation.promptList, (void (*)(void*,void*)) MLV_draw_prompt , MLV_data->screen);
    unlockPromptInformation( );
}

void init_prompt_mechanism(){
    SDL_EnableUNICODE(SDL_ENABLE);
    promptInformation.semaphore = SDL_CreateSemaphore(1);
    lockPromptInformation();
    promptInformation.promptList = NULL;
    promptInformation.promptActivated=NULL;
    unlockPromptInformation();
}

void quit_prompt_mechanism(){
	SDL_DestroySemaphore(promptInformation.semaphore);
}

inline void replaceEntreeByHistory_NTS( MLV_Prompt* prompt ){
    char* dst = prompt->history->data;
    if( prompt->positionHistory != prompt->history ){
        char* src = (char*) prompt->positionHistory->data;
        if( strlen(dst) >= strlen(src) ){
            strcpy( dst, src );
        }else{
//			DEBUG("Add Hystory Entry");
            prompt->history->data = MLV_MALLOC( (strlen(src) + 1), char );
            strcpy( prompt->history->data, src );
//			DEBUG("Suppress Hystory Entry");
			MLV_FREE( dst, char );
        }
    }else{
        if(dst){
            dst[0] = '\0';
        }
    }
    prompt->positionPrompt = strlen(prompt->history->data);
}

inline void goDownInHistory_NTS( MLV_Prompt* prompt ){
    if( prompt->positionHistory->prev ){
        prompt->positionHistory = prompt->positionHistory->prev;
    }else{
        return;
    }
    replaceEntreeByHistory_NTS( prompt );
}

inline void goUpInHistory_NTS( MLV_Prompt* prompt ){
    if( prompt->positionHistory->next ){
        prompt->positionHistory = prompt->positionHistory->next;
    }else{
        return;
    }
    replaceEntreeByHistory_NTS( prompt );
}

inline void makeThePromptVisible_NTS( MLV_Prompt* prompt ){
    prompt->lastTime = SDL_GetTicks();
    prompt->promptIsVisible = 1;
}

inline void promptMoveLeftAnswer_NTS( MLV_Prompt* prompt ){
    if( prompt->positionPrompt > 0 ){
        prompt->positionPrompt--;
    }
}

inline void promptMoveRightAnswer_NTS( MLV_Prompt* prompt ){
    if( prompt->positionPrompt < strlen( prompt->history->data ) ){
        prompt->positionPrompt++;
    }
}

inline void validatePrompt_NTS( MLV_Prompt* prompt ){
    SDL_Event event;
    event.type = SDL_USEREVENT;
	event.user.code = MLV_PROMPT;
    event.user.data1 = prompt;
    event.user.data2 = prompt->history->data;
    SDL_PushEvent(&event);
    prompt->history = g_list_prepend( prompt->history, prompt->history->data );
    prompt->positionHistory = prompt->history;
//	DEBUG("Add Hystory Entry");
    prompt->history->data = MLV_MALLOC( 1, char );
    ((char*)prompt->history->data)[0] = '\0';
    prompt->positionPrompt = 0;
}


int prompt_events_filter(const SDL_Event *event) {
    int result = 1;
    lockPromptInformation();
    switch(event->type){
        case SDL_KEYUP :
            {     //Une touche a ete appuyee
                if( promptInformation.promptActivated ) result = 0;
            }
            break;
        case SDL_KEYDOWN : 
            {   //Une touche a ete appuyee
                // Si la touche echap a ete appuye, on laisse passe l'evenement
                if ( event->key.keysym.sym == SDLK_ESCAPE ) break;
                if( promptInformation.promptActivated ){
                    lockPrompt( promptInformation.promptActivated );
                    makeThePromptVisible_NTS( promptInformation.promptActivated);
                    //Si la touche est une fleche gauche ou droite on deplace le curseu
                    if( event->key.keysym.sym == SDLK_LEFT ){
                        promptMoveLeftAnswer_NTS( promptInformation.promptActivated );
                    }else if( event->key.keysym.sym == SDLK_RIGHT ){
                        promptMoveRightAnswer_NTS( promptInformation.promptActivated );
                    }else if( event->key.keysym.sym == SDLK_UP ){//Si la touche est une fleche bas ou haut on se  deplace dans l'historique
                        goUpInHistory_NTS( promptInformation.promptActivated );
                    }else if( event->key.keysym.sym == SDLK_DOWN ){
                        goDownInHistory_NTS( promptInformation.promptActivated );
                    }else if( event->key.keysym.sym == SDLK_DELETE ){//Si la touche est une suppession on supprime une lettre
                        suppressCaracPrompt_NTS( promptInformation.promptActivated );
                    }else if( event->key.keysym.sym == SDLK_BACKSPACE ){
                        returnDeletionCaracPrompt_NTS( promptInformation.promptActivated );
                    }else if( (event->key.keysym.sym == SDLK_RETURN) || (event->key.keysym.sym == SDLK_KP_ENTER) ){
                        validatePrompt_NTS(promptInformation.promptActivated);
                    }else if( sldKeyIsACharacter( event->key.keysym.sym )){
                    //Si la lettre n'est pas \n On ajoute la lettre au mot courant
                    //Sinon on vide le mot et on cree un evenement du mot qui a ete attrappee par le prompt
                        addCaracPrompt_NTS( promptInformation.promptActivated, event->key.keysym.unicode );
                    }
                    unlockPrompt( promptInformation.promptActivated );
                    result = 0;
                }
            }
            break;
        case SDL_MOUSEBUTTONDOWN :
            {     //Si on est dans une zone de prompt on active l'enregistrement des données.
                MLV_Prompt* tmp;
                if( ( tmp = isInAPrompt_NTS( event->button.x, event->button.y ) ) ){
                    tmp->isListenning = 1;
                    result=0;
                }
                if( promptInformation.promptActivated ){
                    promptInformation.promptActivated->isListenning = 0;
                }
                promptInformation.promptActivated = tmp;
            }
            break;
        default :;
    }
    unlockPromptInformation();
    return result;
}

