/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \file MLV_figures.h
 *
 * \author Adrien Boussicault
 * \author Marc Zipstein
 * \author Nicolas Dennewald
 *
 * \brief Ce fichier définit tout les prototypes de fonction permettant de dessiner des figures avec la librairies MLV.
 */

#ifndef __MLV_FIGURES_H__
#define __MLV_FIGURES_H__

#include "MLV_color.h"

/**
 * \brief Dessine un cercle à une position et un rayon spécifiés en paramètres.
 *
 * \param x Coordonnée en X du centre
 * \param y Coordonnée en Y du centre 
 * \param radius Rayon du cercle
 * \param color Couleur du tracé 
 */
void MLV_draw_circle(int x, int y, int radius, MLV_Color color);

/**
 * \brief Dessine un disque plein dont le centre et le rayon sont passés en parametres.
 *
 * \param x Coordonnée en X du centre
 * \param y Coordonnée en Y du centre 
 * \param radius Rayon du cercle
 * \param color Couleur du tracé 
 */
void MLV_draw_filled_circle(int x, int y, int radius, MLV_Color color);

/**
 * \brief Dessine une ellipse.
 *
 * \param x Coordonnée en X du centre de l'ellipse
 * \param y Coordonnée en Y du centre de l'ellipse
 * \param radius_x Rayon en X de l'ellipse
 * \param radius_y Rayon en Y de l'ellipse
 * \param color Couleur du tracé 
 */
void MLV_draw_ellipse(int x, int y, int radius_x, int radius_y, MLV_Color color);

/**
 * \brief Dessine une ellipse pleine.
 *
 * \param x Coordonnée en X du centre de l'ellipse
 * \param y Coordonnée en Y du centre de l'ellipse
 * \param radius_x Rayon en X de l'ellipse
 * \param radius_y Rayon en Y de l'ellipse
 * \param color Couleur du tracé 
 */
void MLV_draw_filled_ellipse(int x, int y, int radius_x, int radius_y, MLV_Color color);

/**
 * \brief Dessine un polygone à partir d'une liste de sommets.
 *
 * \param vx La liste des coordonnées en X des différents sommets du polygone.
 * \param vy La liste des coordonnées en Y des différents sommets du polygone.
 * \param npoints Le nombre de sommets du polygone.
 * \param color La couleur du tracé.
 */
void MLV_draw_polygon( const int* vx, const int* vy, int npoints, MLV_Color color);

/**
 * \brief Dessine un polygone plein à partir d'une liste de sommets.
 *
 * \param vx La liste des coordonnées en X des différents sommets du polygone.
 * \param vy La liste des coordonnées en Y des différents sommets du polygone.
 * \param npoints Le nombre de sommets du polygone.
 * \param color La couleur du tracé.
 */
void MLV_draw_filled_polygon( const int* vx, const int* vy, int npoints, MLV_Color color);

/**
 * \brief Dessine une courbe de Bézier à partir d'une liste de sommets.
 *
 * \param vx La liste des coordonnées en X des différents sommets de la courbe.
 * \param vy La liste des coordonnées en Y des différents sommets de la courbe.
 * \param npoints Le nombre de sommets disponibles pour dessiner la courbe de Bézier.
 * \param color La couleur du tracé.
 */
void MLV_draw_bezier_curve( const int* vx, const int* vy, int npoints, MLV_Color color);

/**
 * \brief Dessine un polygone plein à partir d'une liste de sommets.
 *
 * \param vx La liste des coordonnées en X des différents sommets du polygone.
 * \param vy La liste des coordonnées en Y des différents sommets du polygone.
 * \param npoints Le nombre de sommets du polygone.
 * \param color La couleur du tracé.
 */
void MLV_draw_filled_polygon( const int* vx, const int* vy, int npoints, MLV_Color color);

/**
 * \brief Dessine un rectangle dont la taille, la couleur et la position du sommet Nord-Ouest sont données en paramètres.
 *
 * \param x Coordonnée en X du sommet Nord-Ouest du rectangle
 * \param y Coordonnée en Y du sommet Nord-Ouest du rectangle
 * \param width Largeur du rectangle
 * \param height Hauteur du rectangle
 * \param color Couleur du rectangle
 */
void MLV_draw_rectangle(int x, int y, int width, int height, MLV_Color color);

/**
 * \brief Dessine un rectangle dont la taille, la couleur et la position du sommet Nord-Ouest sont données en paramètres.
 *
 * \param x Coordonnée en X du sommet Nord-Ouest du rectangle
 * \param y Coordonnée en Y du sommet Nord-Ouest du rectangle
 * \param width Largeur du rectangle
 * \param height Hauteur du rectangle
 * \param color Couleur du rectangle
 */
void MLV_draw_rectangle(int x, int y, int width, int height, MLV_Color color);

/**
 * \brief Dessine un rectangle plein dont la taille, la couleur et la position du sommet Nord-Ouest sont données en paramètres.
 *
 * \param x Coordonnée en X du sommet Nord-Ouest du rectangle
 * \param y Coordonnée en Y du sommet Nord-Ouest du rectangle
 * \param width Largeur du rectangle
 * \param height Hauteur du rectangle
 * \param color Couleur du rectangle
 */
void MLV_draw_filled_rectangle(int x, int y, int width, int height, MLV_Color color);

/**
 * \brief Dessine une ligne.
 *
 * \param x1 Coordonnée en X de la première extrémité de la ligne
 * \param y1 Coordonnée en Y de la première extrémité de la ligne
 * \param x2 Coordonnée en X de la deuxième extrémité de la ligne
 * \param y2 Coordonnée en Y de la deuxième extrémité de la ligne
 * \param color Couleur du tracé
 */
void MLV_draw_line(int x1, int y1, int x2, int y2, MLV_Color color);

/**
 * \brief Dessine un pixel dont les coordonnées sont passées en paramètres.
 *
 * \param x Coordonnée en X du pixel
 * \param y Coordonnée en Y du pixel
 * \param color Couleur du tracé
 */
void MLV_draw_pixel(int x, int y, MLV_Color color);

/**
 * \brief Dessine un point dont les coordonnées sont passées en paramètres. Cette fonction est identique à MLV_draw_pixel.
 *
 * \param x Coordonnée en X du point
 * \param y Coordonnée en Y du point
 * \param color Couleur du tracé
 */
void MLV_draw_point(int x, int y, MLV_Color color);

/**
 * \brief Enregistre l'image de l'écran dans un presse papier interne à la librairie MLV..
 *
 */
void MLV_save_screen();

/**
 * \brief Charge l'image présent dans le presse papier interne de la librairie MLV.
 *
 */
void MLV_load_screen();

#endif
