/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "MLV_window.h"
#include "MLV_color.h"
#include "MLV_figures.h"

#include "text.h"

#include "prompt.h"
#include "MLV_color.h"
#include "MLV_sound.h"
#include <stdio.h>
#include <stdlib.h>

#ifndef MEMORY_DEBUG
#include <SDL/SDL.h>
#include <SDL/SDL_gfxPrimitives.h>
#else
#include "memory_debug.h"
#endif

#include "warning_error.h"

#include "data_structure.h"

#include "memory_management.h"

#define SIZE_DEFAULT_FONT 12

DataMLV *MLV_data = NULL;
void (*MLV_call_back_function_for_exit)(void*) = NULL;
void *MLV_call_back_data = NULL;

int events_filter(const SDL_Event *event){
	if( event->type == SDL_QUIT ){
		if( MLV_call_back_function_for_exit ){
			(*(MLV_call_back_function_for_exit))(MLV_call_back_data);
			return 1;
		}else{
			exit(0);
		}
	}
	return prompt_events_filter(event);
}

void MLV_create_window(const char* windows_name, const char* icone_name, int width, int height){
	if( MLV_data ){
		ERROR("The MLV library has yet been initialised.");
	}
	/**************************************************************************/
	/* Initialisation de la strcutre de MLV                                   */
	/**************************************************************************/
	MLV_data = MLV_MALLOC( 1, DataMLV );
	MLV_data->screen = NULL;

	/**************************************************************************/
	/* Initialisation de la libriaire SDL pour l'uilisation de la video et du */
	/*   temps                                                                */
	/**************************************************************************/
	if(
		SDL_Init(
			SDL_INIT_VIDEO | SDL_INIT_TIMER | SDL_INIT_AUDIO | 
			SDL_INIT_EVENTTHREAD  // Cette option est importante !
			                      // Elle permet de faire en sorte 
			      // que le filtrage des évenements soit fait dés 
			      // réception d'un evenement de la part de l'OS
			      // Si cette option n'est pas activée, le filtre 
			      // d'evenement est applique seulement après un 
			      // appel à poll_event. Dans ce cas la gestion
			      // de l'arret automatique du programme après 
			      // utilisation de la croix des fenetre ne 
			      // fonctione plus correctement. En effet, si 
			      // l'utilisateur n'utilise pas de fonction mettant
			      // en jeu des évènement, le filtre n'est jamais
			      // appelé et le mécanisme d'arret automatique 
			      // non plus.
		) <0 
	){
		printf("Unable to init SDL: %s\n", SDL_GetError());
		exit(1);
	}



	/**************************************************************************/
	/* Creation d'un fenetre graphique                                        */
	/**************************************************************************/
	MLV_data->width = width;
	MLV_data->height = height;
	MLV_data->rectangle.x = 0;
	MLV_data->rectangle.y = 0;
	MLV_data->rectangle.w = width;
	MLV_data->rectangle.h = height;
	MLV_data->screen = SDL_SetVideoMode(width, height, 32, SDL_HWSURFACE|SDL_DOUBLEBUF);
	MLV_data->save_screen = SDL_CreateRGBSurface(SDL_HWSURFACE|SDL_SRCALPHA, width, height, 32, 0, 0, 0, 0);
	if ( MLV_data->screen == NULL ){
		fprintf(stderr,"Unable to set %dx%d video: %s\n", width, height, SDL_GetError());
		exit(1);
	}

	/**************************************************************************/
	/* Configuration des noms de la fenetre                                   */
	/**************************************************************************/
	SDL_WM_SetCaption(windows_name, icone_name);

	/**************************************************************************/
	/* Configuration du mechanisme permettant d'utiliser les prompts           */
	/**************************************************************************/
	init_prompt_mechanism();

	/**************************************************************************/
	/* Configuration des fonts                                                */
	/**************************************************************************/
	init_font();
	
	/**************************************************************************/
	/* Création d'une font par default                                        */
	/**************************************************************************/
	MLV_data->defaultFont = MLV_load_font( DATA_PATH "/font/DejaVuSerif-Bold.ttf" , SIZE_DEFAULT_FONT );

	SDL_SetEventFilter(events_filter);
	
	/**************************************************************************/
	/* Initialisation de la couleur du fond de la fenetre en noir.            */
	/**************************************************************************/
	MLV_draw_filled_rectangle(0,0,width,height,MLV_COLOR_BLACK);
	MLV_actualise_window();
}

void MLV_close_window(){
	if( ! MLV_data ){
		ERROR("No windows has been created.");
	}
	SDL_FreeSurface(MLV_data->screen);
	SDL_FreeSurface(MLV_data->save_screen);
	MLV_close_font( MLV_data->defaultFont );
	quit_font();
	quit_prompt_mechanism();
	MLV_FREE( MLV_data, DataMLV );
	SDL_Quit();
}

void MLV_actualise_window(){
	if( (! MLV_data ) || (! MLV_data->screen) ){
		ERROR("A windows can't be displayed whitout being created.");
	}
	SDL_Flip(MLV_data->screen);
}

void MLV_execute_at_exit( void (*function)(void*), void* data ){
	if( MLV_data ){
		ERROR("The mlv windows is yet initialized. The function execute_at_exit() must be executed before the create_windows() function.");
	}
	if( MLV_call_back_function_for_exit ){
		ERROR("The call back function for exiting program is yet initialized.");
	}
	MLV_call_back_function_for_exit = function;
	MLV_call_back_data = data;
}


