/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "MLV_image.h"
#include "data_structure.h"

#ifndef MEMORY_DEBUG
#include <SDL/SDL.h>
#include <SDL/SDL_rotozoom.h>
#else
#include "memory_debug.h"
#endif

#include "warning_error.h"

#include "memory_management.h"

struct _MLV_Image {
    SDL_Surface * surface;
};

extern DataMLV* MLV_data; 

MLV_Image* MLV_load_image_bmp( const char* file_image ){
//    image->surface = IMG_Load( path );
	MLV_Image* image = (MLV_Image*) malloc( sizeof( MLV_Image ) );
	SDL_Surface* surface = SDL_LoadBMP( file_image );
	image->surface = SDL_CreateRGBSurface( SDL_HWSURFACE|SDL_SRCALPHA, surface->w, surface->h, 32, 0, 0, 0, SDL_ALPHA_TRANSPARENT);
	SDL_Rect src,dst;
	dst.x = 0; dst.y = 0;
	src.x = 0; src.y = 0;
	src.w = surface->w; src.h = surface->h;
	SDL_BlitSurface( 
		surface, &src,
		image->surface, &dst
	);
	SDL_SetAlpha(image->surface, SDL_SRCALPHA | SDL_RLEACCEL , SDL_ALPHA_OPAQUE);
	SDL_FreeSurface( surface );
	return image;
}

void MLV_resize_image( MLV_Image* image, int width, int height ){
	double scalar_x = ((double) width) / ((double) image->surface->w);
	double scalar_y = ((double) height) / ((double) image->surface->h);
	MLV_scale_xy_image( image,  scalar_x, scalar_y );
}

void MLV_proportionnal_resize_image( MLV_Image* image, int width, int height ){
	double scalar_x = -1.0;
	double scalar_y = -1.0;
	if( width<=0 && height<=0 ) return;
	if( width > 0){
		scalar_x = ((double) width) / ((double) image->surface->w);
	}
	if( height > 0){
		scalar_y = ((double) height) / ((double) image->surface->h);
	}
	if( scalar_x < 0 ){
		scalar_x = scalar_y;
	}
	if( scalar_y < 0 ){
		scalar_y = scalar_x;
	}
	MLV_scale_image( image, ( scalar_x < scalar_y )? scalar_x : scalar_y );
}

void MLV_get_size_image( MLV_Image* image, int* width, int* height ){
	if( width ){
		*width = image->surface->w;
	}
	if( height ){
		*height = image->surface->h;
	}
}

void MLV_scale_xy_image( MLV_Image* image, double scalar_x, double scalar_y ){
	MLV_rotate_scale_xy_image( image, 0.0, scalar_x, scalar_y );
}

void MLV_rotate_scale_xy_image(
	MLV_Image* image, double rotation, double scalar_x, double scalar_y 
){
	SDL_Surface* dst;
	dst = rotozoomSurfaceXY( image->surface, rotation, scalar_x, scalar_y, 0);
	SDL_FreeSurface( image->surface );
	image->surface = dst; 
}

void MLV_scale_image( MLV_Image* image, double scalar ){
	MLV_rotate_scale_image( image, 0.0, scalar );
}

void MLV_rotate_image( MLV_Image* image, double rotation ){
	MLV_rotate_scale_image( image, rotation, 1.0 );
}

void MLV_rotate_scale_image( MLV_Image* image, double rotation, double scalar ){
	SDL_Surface* dst;
	dst = rotozoomSurface( image->surface, rotation, scalar, 0);
	SDL_FreeSurface( image->surface );
	image->surface = dst; 
}

void MLV_free_image( MLV_Image* image ){
    SDL_FreeSurface( image->surface );
}

MLV_Image* MLV_copy_image( MLV_Image* image ){
	MLV_Image* result = (MLV_Image*) malloc( sizeof( MLV_Image ) );
	result->surface = SDL_CreateRGBSurface(
		SDL_HWSURFACE|SDL_SRCALPHA,
		image->surface->w, image->surface->h,
		32, 0, 0, 0,
		SDL_ALPHA_TRANSPARENT
	);
	SDL_Rect src,dst;
	dst.x = 0; dst.y = 0;
	src.x = 0; src.y = 0;
	src.w = image->surface->w; src.h = image->surface->h;
	SDL_BlitSurface( 
		image->surface, &src,
		result->surface, &dst
	);
	return result;
}

void MLV_draw_image(  MLV_Image* image, int x, int y ){
	SDL_Rect rectangle;
	rectangle.x = x;
	rectangle.y = y;
	rectangle.h = image->surface->h;
	rectangle.w = image->surface->w;
	SDL_BlitSurface( image->surface, NULL, MLV_data->screen, &rectangle);
}

void MLV_draw_partial_image(
	MLV_Image* image, int x_source, int y_source, 
	int width_source, int height_source, 
	int x, int y
){
	SDL_Rect rectangle_source;
	rectangle_source.x = x_source;
	rectangle_source.y = y_source;
	rectangle_source.h = height_source;
	rectangle_source.w = width_source;

	SDL_Rect rectangle_dest;
	rectangle_dest.x = x;
	rectangle_dest.y = y;
	rectangle_dest.h = height_source;
	rectangle_dest.w = width_source;

	SDL_BlitSurface( image->surface, &rectangle_source, MLV_data->screen, &rectangle_dest);
}


void MLV_draw_scaled_rotated_image(  MLV_Image* image, int centre_x, int centre_y, double rotation, double scalar ){
	SDL_Surface* dst;
	dst = rotozoomSurface( image->surface, rotation, scalar, 0);
	SDL_Rect rectangle_dest;
	rectangle_dest.x = centre_x;
	rectangle_dest.y = centre_y;
	rectangle_dest.h = dst->h;
	rectangle_dest.w = dst->w;
	SDL_BlitSurface( dst, NULL, MLV_data->screen, &rectangle_dest);
	SDL_FreeSurface( dst );
}

void MLV_draw_rotated_image(  MLV_Image* image, int centre_x, int centre_y, double roation ){
	MLV_draw_scaled_rotated_image( image, centre_x, centre_y, roation, 1.0 );
}

void MLV_draw_scaled_image(  MLV_Image* image, int centre_x, int centre_y, double scalar ){
	MLV_draw_scaled_rotated_image( image, centre_x, centre_y, 0.0, scalar );
}
