/**
 * Ce programme explique comment, si elle n'est pas utilisée pour réaliser des 
 * effets de transparence, il est possible d'utiliser la composante alpha d'une 
 * image pour réaliser des zones de selection.
 *
 * Attention, si vous ouvrez les images utilisées par cette démonstration avec
 * un éditeur d'image, vous obtiendrez une image vide.
 * En effet, nous avons détourné la composante alpha pour un autre usage. 
 * Dans notre cas elle varie entre 0 et 4. L'image disparait donc par rapport au
 * fond. Si vous voulez voir l'image dans votre éditeur d'image, vous devez 
 * suprimer la composante alpha.
 *
 * Dans la pratique, vous n'êtes pas obligé d'utiliser la composante alpha pour
 * réaliser une zone de selection. Vous pouvez utiliser plmusieurs images, une 
 * pour l'affichage et une pour définir les zones de selection.
 * Ainsi, vous conservez la composante alpha pour réaliser des effets de 
 * transparence évolués.
 *
 *
 * Pour cela, le programme utilise les fonctions suivantes:
 *
 *------------------------------------------------------------------------------
 * MLV_get_pixel : Renvoie la couleur du pixel présent à une position donnée 
 *                 d'une image donnée.
 *
 *                 Les paramètres red, blue, green et alpha peuvent être égale 
 *                 à NULL. Dans ce cas, ils sont ignorés.
 *
 * void MLV_get_pixel(
 *     MLV_Image*     image,     Image à inspecter.
 *     int            x,         Coordonnée en X de la position du pixel à
 *                               inspecter
 *     int            y,         Coordonnée en Y de la position du pixel à
 *                               inspecter
 *     int*           red,       Composante rouge du pixel
 *     int*           green,     Composante verte du pixel
 *     int*           blue,      Composante vleue du pixel
 *     int*           alpha      Composante alpha (transparence) du pixel
 * );
 *
 *------------------------------------------------------------------------------
 */

#include  <MLV/MLV_all.h>

//
// Attention ! 
// Pour pouvoir compiler ce programme sous windows et sous macintosh,
// il faut, pour la déclaration du main, respecter strictement la syntaxe
// suivante :
//
int main( int argc, char *argv[] ){

	MLV_Event event_type=MLV_NONE;
	MLV_Keyboard_button touche;
	int width=640,height=480;
	int alpha;
	int x,y;
	MLV_Image *map, *display_map;

	//
	// On associe à chaque valeur de la composante alpha du fichier map.png,
	// un nom de pays.
	//
	const char* code_to_country[ 5 ] = {"", "Espagne", "France", "Italie", "Portugal" };

	//
	// On créé la fenêtre
	//
	MLV_create_window( "advanced - 8 - zone click", "zone click", width, height );

	//
	// On charge en mémoire deux fois la même image:
	// La version chargée dans la variable map va servir à connaitre la position
	// des différents pays en utilsant la valeur de la composante alpha.
	// La version chargée dans la variable display_map va servir pour
	// l'affichage. Pour cela, on modifie la composante alpha de l'image de
	// façon qu'elle soit homogènement opaque.
	//
	map = MLV_load_image("map.png");
	display_map = MLV_load_image("map.png");
	MLV_set_alpha_on_image( MLV_ALPHA_OPAQUE, display_map );


	//
	// On réajuste la taille de la fenêtre pour qu'elle soit adaptée à celle de
	// l'image.
	//
	MLV_get_image_size( map, &width, &height );
	MLV_change_window_size( width, height );

	//
	// Initialisation de l'affichage
	//
	MLV_draw_image( display_map, 0, 0 );
	MLV_actualise_window();

	//
	// Dès que l'utilisateur clique sur l'image, on affiche le nom du pays
	// correspondant à la zone cliquée.
	// Si l'utilisateur appuie sur une touche du clavier, on quitte
	//
	while( event_type!=MLV_KEY || touche != MLV_KEYBOARD_ESCAPE ){
		event_type = MLV_wait_keyboard_or_mouse( &touche, NULL, NULL, &x, &y );
		if( event_type == MLV_MOUSE_BUTTON ){
			MLV_draw_image( display_map, 0, 0 );
			MLV_get_pixel_on_image( map, x, y, NULL, NULL, NULL, &alpha );
			if( 1 <= alpha && alpha <= 4 ){
				MLV_draw_text( x,y, code_to_country[alpha], MLV_COLOR_BROWN );
			}
			MLV_actualise_window();
		}
	}

	//
	// On libère l'espace mémoire allouée aux images
	//
	MLV_free_image( display_map );
	MLV_free_image( map );

	//
	// On ferme la fenêtre
	//
	MLV_free_window();

	return 0;
}
