# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
XML_ENCODING='utf-8'
import libxml2
import mobius.model.case
import mobius.model.item
import mobius

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Persistence layer for case
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __set_prop (self, node, name, value):
    if value != None:
      node.setProp (name, value.encode (XML_ENCODING))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load application data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    doc = libxml2.parseFile (path)
    node = doc.getRootElement ()

    case = self.load_case (node)
    case.filename = path
    case.set_new (False)
    
    return case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load case from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_case (self, node):
    case = mobius.model.case.Case ()
    case.set_rootdir (self.__get_prop (node, 'rootdir'))
    case.set_next_uid (int (self.__get_prop (node, 'next_uid') or '1'))

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'item':
        child = self.load_item (node, case)
        child.parent = case
        case.add_child (child)

      elif node.type == 'element' and node.name == 'attribute':
        id, value = self.load_attribute (node)
        case.set_attribute (id, value)

      elif node.type == 'element' and node.name == 'counter':
        id, value = self.load_counter (node)
        case.set_counter (id, value)

      node = node.next

    return case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load item from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_item (self, node, case):
    item = mobius.model.item.Item ()
    item.uid = int (self.__get_prop (node, 'uid'))
    item.persistent = True
    item.case = case

    category_id = self.__get_prop (node, 'category')
    item.category = mobius.mediator.call ('category.get', category_id)

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'item':
        child = self.load_item (node, case)
        item.add_child (child)

      elif node.type == 'element' and node.name == 'attribute':
        id, value = self.load_attribute (node)
        item.set_attribute (id, value)

      node = node.next

    return item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load attribute from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    name = self.__get_prop (node, 'name')
    value = self.__get_prop (node, 'value')
    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load counter from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_counter (self, node):
    name = self.__get_prop (node, 'name')
    value = int (self.__get_prop (node, 'value'))
    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save case into XML file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save (self, case):
    doc = libxml2.newDoc ('1.0')
    node = self.save_case (case)
    doc.addChild (node)
    doc.saveFormatFileEnc (case.filename, XML_ENCODING, 1)
    doc.freeDoc ()

    case.set_new (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_case (self, case):
    node = libxml2.newNode ('case')
    self.__set_prop (node, 'rootdir', case.get_rootdir ())
    self.__set_prop (node, 'next_uid', str (case.get_next_uid ()))

    # attributes
    for name, value in case.iter_attributes ():
      child = self.save_attribute (name, value)
      node.addChild (child)

    # counters
    for name, value in case.iter_counters ():
      child = self.save_counter (name, value)
      node.addChild (child)

    # items
    for item in case:
      if item.is_persistent:
        child = self.save_item (item)
        node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_item (self, item):
    node = libxml2.newNode ('item')
    self.__set_prop (node, 'uid', str (item.uid))
    self.__set_prop (node, 'category', item.category.id)

    # attributes
    for id, value in item.iter_attributes ():
      child = self.save_attribute (id, value)
      node.addChild (child)

    # sub-items
    for subitem in item:
      if subitem.is_persistent:
        child = self.save_item (subitem)
        node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save atttribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_attribute (self, name, value):
    node = libxml2.newNode ('attribute')
    self.__set_prop (node, 'name', name)
    self.__set_prop (node, 'value', value)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save counter
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_counter (self, name, value):
    node = libxml2.newNode ('counter')
    self.__set_prop (node, 'name', name)
    self.__set_prop (node, 'value', str (value))

    return node
