# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import gobject

(ATTR_ID, ATTR_NAME, ATTR_VALUE) = range (3)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
# @brief Attribute viewer
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
class AttributeViewer (gtk.TreeView):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, *args):
    datastore = gtk.ListStore (str, str, str)
    gtk.TreeView.__init__ (self, datastore)
    self.set_rules_hint (True)
    self.set_enable_search (False)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Property')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_NAME)
    self.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_edited)

    tvcolumn = gtk.TreeViewColumn ('Value')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_VALUE)
    self.append_column (tvcolumn)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # Clear attributes
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def clear (self):
    model = self.get_model ()
    model.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # Show attributes
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def view (self, obj):
    model = self.get_model ()
    model.clear ()
    self.object = obj

    for attr in obj.category.get_attribute_list ():
      self.add (attr.id, attr.name, obj.get_attribute (attr.id))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # Refresh view
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def refresh (self):
    model = self.get_model ()

    for row in model:
      id = row[ATTR_ID]
      row[ATTR_VALUE] = self.object.get_attribute (id) 

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # Add property
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add (self, id, name, value):
    model = self.get_model ()
    model.append ((id, name, value))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # On attribute edited
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_attribute_edited (self, cell, path_string, new_text, *args):
    model = self.get_model ()
    iter = model.get_iter_from_string (path_string)
    id = model.get_value (iter, ATTR_ID)
    old_text = model.get_value (iter, ATTR_VALUE)

    if old_text != new_text:
      model.set_value (iter, ATTR_VALUE, new_text)
      self.object.set_attribute (id, new_text)
      self.refresh ()
      self.emit ('attribute-edited', id, new_text)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
# @brief create widget's signal
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
gobject.type_register (AttributeViewer)
gobject.signal_new ('attribute-edited', AttributeViewer,
   gobject.SIGNAL_RUN_LAST, None,
  (gobject.TYPE_STRING, gobject.TYPE_STRING))
