# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case class
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Case (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.filename = None
    self.__rootdir = None
    self.__next_uid = 1
    self.__is_new = True
    self.__is_modified = False
    self.__attributes = {}
    self.__children = []
    self.__counters = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return iterator to children
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __iter__ (self):
    return iter (self.__children)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief set attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_attribute (self, id, value):
    if self.__attributes.get (id) != value:
      self.set_modified (True)
      self.__attributes[id] = value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    return self.__attributes.get (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return attribute iterator
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def iter_attributes (self):
    return self.__attributes.iteritems ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set root directory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_rootdir (self, path):
    if self.__rootdir != path:
      self.set_modified (True)
      self.__rootdir = path

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return root directory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_rootdir (self):
    return self.__rootdir

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set next uid
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_next_uid (self, uid):
    self.__next_uid = uid

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return next uid
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_next_uid (self):
    return self.__next_uid

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set counter
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_counter (self, id, value):
    self.__counters[id] = value
    self.set_modified (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return counter
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_counter (self, id):
    value = self.__counters.get (id, 0) + 1
    self.set_counter (id, value)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return counter iterator
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def iter_counters (self):
    return self.__counters.iteritems ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Change case modification flag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_modified (self, flag):
    self.__is_modified = flag

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return True if case is modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def is_modified (self):
    return self.__is_modified

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Change case new flag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_new (self, flag):
    self.__is_new = flag

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return True if case is new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def is_new (self):
    return self.__is_new

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_child (self, child):
    child.parent = self
    child.case = self
    self.__children.append (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Remove child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def remove_child (self, child):
    self.__children.remove (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return True if case has at least a child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def has_child (self):
    return len (self.__children) > 0

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def create_item (self, category):
    item = mobius.model.item.Item ()
    item.category = category
    item.case = self

    number = self.get_counter ('cat.%s' % category.id)
    item.set_attribute ('id', '%s_%d' % (category.id, number))
    item.set_attribute ('name', '%s %d' % (category.name, number))

    item.uid = self.__next_uid
    self.__next_uid += 1

    return item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get icon
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_icon (self):
    return CASE_ICON

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# case icon data (PNG 32x32)
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABGdBTUEAALGPC/xhBQAAAAZiS0dE
AP8A/wD/oL2nkwAAAAlwSFlzAAAN1wAADdcBQiibeAAAAAd0SU1FB9gHDxctIjTZVloAAAdmSURB
VFjDtVdrbBTXFf7unZl9+bHrfWCDbQzGBhu7QMwCBYfwMKkBA00NSmPiqo+A2tCoUqukClGqtmmq
FEgRjVSIqxYRhVQy0ATUgmxCABcDNjVyEtPYIQ6xedjx296d3Zmd1+2P9WvXi0kseqWruTtzdr7v
nPPdM+cCUxzXBCy6AMRPZsMOIPlB76EPMsjLq68uKLhWPv5eA2CzJuEdlwfHGEBiglfiZ8yO/7IK
CFN1Evn59S+cOtUhHz7c3u/11h8CLvDHAO5jN84opTACGyB/5MSfIoDfQKLxLt4N3YOoNqOPHUTm
lMCTkmpn7tjRdKuxsZs1Nnaz6uoOrbDwWk2901IpbYbCysBYGVj3CvQ2OvADAGBHsUSvxseyDCYz
MKUTOvsL1kW/Oyvr1C/i448dAAD+fgSysrjDTz3lmO3zKSCEwGYDt3/WwcfmJRqGJX4sdZ4MuKQh
/EZ8Cbl6Lsq0AqSPRiMB1LCjAMA5APB4TmQXFDgPybKR19oa+MZ9CWRn1z5fXp5UKEkqZJmAEMBx
fC/mDh6D3aFE6oYA6TuQoZTgeS07UlMsDgCPbOACv3Rp6NUtW9K279w5J7209NKbwPbemASs1tqZ
eXmmXW43LH6/CkKAaWcrMPfLo3AkBSONzYCxCtC2AYiPLehLnxfMKSlB7b59y7zz5tm53bs/bL18
ueelkedRBBiZPbv2SFGRefYI+PSrRzH/7mG4Pf5IyxTAWAtoG2PvJUXh8ecDO2GhTxe+/bbXxPME
t2+LqKnpOgM8PRCTQHp6zW/Xr7cWSpIGWQbSGv+Bgs4KpKQMRoSc5QF6MaAvvk+NqMtH5TvP4Xvl
TyA1Nc4kyxo4jmDPnubP6us7Xh5vO0rAbj+/OCeH/5HNZpj8fgWzWk5jec+bSJ0+DpwD2HJALQ1H
IHpIkgn79zwLR+IT2PXsAnAcgSxroJQgENDQ0NB/HnjGH4PAFavHI1fk5uqpfr+OOW0fYHXfQWSk
949ZJgDG6jA4zBPB/33xEZyo/Am+vbkYHo8FkhT2OjwpKipa2xsaBl6O/h8PAMnJwbe8XhT4/Tqy
Oq+g2HcQmRlj4CwDMIoArSicAkXhYTJpAABRtGDv73ch3laCJ7dlg1IyCk5pGFzTNNTV9V8eUX4E
gYSE91/MyVE2qSojM7qbsMl3EHOze8fyvQjQSwB9PqDrFAf+WI6q0wU4e/HnOFu1DO8d/yFWr1oF
h0OAJGnDoGHgMAkD1VX3Bm7eDP4ull54TeMu9PcrL5jNqrWk/RAWLO0ajQ1bCahbAeYEPrmRgX2v
PYf5uY9hxXIJJd/aC+/iRVi7Jg0ch3Ehp6AUowQ4jqCp4d4Xfv+2lpgEJGlt/Z07Vdfj4wOPdzE3
gPZw2FcByvfHVLL7ly+iaM0S8DxFYqIZGz1FsNn4qHBHes5xBKKowRA7Wyf9GoZC3Ct379LBi2wZ
hvrCm5qxyE2amMAhFDIgSTokSQNjDJKkDU8dsqyPPhuZfr+Kf711Ebn5GXsnJSBJj9f29lquf6A/
gqY2Tzj9QwBYZGEJ14cxgDDo+N9ja1HUcPLIJVb05HKs25pzoq7uzoJJ+wFV5f/QPWDzNQ8Nb/Ae
gPjCS5/PCsa4CLD7kRgBr6q8ahRt9ZKUFAsEgZtFKb3a0tL3zH0JaNr6c6IYf/20byEGeylIL0Dv
DBeYoBmGQaNA9Rie6wgENFz8Z4O+ctNC6nabwfMUgkAhCJxNEMhfu7rEvzPGEmN2RIyZXj8jPSo2
fu4CJIDcHalwAjSNGwWJDvfIPVFUce1co/bNdfM5t9syAjx8Da9NJmobn9yoz8iGMyqmNX4ykAww
gAxXYVG0QAnxMXI+FolgUEfT5SZ14aO5vMttjQLnIAiUmUz0VYfD+h1CiH+SntC6v1pcGhrso8Aw
gYBogSxzE8I9QiQY1NDyn2Y1f8Vc3u0xx/CcBgSB/25KSsKvCCHsAU1p8cn31fXXP2x1hSOgAqJo
haKwqC03Fom2pptq7pJM3uk0k/HA4fxz7TxPCjMzHce/clcsw/5Gy0CKQToB0gf4fDZomjFB7cGg
hi8/u6XOWTiLJjnNZJzghq/8WUGg3vz85I++ZlteXHkpsPSWv42CtgHdXXaoqjZB+YN327WM3FQk
Oc1cGHTUe0MQ6Ctr1szc4PXO6J3SuaBKKdrTeMPFSFe4DiiKEan8nk4tZfYMZndahXG5hiBwTBDI
9o0bs39NCDGmfDDpR9nf2oamDZEhwO+zIBQaU77h69Y9M11aktssRAuO52mvruee+Krt/yQnI8Ja
A2kntW6CoSErZFmFJOng5D7DMd0ZsjutlhiCg8nExSUkfJH1EAgAaXLTTzta4nRZDNcAGwYNhydO
drgstomC4yAIdFAQuOINGzI/fSgEfoyOoNEsXVGHKOwmkblcZiXBFW8bFlyPINAbgkA/FQROEgTa
bTbza0tL59V+nRMY/yAD+bb+2u0uz8pZLukWkjzvCQKtoVSvP3Jkc8+ITUVFgw1g9rKyvE487MEA
cn5N+hb8n8b/AM4MzLllmV4aAAAAAElFTkSuQmCC'''

import mobius.ui
CASE_ICON = mobius.ui.create_icon (ICON_DATA)
