# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import os.path
import gtk
import gobject
import mobius
import mobius.config
import mobius.ui.about_dialog
import mobius.ui.case_treeview
import mobius.ui.attribute_viewer
import mobius.ui.extension_manager_dialog
import mobius.ui.add_item_dialog

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case properties dialog
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class CasePropertiesDialog (gtk.Dialog):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, case):
    gtk.Dialog.__init__ (self, mobius.config.APP_TITLE, None, gtk.DIALOG_MODAL,
       (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    self.set_position (gtk.WIN_POS_CENTER)
    self.set_size_request (580, 480)
    self.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
    self.set_border_width (10)

    table = gtk.Table (5, 2)
    table.set_border_width (10)
    table.set_col_spacings (5)
    table.set_row_spacings (5)
    table.show ()
    self.vbox.pack_start (table)

    icon = case.get_icon ()
    image = gtk.Image ()
    image.set_from_pixbuf (icon)
    image.show ()
    table.attach (image, 0, 1, 0, 1, 0, 0)

    label = gtk.Label ()
    label.set_markup ('<b>Case Properties</b>')
    label.set_alignment (0, -1)
    label.show ()
    table.attach (label, 1, 2, 0, 1, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('ID')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 1, 2, 0, 0)

    self.case_id_entry = gtk.Entry ()
    self.case_id_entry.set_text (case.get_attribute ('id'))
    self.case_id_entry.show ()
    table.attach (self.case_id_entry, 1, 2, 1, 2, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('Name')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 2, 3, 0, 0)

    self.case_name_entry = gtk.Entry ()
    self.case_name_entry.set_text (case.get_attribute ('name'))
    self.case_name_entry.show ()
    table.attach (self.case_name_entry, 1, 2, 2, 3, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('Base dir')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 3, 4, 0, 0)

    hbox = gtk.HBox ()
    hbox.set_spacing (5)
    hbox.show ()
    table.attach (hbox, 1, 2, 3, 4, gtk.FILL | gtk.EXPAND, 0)

    self.case_rootdir_entry = gtk.Entry ()
    self.case_rootdir_entry.set_text (case.get_rootdir ())
    self.case_rootdir_entry.show ()
    hbox.pack_start (self.case_rootdir_entry)

    button = gtk.Button (stock=gtk.STOCK_OPEN)
    button.connect ('clicked', self.on_folder_choose)
    button.show ()
    hbox.pack_end (button, False, False)

    label = gtk.Label ('Description')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 4, 5, 0, 0)

    frame = gtk.Frame ()
    frame.show ()
    table.attach (frame, 1, 2, 4, 5)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.case_description_textview = gtk.TextView ()
    self.case_description_textview.set_wrap_mode (gtk.WRAP_WORD)
    self.case_description_textview.show ()
    sw.add (self.case_description_textview)

    buffer = self.case_description_textview.get_buffer ()
    buffer.set_text (case.get_attribute ('description'))
    self.case = case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Run dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def run (self):
    rc = gtk.Dialog.run (self)

    # set new values, if OK
    if rc == gtk.RESPONSE_OK:
      id = self.case_id_entry.get_text ().decode ('utf-8')
      name = self.case_name_entry.get_text ().decode ('utf-8')
      rootdir = self.case_rootdir_entry.get_text ().decode ('utf-8')

      buffer = self.case_description_textview.get_buffer ()
      start, end = buffer.get_bounds ()
      description = buffer.get_text (start, end).decode ('utf-8').strip ()

      self.case.set_attribute ('id', id)
      self.case.set_attribute ('name', name)
      self.case.set_attribute ('description', description)
      self.case.set_rootdir (rootdir)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief On folder choose
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_folder_choose (self, widget, *args):
    dialog = gtk.FileChooserDialog ('Select case rootdir', self,
         gtk.FILE_CHOOSER_ACTION_CREATE_FOLDER,
        (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    dialog.set_current_folder (self.case_rootdir_entry.get_text ())

    rc = dialog.run ()

    if rc == gtk.RESPONSE_OK:
      self.case_rootdir_entry.set_text (dialog.get_filename ())

    dialog.destroy ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case window
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Window (gtk.Window):

  def __init__ (self, *args):
    self.case = None

    gtk.Window.__init__ (self, *args)
    self.connect ('delete-event', self.on_file_close)
    self.set_size_request (800, 600)
    self.set_title ('Mobius v%s' % mobius.config.APP_VERSION)

    # accel_group
    accel_group = gtk.AccelGroup ()
    self.add_accel_group (accel_group)

    # accel_group
    accel_group = gtk.AccelGroup ()
    self.add_accel_group (accel_group)

    # vbox
    vbox = gtk.VBox (False, 1)
    vbox.set_border_width (1)
    self.add (vbox)
    vbox.show ()

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    vbox.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.ImageMenuItem (gtk.STOCK_NEW, accel_group)
    item.connect ("activate", self.on_file_new)
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_OPEN, accel_group)
    item.connect ("activate", self.on_file_open)
    item.show ()
    menu.append (item)

    self.save_menuitem = gtk.ImageMenuItem (gtk.STOCK_SAVE, accel_group)
    self.save_menuitem.connect ("activate", self.on_file_save)
    self.save_menuitem.set_sensitive (False)
    self.save_menuitem.show ()
    menu.append (self.save_menuitem)

    self.save_as_menuitem = gtk.ImageMenuItem (gtk.STOCK_SAVE_AS, accel_group)
    self.save_as_menuitem.connect ("activate", self.on_file_save_as)
    self.save_as_menuitem.set_sensitive (False)
    self.save_as_menuitem.show ()
    menu.append (self.save_as_menuitem)

    item = gtk.ImageMenuItem (gtk.STOCK_CLOSE, accel_group)
    item.connect ("activate", self.on_file_close)
    item.show ()
    menu.append (item)

    item = gtk.SeparatorMenuItem ()
    item.show ()
    menu.append (item)

    self.properties_menuitem = gtk.ImageMenuItem (gtk.STOCK_PROPERTIES, accel_group)
    self.properties_menuitem.connect ("activate", self.on_file_properties)
    self.properties_menuitem.set_sensitive (False)
    self.properties_menuitem.show ()
    menu.append (self.properties_menuitem)

    item = gtk.SeparatorMenuItem ()
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_QUIT, accel_group)
    item.connect ("activate", self.on_file_quit)
    item.show ()
    menu.append (item)

    item = gtk.MenuItem ('_Tools')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.MenuItem ('Extensions...')
    item.connect ("activate", self.on_extension_manager)
    item.show ()
    menu.append (item)

    item = gtk.MenuItem ('_Help')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.ImageMenuItem (gtk.STOCK_ABOUT, accel_group)
    item.connect ("activate", self.on_help_about)
    item.show ()
    menu.append (item)

    # toolbar
    self.tooltips = gtk.Tooltips ()

    handlebox = gtk.HandleBox ()
    handlebox.show ()
    vbox.pack_start (handlebox, False, False)

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    handlebox.add (toolbar)

    toolitem = gtk.ToolButton (gtk.STOCK_NEW)
    toolitem.connect ("clicked", self.on_file_new)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Start new case")
    toolbar.insert (toolitem, -1)

    toolitem = gtk.ToolButton (gtk.STOCK_OPEN)
    toolitem.connect ("clicked", self.on_file_open)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Open an existing case")
    toolbar.insert (toolitem, -1)

    self.save_toolitem = gtk.ToolButton (gtk.STOCK_SAVE)
    self.save_toolitem.set_sensitive (False)
    self.save_toolitem.connect ("clicked", self.on_file_save)
    self.save_toolitem.show ()
    self.save_toolitem.set_tooltip (self.tooltips, "Save current case")
    toolbar.insert (self.save_toolitem, -1)

    self.save_as_toolitem = gtk.ToolButton (gtk.STOCK_SAVE_AS)
    self.save_as_toolitem.set_sensitive (False)
    self.save_as_toolitem.connect ("clicked", self.on_file_save_as)
    self.save_as_toolitem.show ()
    self.save_as_toolitem.set_tooltip (self.tooltips, "Save case as")
    toolbar.insert (self.save_as_toolitem, -1)

    toolitem = gtk.SeparatorToolItem ()
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    self.add_toolitem = gtk.ToolButton (gtk.STOCK_ADD)
    self.add_toolitem.set_sensitive (False)
    self.add_toolitem.connect ("clicked", self.on_add_item)
    self.add_toolitem.show ()
    self.add_toolitem.set_tooltip (self.tooltips, "Add item to case")
    toolbar.insert (self.add_toolitem, -1)

    self.remove_toolitem = gtk.ToolButton (gtk.STOCK_REMOVE)
    self.remove_toolitem.set_sensitive (False)
    self.remove_toolitem.connect ("clicked", self.on_remove_item)
    self.remove_toolitem.show ()
    self.remove_toolitem.set_tooltip (self.tooltips, "Remove item from case")
    toolbar.insert (self.remove_toolitem, -1)

    toolitem = gtk.SeparatorToolItem ()
    self.properties_toolitem = gtk.ToolButton (gtk.STOCK_PROPERTIES)
    self.properties_toolitem.set_sensitive (False)
    self.properties_toolitem.connect ("clicked", self.on_file_properties)
    self.properties_toolitem.show ()
    self.properties_toolitem.set_tooltip (self.tooltips, "Case properties")
    toolbar.insert (self.properties_toolitem, -1)

    toolitem = gtk.SeparatorToolItem ()
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    toolitem = gtk.ToolButton (gtk.STOCK_QUIT)
    toolitem.connect ("clicked", self.on_file_quit)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Exit from Mobius")
    toolbar.insert (toolitem, -1)

    # workarea
    hpaned = gtk.HPaned ()
    #hpaned.set_position (200)
    hpaned.show ()
    vbox.pack_start (hpaned)

    # case treeview
    frame = gtk.Frame ()
    frame.show ()
    hpaned.pack1 (frame, True)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.case_treeview = mobius.ui.case_treeview.CaseTreeView ()
    self.case_treeview.connect ('item-selection', self.on_item_selection)
    self.case_treeview.connect ('case-selection', self.on_case_selection)
    self.case_treeview.connect ('case-modified', self.on_case_modified)

    self.case_treeview.show ()
    sw.add (self.case_treeview)

    # attributes viewer
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    hpaned.pack2 (sw)

    self.attribute_viewer = mobius.ui.attribute_viewer.AttributeViewer ()
    self.attribute_viewer.connect ('attribute-edited', self.on_attribute_edited)
    self.attribute_viewer.show ()
    sw.add (self.attribute_viewer)

    # status bar
    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_IN)
    frame.show ()
    vbox.pack_end (frame, False, False)

    self.status_label = gtk.Label ()
    self.status_label.set_alignment (0, -1)
    self.status_label.show ()
    frame.add (self.status_label)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief update title
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_modified (self, *args):
    title = 'Mobius v%s' % mobius.config.APP_VERSION
    if self.case:
      title += ' [%s]' % self.case.get_attribute ('name')
      if self.case.is_modified ():
        title += '*'

    self.set_title (title)
    self.save_menuitem.set_sensitive (self.case.is_modified ())
    self.save_toolitem.set_sensitive (self.case.is_modified ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle item selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_item_selection (self, treeview, iter, obj, *args):
    self.attribute_viewer.view (obj)
    self.add_toolitem.set_sensitive (True)
    self.remove_toolitem.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle case selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_selection (self, treeview, iter, obj, *args):
    self.attribute_viewer.clear ()
    self.add_toolitem.set_sensitive (True)
    self.remove_toolitem.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_edited (self, viewer, id, value):
    self.case.set_modified (True)
    self.on_case_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief add case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def add_case (self, case):

    # create a new window for case, if necessary
    if self.case:
      window = Window ()
      window.show ()
      self.manager.add_window (window)
    else:
      window = self

    # configure window
    window.case = case
    window.on_case_modified ()
    window.properties_menuitem.set_sensitive (True)
    window.properties_toolitem.set_sensitive (True)
    window.save_as_menuitem.set_sensitive (True)
    window.save_as_toolitem.set_sensitive (True)
    window.case_treeview.set_case (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief new case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_new (self, widget, *args):
    case = mobius.app.new_case ()
    self.add_case (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief open case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_open (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Open Case', parent=self)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.case')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # populate case
    case = mobius.app.open_case (filename)
    self.add_case (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_close (self, widget, *args):

    if self.case:

      # show 'save changes' dialog if necessary
      if self.case.is_modified ():
        dialog = gtk.MessageDialog (self,
                        gtk.DIALOG_MODAL,
                        gtk.MESSAGE_QUESTION,
                        gtk.BUTTONS_YES_NO,
                        "Save changes to '%s'?" % self.case.filename)
        dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
        rc = dialog.run ()
        dialog.destroy ()

        if rc == gtk.RESPONSE_CANCEL:
          return True

        elif rc == gtk.RESPONSE_YES:
          mobius.app.save_case (self.case)

      mobius.app.close_case (self.case)

    # close window
    self.manager.remove_window (self)
    self.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_save (self, widget, *args):

    if self.case.is_new ():
      self.on_file_save_as (widget, *args)
    else:
      mobius.app.save_case (self.case)
      self.on_case_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_save_as (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Save Case', parent=self, action=gtk.FILE_CHOOSER_ACTION_SAVE)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)
    fs.set_do_overwrite_confirmation (True)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.case')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # save file
    self.case.filename = filename
    mobius.app.save_case (self.case)
    self.on_case_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief case properties
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_properties (self, widget, *args):
    dialog = CasePropertiesDialog (self.case)
    dialog.run ()
    dialog.destroy ()

    self.on_case_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief quit application
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_quit (self, widget, *args):

    # if application is modified, show save/ignore/cancel dialog
    if mobius.app.is_modified ():
      dialog = gtk.MessageDialog (self,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "There are unsaved cases. Save them before quitting?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return

      elif rc == gtk.RESPONSE_YES:
        mobius.app.save_all_cases ()

    # else show 'do you want to quit' dialog
    else:
      dialog = gtk.MessageDialog (self,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Do you really want to quit?")

      rc = dialog.run ()
      dialog.destroy ()

      if rc != gtk.RESPONSE_YES:
        return

    # stop UI
    mobius.app.ui_manager.stop ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call add item dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_add_item (self, widget, *args):
    dialog = mobius.ui.add_item_dialog.Dialog ()
    rc = dialog.run ()

    amount = dialog.get_amount ()
    category = dialog.get_category ()
    dialog.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    for i in range (amount):
      item = self.case.create_item (category)
      self.case_treeview.add_child (item)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call remove item dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_remove_item (self, widget, *args):
    item = self.case_treeview.get_selected_item ()

    title = 'You are about to remove an item'
    if item.has_child ():
      title += ' and its sub-items'
    title += '. Are you sure?'
    
    dialog = gtk.MessageDialog (self,
                    gtk.DIALOG_MODAL,
                    gtk.MESSAGE_QUESTION,
                    gtk.BUTTONS_YES_NO,
                    title)
    rc = dialog.run ()
    dialog.destroy ()

    if rc != gtk.RESPONSE_YES:
      return

    self.case_treeview.remove_selected_item ()
    item.parent.remove_child (item)
    self.remove_toolitem.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call extension manager dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_manager (self, widget, *args):
    dialog = mobius.ui.extension_manager_dialog.Dialog ()
    rc = dialog.run ()
    dialog.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call 'about' dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_help_about (self, widget, *args):
    dialog = mobius.ui.about_dialog.Dialog ()
    dialog.run ()
    dialog.destroy ()
