# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Attribute viewer window
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
(ATTR_ID, ATTR_NAME, ATTR_VALUE) = range (3)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Attribute Viewer widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.ScrolledWindow):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.ScrolledWindow.__init__ (self)
    self.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    self.show ()

    # listview
    datastore = gtk.ListStore (str, str, str)
    self.listview = gtk.TreeView (datastore)
    self.listview.set_rules_hint (True)
    self.listview.set_enable_search (False)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Property')
    tvcolumn.set_resizable (True)
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_NAME)
    self.listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_edited)

    tvcolumn = gtk.TreeViewColumn ('Value')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_VALUE)
    self.listview.append_column (tvcolumn)
    self.listview.show ()
    self.add (self.listview)

    self.mediator = mediator.new_client_mediator ()
    self.mediator.emit ('attribute-viewer-started')

    self.mediator.connect ('item-selected', self.on_item_selected)
    self.mediator.connect ('case-selected', self.on_case_selected)
    self.mediator.connect ('attribute-viewer-started', self.on_started)
    self.mediator.connect ('item.attribute-modified', self.on_item_attribute_modified)
    self.item = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle started event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_started (self):
    self.mediator.disconnect ('item-selected')
    self.mediator.disconnect ('case-selected')
    self.mediator.disconnect ('attribute-viewer-started')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle stop event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_widget_stopped (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle case selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_selected (self, case):
    model = self.listview.get_model ()
    model.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle item selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_item_selected (self, item):
    model = self.listview.get_model ()
    model.clear ()

    for attr in self.mediator.call ('category.get-attribute-list', item.category):
      model.append ((attr.id, attr.name, item.get_attribute (attr.id)))

    self.item = item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_edited (self, cell, path_string, new_text, *args):
    model = self.listview.get_model ()
    iter = model.get_iter_from_string (path_string)
    attr_id = model.get_value (iter, ATTR_ID)

    self.mediator.call ('item.set-attribute', self.item, attr_id, new_text)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Event handler: item.attribute-modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_item_attribute_modified (self, item, id, old_value, new_value):
    if item == self.item:
      model = self.listview.get_model ()

      # set values
      for row in model:
        attr_id = row[ATTR_ID]
 
        if attr_id == id:
          row[ATTR_VALUE] = new_value
        else:
          row[ATTR_VALUE] = self.item.get_attribute (attr_id) 
  
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAIiUlEQVRYhcWXSaxm11GAvzrDvfd/
//C6+3W/bne3e4jbcRgNQQokYClKFkbyBgmhKItIWSCkLCzkXRZIUZCQvEAoGyQWLCKBBQIhWEKE
hCEIiXasGNvYJm56fD28of/3/ulOZygWfm2SuN02bFJS6SxunarvlKrqniOqyk9SzE80OuD+P5t+
6zuYsefE0HGxtGzWkauLwK0/+yKz/6sveebb2x9pdPTE8aPjylwYey5NPJ8aeS6tOU57w6SwuC7S
dZlpHdlaBN6tI+/MA1eWq3br7rVbq0f5/jgZOPtTa80fFoXfXCtMVRg1okoOqh0QISrYwnC8NLI5
LOTTs5xj17Td3d35K8DXgfyhACmEjwIobh+068Zn8EUoysIPjdqRhOKYbeXnj8F/TC1Xu0K3Wh93
liE281VIzcLmZnWEnOSRGUjxIwFUk8e45HLKNgS188L45Ep7tSntSYv7x33idk9KGUwXssnB2xxz
ThFN8ZHO3YtfuwjAxpkz/PbXX36YjWiKRpMVI0Gsihn5yu53+Gtz3K0ZDkUmHheCdjH3XlOIIQRy
COZRAH/yzc/+aBv+5UtfIcf4AdWUjKZkDEkGNrG/DPr6FPGCc5YCxScF+k4kBnIfyDGaHIN5mL8c
Iyef+02+cfn0B+fA3/zDC/zdNz+LHqZPUyTHmDXGXGpOB3U0b9zrKhPzmjVUGUxvqDRRaeiL2PfE
2Kcc+pxDyDn0/Lief+HL78f70EH093/0LJ1xdMZZix4pHJtdzI//YLc7VREnlXQ+KyQDQcGGrihS
P1kz+dTQcsYbPW40jTVGflh/9oXnfyTOIyfhG9/6Ao8t9lKwbvOgzYN/v9Pa0IdkNWWfs3pLjpCt
kE3qNHR17upGm7ZzIcRRSvmoJiuaLJosv/a7f/CBGB82ByxwArjw4vf/9JlvbTxvb68QV060HG+Q
19Y1lZacoW6V3ClajDQPvYZwkJvZvtEu2tiSzv/678d49hWeHn6Rh02kHwZ4CvgF4EmeffZSGA7P
TAs3dm59bftgaWfjszIZjhmEDp3eoUUVdfSm4mDHqHMW7JrK+JQU5TFj7v8njttj4BKwBbQPO+kD
ANELF55nff2XjPcZa6UFmRsriTb9xuyyvhym8rr5pN4rj4kxQ47v7jHc25O3r2Z62aSWitjdcFbv
sF7+N+fPz9KJn5mePPXExT9uOLPTsLgJvHuo14F7QHIPsJZ376pvmt73vRuHYEaZUoYF/cWxu/Sl
p+R3bvxtMN/+cw52jshfnHvO/NX5Lxg+/6vYUDF/5235xMnbPHVimh8/spdGm62mycDNESnr9bIa
6NmhGZ8t8+BX4ix10/nB6v5s5/eA/3ofYGCM6a21TVEY671ziFvzpSsvv+6KcxPhX79j5VXVo+zn
r916Sb/6by/FFz/3c+7uxcf4zOeSlufqlKOR2+0RKXePmE+n50T99eL7R/9a7qTrZm++SG0dEjXq
WioT8QDuwSBuQQKYYIwJRmw3nDi/vVP420sv+xluWumo9F45MAs7kNw4bs93SOEeGj4j9dZYWrfA
jVZmfuEqu+Utubp6y7zV7BbrNVrUqOuxNpAJqPaHNfD5Q4DXDqvEGZPjcBibg2nUt94oj4GjLolT
z45Rve+drqSgo6BJayb2TtKiSiYaHfkJmhq9Mn3bPLN5WufzqZ5JDGxDpIfcoLRkbeF9gBOHRehB
O6CqKtMsl1X32mtDAdcDRa7YdsIBxiytSaustKTYB2tzZ8TUKQ18peTefG/2z7IcqI7TwJR9Ybqm
cdqQtAUa0A6jHRpr5EEXmASFWnucstwMMRarV181IxBnbVpLKUZrzZb3eT+jKxF6jXQoqS9Ua4cd
mqRNkMv7/8TK12ZgEMkgMYnpGUkvLrWqqaOjBW3R5ex/27ACxhnW6vnc3H/zzbSWc6yMscWhwVrf
pCdSy80Y9Z2c8pYOtIO+boZunL2lQb9392UaM2c4MZpXGemMSBCVGu+60vrkBiJmPRKbkEKDhuIB
wAA40tW1b7a2dCNnOS8ix1S1zFk6IMVeqtTpLwbNT4dFuskyXRHJr/UnjYYhb137rs5lJx5bFzUZ
FxUhKRok5yg59lm16aN2QupjkbvsUYZB3wMoBAbcuxcmKcmGiOlF5I6qJBFRVZ10HU0AA7YAqVB5
UtWcur/Uf/nBzXwwnqXJWdCJilZKatHlTpv67UiPqtE+Zz0svAa0JaKMECoHZAP9T49GL8+N2VvU
9ePbIZzuVdc0ZwvoY1mZimirSg25gxbL9Vf6m+e2N/J4OJIitygNJiSVGJE+JdobSTOYfPjTUyGb
HrQhacEAz9ABTYZZdu7G8aJoLnr/huS83h8cbE5TOr2YTM42ymAK3Sbc+IRwrbJsHQyY/rKrHmNm
ntw76C6EFE9J1IFmoY2qq6I1/VXRnFWsBRxRCxY64DpD3lVhVxLOHba/Lry/2VTVMqVUlHt7g7Lv
r50D5xeLIl6+XD4NS2C2hPaeEG9ZUj8s9jdODK6k0cDXGk/Vy+UT7ax/8ni3/njVDCvtyGwwS0e4
pgOuqeeOGO6bTE3DFKF3QFJol97v2sGgTtvbvlgu/QiKBMVA1XjInUicQZhBfx/iQSavrC17bGVF
HD7shvV4haN8t5X+9JXinbN5XRu13AWWkqkl00g8XJUp0DhAAV15P1vNZvXa7q6cAFuBUbAJBFCn
qhWkhZKaRF4EaJOzViqX09IswsyJzaVXqkaa63fDbSteLElVlB6lFWhF6UwikpkrhAcAOezsLKvF
wmyCjAEDUh+qiGh+z44J6CczOu7gzc7J/UVvpvUODLOREqMGa8CbZE3OUVAykCQTJZMkkqRHgR4h
v58BoH/YhWEDZBEj3SHAw2RukogCHaIdoor0BNHDLXJ4SAFUUZ8A+97Hj/M0+zjvd/2x9WPL/wD7
d1t5fjif1gAAAABJRU5ErkJggg=='''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'attribute-viewer'
    self.name = 'Attribute Viewer'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Attribute viewer/manager'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.call ('toolbox.add', self.id, self.icon_data, 'Attribute\nViewer', self.on_activate)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    self.mediator.clear ()
    #self.mediator.call ('toolbox.remove', self.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_activate
  # @begin-sequence
  #   call-service ui.working-area.new
  #   message WorkingArea.set_default_size
  #   message WorkingArea.set_title
  #   message WorkingArea.set_icon
  #   message WorkingArea.set_widget
  #   message WorkingArea.show
  # @end-sequence
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_activate (self, item_id):
    widget = Widget (self.mediator)

    working_area = self.mediator.call ('ui.working-area.new', self.id)
    working_area.set_default_size (400, 500)
    working_area.set_title (self.name)
    working_area.set_icon (self.icon_data)
    working_area.set_widget (widget)
    working_area.show ()
