# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @TODO ler grupos hierarquicamente
# @TODO ver ICQTIC
# @TODO ver formato do icone
# @TODO tlv 0x66
# @TODO tlv 0xc8
# @TODO tlv 0xc9
# @TODO tlv 0xcb
# @TODO tlv 0xcc
# @TODO tlv 0x144
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# LAYOUT
# 1. Header
#    0	  16	signature 'AOL Feedbag vx.x\0'
#   17	   2	number of records
#   19     X    records ...
#
# 2.Record
#    0	   2	id length (n1)
#    2	  n1	id data
#    2+n1  2	group id
#    4+n1  2    user id
#    6+n1  2    type
#    8+n1  2	tlvs length (n2)
#
# 3. TLV (type, length, value) tuples
#    0	   2	type
#    2	   2	length (n3)
#    4     n3   value
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import struct
import datetime

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Domains
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Privacy mode:
#   1 - always visible to all users
#   2 - never visible to anyone
#   3 - only visible to always visible contacts
#   4 - always visible except to invisible contacts
#   5 - always visible to all contacts

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
base_datetime = datetime.datetime (1970, 1, 1)

ICQ_NORMAL = 0
ICQ_GROUP = 1
ICQ_INVISIBLE = 3
ICQ_VISIBILITY = 4
ICQ_ICQTIC = 9
ICQ_IGNORE = 14
ICQ_LASTUPD = 15
ICQ_ICON = 20

tlv_types = {0x0066 : 'unknown', #reqauth
             0x00c8 : 'unknown', #group items
             0x00c9 : 'unknown', #unknidle
             0x00ca : 'privacy',
             0x00cb : 'unknown', #visibility
             0x00cc : 'unknown', #import
             0x00cd : 'icqtic',
             0x00d5 : 'icon',
             0x0131 : 'alias',
             0x0137 : 'email',
             0x013a : 'sms',
             0x0144 : 'unknown', #identificar tipo
             0x0145 : 'lastupd'}

ROASTING_ARRAY = [0xF3, 0x26, 0x81, 0xC4, 0x39, 0x86, 0xDB, 0x92, 0x71, 0xA3, 0xB9, 0xE6, 0x53, 0x7A, 0x95, 0x7C]

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Check signature
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def signature (data):
  return data.startswith ('AOL Feedbag ')

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Decode AOL Feedbag files
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Decoder (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Buddy list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  class BuddyList (object):
    def __init__ (self):
      self.groups = {}
      self.buddies = {}
      self.last_update = None
      self.privacy = 0
      self.icqtic = ''
      self.ignore = []
      self.invisible = []
      self.icon = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Buddy group
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  class Group (object):
    def __init__ (self, id):
      self.id = id
      self.name = ''
      self.buddies = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Buddy
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  class Buddy (object):
    def __init__ (self, id, gid, uid):
      self.id = id
      self.gid = gid
      self.uid = uid
      self.alias = ''
      self.email = ''
      self.sms = ''
      self.last_update = None
      self.ignored = False
      self.invisible = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get word value
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def getword (self):
    value = (ord (self.data[self.pos]) << 8) + ord (self.data[self.pos+1])
    self.pos += 2

    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get variable length data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def getdata (self):
    length = self.getword ()
    value = self.data[self.pos:self.pos + length]
    self.pos += length

    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get record
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def getrecord (self):
    id = self.getdata ()
    gid = self.getword ()
    uid = self.getword ()
    type = self.getword ()
    tlvs = self.decode_tlvs ()

    return id, gid, uid, type, tlvs

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Decode data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def decode (self, data):
    if not data.startswith ('AOL Feedbag '):
      raise Exception, 'Invalid file format'

    # generate buddy list
    buddylist = self.BuddyList ()
    buddylist.version = data[12:15]

    # read and decode records
    self.pos = 17
    self.data = data
    records = self.getword ()

    for i in range (records):
      id, gid, uid, type, tlvs = self.getrecord ()

      if type == ICQ_NORMAL:
	uin = int (id) 
        buddy = self.Buddy (uin, gid, uid)
        buddy.alias = tlvs.get ('alias', '')
        buddy.email = tlvs.get ('email', '')
        buddy.sms = tlvs.get ('sms', '')
        buddy.last_update = self.decode_date (tlvs.get ('lastupd'))

        group = buddylist.groups.get (gid)
        group.buddies[uin] = buddy
        buddylist.buddies[uin] = buddy

      elif type == ICQ_GROUP:
        group = self.Group (gid)
        group.name = id
        buddylist.groups[gid] = group

      elif type == ICQ_INVISIBLE:
	uin = int (id) 
        buddy = self.Buddy (uin, gid, uid)
        buddy.alias = tlvs.get ('alias', '')
        buddy.email = tlvs.get ('email', '')
        buddy.sms = tlvs.get ('sms', '')
        buddy.last_update = self.decode_date (tlvs.get ('lastupd'))
        buddy.invisible = True

        buddylist.invisible.append (buddy)
        buddylist.buddies[uin] = buddy

      elif type == ICQ_VISIBILITY:
        buddylist.privacy = ord (tlvs.get ('privacy', '\0'))

      elif type == ICQ_ICQTIC:
        buddylist.icqtic = tlvs.get ('icqtic')

      elif type == ICQ_IGNORE:
	uin = int (id) 
        buddy = self.Buddy (uin, gid, uid)
        buddy.alias = tlvs.get ('alias', '')
        buddy.email = tlvs.get ('email', '')
        buddy.sms = tlvs.get ('sms', '')
        buddy.last_update = self.decode_date (tlvs.get ('lastupd'))
        buddy.ignored = True

        buddylist.ignore.append (buddy)
        buddylist.buddies[uin] = buddy

      elif type == ICQ_LASTUPD:
        buddylist.last_update = self.decode_date (tlvs.get ('lastupd'))

      elif type == ICQ_ICON:
        buddylist.icon = tlvs.get ('icon')

      else:
        print 'Unknown record type: %d' % type

    return buddylist

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Decode TLV block. Return dictionary
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def decode_tlvs (self):
    tlv_length = self.getword ()
    tlv_end = self.pos + tlv_length
    tlv = {}

    while self.pos < tlv_end:
      type = self.getword ()
      value = self.getdata ()
      var = tlv_types.get (type)

      if not var or var == 'unknown': # @DEPRECATED only for unknown types
        print '0x%04x' % type, len (value), ' '.join (['%02x' % ord (c) for c in value])

      else:
        tlv[var] = value

    return tlv

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Decode date
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def decode_date (self, data):
    if data:
      return base_datetime + datetime.timedelta (0, struct.unpack ('>I', data)[0])
    return None
