# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import gtk.gdk
import os.path

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
(ITEM_ICON, ITEM_NAME, ITEM_OBJ) = range (3)
(CATEGORY_ICON, CATEGORY_NAME, CATEGORY_OBJECT) = range (3)
DND_TARGETS = [('application/x-mobius-case-item', 0, 0)]

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case treeview
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class CaseTreeView (gtk.TreeView):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator, *args):
    self.mediator = mediator    
    datastore = gtk.TreeStore (gtk.gdk.Pixbuf, str, object)

    gtk.TreeView.__init__ (self, datastore)
    self.set_headers_visible (False)
    self.set_enable_tree_lines (True)
    self.connect ('cursor-changed', self.on_cursor_changed)
    self.connect ('test-expand-row', self.on_test_expand_row)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', ITEM_ICON)
    self.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ITEM_NAME)
    self.append_column (tvcolumn)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_case (self, case):
    model = self.get_model ()
    model.clear ()

    # insert case and case children
    iter = self.add_object (case)
    self.case = case

    # expand case, if it has children
    if case.has_child ():
      path = model.get_string_from_iter (iter)
      self.expand_row (path, False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set case name
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_case_name (self, name):
    model = self.get_model ()
    root = model.get_iter_root ()

    if root:
      model.set_value (root, ITEM_NAME, name)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add an object to treeview, including its sub-objects
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_object (self, obj, iter=None):
    model = self.get_model ()

    # eliminate parent placeholder, if necessary
    if iter:
      child_iter = model.iter_children (iter)

      if child_iter:
        child_obj = model.get_value (child_iter, ITEM_OBJ)

        if not child_obj:
          model.remove (child_iter)

    # retrieve object icon
    if not iter:	# case obj
      icon = self.mediator.call ('ui.render-icon', obj.icon_data, 32, 32)
    else:		# item obj
      icon = self.get_category_icon (obj.category)

    # insert object
    name = obj.get_attribute ('name')
    obj_iter = model.append (iter, (icon, name, obj))

    # insert place holder if obj has sub-items
    if obj.has_child ():
      model.append (obj_iter)

    # expand item at iter
    if iter:
      path = model.get_string_from_iter (iter)
      self.expand_row (path, False)

    # return iterator
    return obj_iter

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get icon
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_category_icon (self, category):
    icon_data = self.mediator.call ('category.get-icon-data', category)
    icon = self.mediator.call ('ui.render-icon', icon_data, 32, 32)
    return icon

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add child to currently selected item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_child (self, obj):
    model, iter = self.get_selection ().get_selected ()

    if iter:
      parent = model.get_value (iter, ITEM_OBJ)
      parent.add_child (obj)
      self.add_object (obj, iter)
      self.mediator.emit ('case-modified', self.case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Get selected item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_selected_item (self):
    model, iter = self.get_selection ().get_selected ()

    if iter:
      item = model.get_value (iter, ITEM_OBJ)
    else:
      item = None
   
    return item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Set selected item's name
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_selected_item_name (self, name):
    model, iter = self.get_selection ().get_selected ()

    if iter:
      model.set_value (iter, ITEM_NAME, name)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Remove selected item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def remove_selected_item (self):
    model, iter = self.get_selection ().get_selected ()
    if iter:
      item = model.get_value (iter, ITEM_OBJ)
      model.remove (iter)

      self.case.set_modified (True)
      self.mediator.emit ('case-modified', self.case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle test_expand_row event
  # @detail This function implements lazy evaluation. Every item inserted
  # receive a null child (obj = None). When row is expanded, three situations
  # arise:
  # 1. row has no child: object already loaded but no subitems were found
  # 2. row has null child: just a markplace. Object must be loaded
  # 3. row has valid child: object already loaded
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_test_expand_row (self, treeview, iter, path, *args):
    model = treeview.get_model ()
    obj = model.get_value (iter, ITEM_OBJ)
    child_iter = model.iter_children (iter)

    if child_iter:
      child_obj = model.get_value (child_iter, ITEM_OBJ)

      if not child_obj:
        model.remove (child_iter)

        for i in obj:
          self.add_object (i, iter)

    return False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle cursor changed event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_cursor_changed (self, treeview, *args):
    model, iter = treeview.get_selection ().get_selected ()

    if iter:
      obj = model.get_value (iter, ITEM_OBJ)

      if model.get_path (iter) == (0,):
        self.mediator.emit ('case-selected', obj)
      else:
        self.mediator.emit ('item-selected', obj)


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case properties dialog
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class CasePropertiesDialog (gtk.Dialog):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, case, mediator):
    gtk.Dialog.__init__ (self,
       'Case Properties [%s]' % case.get_attribute ('name'), None, gtk.DIALOG_MODAL,
       (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    self.set_position (gtk.WIN_POS_CENTER)
    self.set_default_size (580, 480)
    self.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
    self.set_border_width (10)
    self.mediator = mediator.new_client_mediator ()

    table = gtk.Table (5, 2)
    table.set_border_width (10)
    table.set_col_spacings (5)
    table.set_row_spacings (5)
    table.show ()
    self.vbox.pack_start (table)

    icon = self.mediator.call ('ui.render-icon', case.icon_data, 32, 32)
    image = gtk.Image ()
    image.set_from_pixbuf (icon)
    image.show ()
    table.attach (image, 0, 1, 0, 1, 0, 0)

    label = gtk.Label ()
    label.set_markup ('<b>Case Properties</b>')
    label.set_alignment (0, -1)
    label.show ()
    table.attach (label, 1, 2, 0, 1, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('ID')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 1, 2, 0, 0)

    self.case_id_entry = gtk.Entry ()
    self.case_id_entry.set_text (case.get_attribute ('id'))
    self.case_id_entry.show ()
    table.attach (self.case_id_entry, 1, 2, 1, 2, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('Name')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 2, 3, 0, 0)

    self.case_name_entry = gtk.Entry ()
    self.case_name_entry.set_text (case.get_attribute ('name'))
    self.case_name_entry.show ()
    table.attach (self.case_name_entry, 1, 2, 2, 3, gtk.FILL | gtk.EXPAND, 0)

    label = gtk.Label ('Base dir')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 3, 4, 0, 0)

    hbox = gtk.HBox ()
    hbox.set_spacing (5)
    hbox.show ()
    table.attach (hbox, 1, 2, 3, 4, gtk.FILL | gtk.EXPAND, 0)

    self.case_rootdir_entry = gtk.Entry ()
    self.case_rootdir_entry.set_text (case.get_rootdir () or os.getcwd ())
    self.case_rootdir_entry.show ()
    hbox.pack_start (self.case_rootdir_entry)

    button = gtk.Button (stock=gtk.STOCK_OPEN)
    button.connect ('clicked', self.on_folder_choose)
    button.show ()
    hbox.pack_end (button, False, False)

    label = gtk.Label ('Description')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 4, 5, 0, 0)

    frame = gtk.Frame ()
    frame.show ()
    table.attach (frame, 1, 2, 4, 5)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.case_description_textview = gtk.TextView ()
    self.case_description_textview.set_wrap_mode (gtk.WRAP_WORD)
    self.case_description_textview.show ()
    sw.add (self.case_description_textview)

    buffer = self.case_description_textview.get_buffer ()
    buffer.set_text (case.get_attribute ('description') or '')
    self.case = case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Run dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def run (self):
    rc = gtk.Dialog.run (self)

    # set new values, if OK
    if rc == gtk.RESPONSE_OK:

      # change id
      id = self.case.get_attribute ('id')
      new_id = self.case_id_entry.get_text ().decode ('utf-8')

      if new_id != id:
        self.case.set_attribute ('id', new_id)
        self.mediator.emit ('case.attribute-modified', self.case, 'id', id, new_id)

      # change name
      name = self.case.get_attribute ('name')
      new_name = self.case_name_entry.get_text ().decode ('utf-8')

      if new_name != name:
        self.case.set_attribute ('name', new_name)
        self.mediator.emit ('case.attribute-modified', self.case, 'name', name, new_name)

      # change description
      description = self.case.get_attribute ('description')
      buffer = self.case_description_textview.get_buffer ()
      start, end = buffer.get_bounds ()
      new_description = buffer.get_text (start, end).decode ('utf-8').strip ()

      if new_description != description:
        self.case.set_attribute ('description', new_description)
        self.mediator.emit ('case.attribute-modified', self.case, 'description', description, new_description)

      # change rootdir
      rootdir = self.case.get_rootdir ()
      new_rootdir = self.case_rootdir_entry.get_text ().decode ('utf-8')

      if new_rootdir != rootdir:
        self.case.set_rootdir (new_rootdir)
        self.mediator.emit ('case.attribute-modified', self.case, 'rootdir', rootdir, new_rootdir)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief On folder choose
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_folder_choose (self, widget, *args):
    dialog = gtk.FileChooserDialog ('Select case rootdir', self,
         gtk.FILE_CHOOSER_ACTION_CREATE_FOLDER,
        (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    dialog.set_current_folder (self.case_rootdir_entry.get_text ())

    rc = dialog.run ()

    if rc == gtk.RESPONSE_OK:
      self.case_rootdir_entry.set_text (dialog.get_filename ())

    dialog.destroy ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
# @brief Add item dialog
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
class AddItemDialog (gtk.Dialog):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    gtk.Dialog.__init__ (self, 'Add item', None, gtk.DIALOG_MODAL,
       (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    self.set_position (gtk.WIN_POS_CENTER)
    self.set_default_size (400, 300)
    self.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
    self.set_border_width (10)
    self.vbox.set_spacing (5)

    # datastore
    datastore = gtk.ListStore (gtk.gdk.Pixbuf, str, object)
    datastore.set_sort_column_id (CATEGORY_NAME, gtk.SORT_ASCENDING)
    categories = mediator.call ('category.get-list')

    for category in categories:
      category.icon = mediator.call ('ui.render-icon', category.icon_data, 32, 32)
      datastore.append ((category.icon, category.name, category))

    # listview
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    self.vbox.pack_start (sw)

    listview = gtk.TreeView (datastore)
    listview.set_rules_hint (True)
    listview.set_headers_visible (False)
    listview.connect ('cursor-changed', self.on_listview_cursor_changed)
    listview.show ()
    sw.add (listview)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', CATEGORY_ICON)
    listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', CATEGORY_NAME)
    listview.append_column (tvcolumn)

    # spinbutton
    hbox = gtk.HBox ()
    hbox.set_spacing (5)
    hbox.show ()
    self.vbox.pack_start (hbox, False)

    label = gtk.Label ('Amount')
    label.show ()
    hbox.pack_start (label, False)

    adj = gtk.Adjustment (1, 1, 9999, 1, 1, 0)
    self.spin = gtk.SpinButton (adj, 0, 0)
    self.spin.show ()
    hbox.pack_start (self.spin, False)

    placeholder = gtk.Label ()
    placeholder.show ()
    hbox.pack_start (placeholder)

    self.__category = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Return amount of items
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_amount (self):
    return int (self.spin.get_value ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Return select category
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_category (self):
    return self.__category

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle cursor changed event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_listview_cursor_changed (self, treeview, *args):
    model, iter = treeview.get_selection ().get_selected ()
    if iter:
      self.__category = model.get_value (iter, CATEGORY_OBJECT)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.set_border_width (5)
    self.set_spacing (5)
    self.show ()
    self.mediator = mediator.new_client_mediator ()
    vbox = self

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    vbox.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    self.save_menuitem = gtk.ImageMenuItem (gtk.STOCK_SAVE)
    self.save_menuitem.connect ("activate", self.on_file_save)
    self.save_menuitem.set_sensitive (False)
    self.save_menuitem.show ()
    menu.append (self.save_menuitem)

    item = gtk.ImageMenuItem (gtk.STOCK_SAVE_AS)
    item.connect ("activate", self.on_file_save_as)
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_CLOSE)
    item.connect ("activate", self.on_file_close)
    item.show ()
    menu.append (item)

    item = gtk.SeparatorMenuItem ()
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_PROPERTIES)
    item.connect ("activate", self.on_file_properties)
    item.show ()
    menu.append (item)

    # toolbar
    self.tooltips = gtk.Tooltips ()

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    vbox.pack_start (toolbar, False, False)

    self.save_toolitem = gtk.ToolButton (gtk.STOCK_SAVE)
    self.save_toolitem.set_sensitive (False)
    self.save_toolitem.connect ("clicked", self.on_file_save)
    self.save_toolitem.show ()
    self.save_toolitem.set_tooltip (self.tooltips, "Save current case")
    toolbar.insert (self.save_toolitem, -1)

    item = gtk.ToolButton (gtk.STOCK_SAVE_AS)
    item.connect ("clicked", self.on_file_save_as)
    item.show ()
    item.set_tooltip (self.tooltips, "Save case as")
    toolbar.insert (item, -1)

    toolitem = gtk.SeparatorToolItem ()
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    self.add_toolitem = gtk.ToolButton (gtk.STOCK_ADD)
    self.add_toolitem.set_sensitive (False)
    self.add_toolitem.connect ("clicked", self.on_add_item)
    self.add_toolitem.show ()
    self.add_toolitem.set_tooltip (self.tooltips, "Add item to case")
    toolbar.insert (self.add_toolitem, -1)

    self.remove_toolitem = gtk.ToolButton (gtk.STOCK_REMOVE)
    self.remove_toolitem.set_sensitive (False)
    self.remove_toolitem.connect ("clicked", self.on_remove_item)
    self.remove_toolitem.show ()
    self.remove_toolitem.set_tooltip (self.tooltips, "Remove item from case")
    toolbar.insert (self.remove_toolitem, -1)

    toolitem = gtk.ToolButton (gtk.STOCK_PROPERTIES)
    toolitem.connect ("clicked", self.on_file_properties)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Case properties")
    toolbar.insert (toolitem, -1)

    # case treeview
    frame = gtk.Frame ()
    frame.show ()
    vbox.pack_start (frame)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.case_treeview = CaseTreeView (self.mediator)
    self.case_treeview.enable_model_drag_source (gtk.gdk.BUTTON1_MASK, DND_TARGETS, gtk.gdk.ACTION_COPY)
    self.case_treeview.connect ('drag-begin', self.on_drag_begin)
    self.case_treeview.connect ('drag_data_get', self.on_drag_data_get)
    self.case_treeview.show ()
    sw.add (self.case_treeview)

    # status bar
    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_IN)
    frame.show ()
    vbox.pack_end (frame, False, False)

    self.status_label = gtk.Label ()
    self.status_label.set_alignment (0, -1)
    self.status_label.show ()
    frame.add (self.status_label)

    self.mediator.connect ('case-modified', self.on_case_modified)
    self.mediator.connect ('case.attribute-modified', self.on_case_attribute_modified)
    self.mediator.connect ('case-selected', self.on_case_selected)
    self.mediator.connect ('item-selected', self.on_item_selected)
    self.mediator.connect ('item.attribute-modified', self.on_attribute_modified)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief set case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def set_case (self, case):
    self.case = case
    self.update_title ()
    self.case_treeview.set_case (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief update title
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def update_title (self):
    self.save_menuitem.set_sensitive (self.case.is_modified ())
    self.save_toolitem.set_sensitive (self.case.is_modified ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle stop event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_widget_stopped (self):

    # show 'save changes' dialog if necessary
    if self.case.is_modified ():
      dialog = gtk.MessageDialog (None,
                      gtk.DIALOG_MODAL,
                      gtk.MESSAGE_QUESTION,
                      gtk.BUTTONS_YES_NO,
                      "Save changes to '%s'?" % self.case.filename)
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return True

      elif rc == gtk.RESPONSE_YES:
        self.mediator.call ('case.save', self.case)

    self.mediator.call ('case.close', self.case)

    # clean-up code
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief case attribute modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_attribute_modified (self, case, attr, old_value, new_value):
    case.set_modified (True)
    self.update_title ()

    if attr == 'name' and case.uid == self.case.uid:
      self.case_treeview.set_case_name (new_value)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief case modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_modified (self, case):
    case.set_modified (True)
    self.update_title ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle item selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_item_selected (self, item):
    self.add_toolitem.set_sensitive (True)
    self.remove_toolitem.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_modified (self, item, id, old_text, new_text):
    item.case.set_modified (True)
    self.update_title ()

    name = item.get_attribute ('name')
    self.case_treeview.set_selected_item_name (name)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle case selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_case_selected (self, case):
    self.add_toolitem.set_sensitive (True)
    self.remove_toolitem.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_close (self, widget, *args):
    self.mediator.call ('ui.working-area.close', self.working_area.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_save (self, widget, *args):

    if self.case.is_new ():
      self.on_file_save_as (widget, *args)
    else:
      self.mediator.call ('case.save', self.case)
      self.update_title ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief save case as
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_save_as (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Save Case', action=gtk.FILE_CHOOSER_ACTION_SAVE)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)
    fs.set_do_overwrite_confirmation (True)

    filter = gtk.FileFilter ()
    filter.set_name ('Mobius case')
    filter.add_pattern ('*.case')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # save file
    root, ext = os.path.splitext (filename)
    if ext != '.case':
      filename += '.case'
    self.case.filename = filename

    self.mediator.call ('case.save', self.case)
    self.update_title ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief case properties
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_properties (self, widget, *args):
    dialog = CasePropertiesDialog (self.case, self.mediator)
    dialog.run ()
    dialog.destroy ()

    self.update_title ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call add item dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_add_item (self, widget, *args):
    dialog = AddItemDialog (self.mediator)
    rc = dialog.run ()

    amount = dialog.get_amount ()
    category = dialog.get_category ()
    dialog.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    for i in range (amount):
      item = self.case.create_item (category)
      self.case_treeview.add_child (item)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call remove item dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_remove_item (self, widget, *args):
    item = self.case_treeview.get_selected_item ()

    title = 'You are about to remove an item'
    if item.has_child ():
      title += ' and its sub-items'
    title += '. Are you sure?'
    
    dialog = gtk.MessageDialog (None,
                    gtk.DIALOG_MODAL,
                    gtk.MESSAGE_QUESTION,
                    gtk.BUTTONS_YES_NO,
                    title)
    rc = dialog.run ()
    dialog.destroy ()

    if rc != gtk.RESPONSE_YES:
      return

    self.case_treeview.remove_selected_item ()
    item.parent.remove_child (item)
    self.remove_toolitem.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: drag-begin
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_drag_begin (self, treeview, context, *args):
    treeselection = treeview.get_selection ()
    model, iter = treeselection.get_selected ()
    pixbuf = model.get_value (iter, ITEM_ICON)
    context.set_icon_pixbuf (pixbuf, 0, 0)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: drag-data-get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_drag_data_get (self, treeview, context, selection, info, timestamp):
    treeselection = treeview.get_selection ()
    model, iter = treeselection.get_selected ()

    if iter:
      item = model.get_value (iter, ITEM_OBJ)
      selection.set ('application/x-mobius-case-item', 8, str (item.uid))

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'case-viewer'
    self.name = 'Case Viewer'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Case viewer'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
     self.mediator.connect ('case-opened', self.on_open)
     self.mediator.connect ('case-created', self.on_open)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create window for open case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_open (self, case):
    widget = Widget (self.mediator)
    widget.set_case (case)

    working_area = self.mediator.call ('ui.working-area.new', self.id)
    working_area.set_default_size (600, 400)
    working_area.set_title (self.name)
    working_area.set_icon (self.icon_data)
    working_area.set_widget (widget)
    working_area.show ()
