/* =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
   Mobius Forensic Toolkit
   Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015 Eduardo Aguiar

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see <http://www.gnu.org/licenses/>.
   =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= */
#include <mobius/crypt/hash_crc32.h>

namespace mobius
{
namespace crypt
{
static const std::uint32_t CRCPOLY = 0xEDB88320;
static std::uint32_t CRCTAB[256];
static bool IS_INITIALIZED = false;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief build internal CRCTAB
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
_build_crctab ()
{
  std::uint32_t c;

  for (std::uint32_t i = 0;i < 256;i++)
    {
      c = i;
      for (std::uint8_t j = 0; j < 8; j++)
        {
          if (c & 1)
            c = (c >> 1) ^ CRCPOLY;
          else
            c = (c >> 1);
        }
      CRCTAB[i] = c;
    }

  IS_INITIALIZED = true;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief update CRC-32 by one byte
//!\param value CRC-32 value
//!\param b byte
//!\return new CRC-32 value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t crc32 (std::uint32_t value, std::uint8_t b)
{
  if (!IS_INITIALIZED)
    _build_crctab ();

  return (value >> 8) ^ CRCTAB [(value ^ b) & 0xff];
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief default constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
hash_crc32::hash_crc32 ()
 : value_ (0xffffffff)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief return hash digest
//!\return hash digest
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const bytearray
hash_crc32::get_digest () const
{
  std::uint32_t v = value_ ^ 0xffffffff;
  mobius::bytearray d (4);
  d[0] = (v >> 24); d[1] = (v >> 16) & 0xff; d[2] = (v >> 8) & 0xff; d[3] = v & 0xff;

  return d;  
}

} // crypt namespace
} // mobius namespace