// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/imagefile/imagefile_impl_solo.h>
#include <mobius/io/file.h>
#include <mobius/charset.h>
#include <mobius/exception.inc>
#include <mobius/string_functions.h>
#include <mobius/regex_impl.h>
#include <stdexcept>
#include <cctype>
#include <algorithm>

namespace mobius
{
namespace imagefile
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get URL for first segment file
//! \param url imagefile URL
//! \return first segment file URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const std::string
get_first_segment_url (const std::string& url)
{
  return url.substr (0, url.length () - 4) + ".001";
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if URL is an instance of imagefile solo
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
imagefile_impl_solo::is_instance (const std::string& url)
{
  bool instance = false;

  mobius::io::file f (url);

  if (f && f.exists ())
    {
      auto reader = f.new_reader ();
      mobius::bytearray data = reader.read (14);

      instance = data == "[SEIZE HEADER]";
    }

  return instance;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief construct object
//! \param url imagefile URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
imagefile_impl_solo::imagefile_impl_solo (const std::string& url)
  : imagefile_impl_base (url), split_imagefile_ (get_first_segment_url (url))
{
  mobius::io::file f (url);
  constexpr mobius::io::file::size_type LOG_MAX_SIZE = 1048576;  // 1 MiB

  if (f.exists () && f.get_size () < LOG_MAX_SIZE)
    {
      mobius::regex REGEX_SEIZE_TIME ("^Seize Time = *([0-9]+):([0-9]+):([0-9]+)$", REG_NEWLINE | REG_EXTENDED);
      mobius::regex REGEX_SEIZE_DATE ("^Seize Date = *([0-9]+)/([0-9]+)/([0-9]+)$", REG_NEWLINE | REG_EXTENDED);      
      mobius::regex REGEX_TOTAL_SECTORS ("^Seize Size = ([0-9]+)$", REG_NEWLINE | REG_EXTENDED);
      mobius::regex REGEX_MD5 ("^MD5: (.+)$", REG_NEWLINE | REG_EXTENDED);
      mobius::regex REGEX_SUSPECT ("Suspect: Model: ([^.]+?)\\. Serial Number: ([^.]+?)\\..*Block size: ([0-9]+)\\..*\\. Evidence");
      mobius::regex REGEX_SOLO_PRODUCT ("^Product = (.+)$", REG_NEWLINE | REG_EXTENDED);
      mobius::regex REGEX_SOLO_SERIAL ("^Serial #: (.+)$", REG_NEWLINE | REG_EXTENDED);
      mobius::regex REGEX_SOLO_SOFTWARE ("^Software Version (.+)$", REG_NEWLINE | REG_EXTENDED);

      // parse .txt file
      auto reader = f.new_reader ();
      mobius::bytearray data = reader.read (f.get_size ());
      const std::string text = conv_charset_to_utf8 (data, "ASCII");
      
      if (REGEX_SUSPECT.match (text))
        {
          set_drive_model (REGEX_SUSPECT[1]);
          set_drive_serial_number (REGEX_SUSPECT[2]);
          set_sector_size (stoi (REGEX_SUSPECT[3]));
        }

      if (REGEX_SEIZE_DATE.match (text) && REGEX_SEIZE_TIME.match (text))
        {       
          mobius::datetime::date date = mobius::datetime::date (
                  stoi (REGEX_SEIZE_DATE[3]),
                  stoi (REGEX_SEIZE_DATE[1]),
                  stoi (REGEX_SEIZE_DATE[2]));

          mobius::datetime::time time = mobius::datetime::time (
                  stoi (REGEX_SEIZE_TIME[1]),
                  stoi (REGEX_SEIZE_TIME[2]),
                  stoi (REGEX_SEIZE_TIME[3]));

          set_acquisition_time (mobius::datetime::datetime (date, time));
        }

      if (REGEX_TOTAL_SECTORS.match (text))
        {
          set_sectors (stoll (REGEX_TOTAL_SECTORS[1]));
          set_size (get_sectors () * get_sector_size ());
        }

      if (REGEX_MD5.match (text))
        {
          std::string value = string_remove_all (REGEX_MD5[1], ' ');
          set_hash_md5 (string_tolower (value));
        }

      if (REGEX_SOLO_PRODUCT.match (text) && REGEX_SOLO_SERIAL.match (text))
        set_acquisition_platform (REGEX_SOLO_PRODUCT[1] + " (s/n: " + REGEX_SOLO_SERIAL[1] + ')');

      if (REGEX_SOLO_SOFTWARE.match (text))
        set_acquisition_tool ("Solo software v" + REGEX_SOLO_SOFTWARE[1]);

      set_segments (split_imagefile_.get_segments ());
      set_segment_size (split_imagefile_.get_segment_size ());
      set_acquisition_user (f.get_user_name ());
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new reader for imagefile
//! \return reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::io::reader
imagefile_impl_solo::new_reader () const
{
  return split_imagefile_.new_reader ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new writer for imagefile
//! \return writer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::io::writer
imagefile_impl_solo::new_writer () const
{
  throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("writer not implemented"));
}

} // namespace imagefile
} // namespace mobius
