// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/tsk/filesystem.h>
#include <mobius/tsk/diskimage.h>
#include <mobius/tsk/entry.h>
#include <mobius/tsk/exception.h>
#include <tsk/libtsk.h>
#include <mobius/tsk/exception.inc>

namespace mobius
{
namespace tsk
{

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief constructor
//!\param image diskimage object
//!\param offset offset in bytes from the start of the diskimage
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
filesystem::filesystem (const diskimage& image, offset_t offset)
{
  fs_info_p_ = tsk_fs_open_img (image.get_pointer (), offset, TSK_FS_TYPE_DETECT);
  if (!fs_info_p_)
    throw MOBIUS_TSK_EXCEPTION;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief destructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
filesystem::~filesystem ()
{
  if (fs_info_p_)
    {
      tsk_fs_close (fs_info_p_);
      fs_info_p_ = 0;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get filesystem offset
//!\return offset in bytes from the start of the diskimage
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
offset_t
filesystem::get_offset () const
{
  return fs_info_p_->offset;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get inodes count
//!\return number of inodes of the filesystem
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inode_t
filesystem::get_inode_count () const
{
  return fs_info_p_->inum_count;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get root inode
//!\return inode number for the root entry
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inode_t
filesystem::get_root_inode () const
{
  return fs_info_p_->root_inum;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get first inode
//!\return number of the first inode
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inode_t
filesystem::get_first_inode () const
{
  return fs_info_p_->first_inum;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get last inode
//!\return number of the last inode
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inode_t
filesystem::get_last_inode () const
{
  return fs_info_p_->last_inum;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get root entry
//!\return the root entry of the filesystem
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const entry
filesystem::get_root_entry () const
{
  TSK_FS_FILE *file_p = tsk_fs_file_open_meta (fs_info_p_, 0, fs_info_p_->root_inum);
  if (!file_p)
    throw MOBIUS_TSK_EXCEPTION;

  return entry (file_p);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get filesystem ID
//!\return filesystem ID as a hexadecimal string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
filesystem::get_id () const
{
  std::string id;
  char text[3];

  for (std::size_t i = 0; i < fs_info_p_->fs_id_used; i++)
    {
      sprintf (text, "%02X", (unsigned char) fs_info_p_->fs_id[i]);
      id = text + id; //54D08DB6D08D9EBC
    }

  return id;
}

} // namespace tsk
} // namespace mobius
