// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include "api_datetime.h"
#include "api_imagefile.h"
#include "api_reader.h"
#include "api_writer.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void api_imagefile_tp_dealloc (api_imagefile_o *);
static PyObject *api_imagefile_tp_new (PyTypeObject *, PyObject *, PyObject *);
static PyObject *api_imagefile_get_is_available (api_imagefile_o *);
static PyObject *api_imagefile_get_is_valid (api_imagefile_o *);
static PyObject *api_imagefile_get_url (api_imagefile_o *);
static PyObject *api_imagefile_get_type (api_imagefile_o *);
static PyObject *api_imagefile_get_size (api_imagefile_o *);
static int api_imagefile_set_size (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_sectors (api_imagefile_o *);
static int api_imagefile_set_sectors (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_sector_size (api_imagefile_o *);
static int api_imagefile_set_sector_size (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_segments (api_imagefile_o *);
static int api_imagefile_set_segments (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_segment_size (api_imagefile_o *);
static int api_imagefile_set_segment_size (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_drive_vendor (api_imagefile_o *);
static int api_imagefile_set_drive_vendor (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_drive_model (api_imagefile_o *);
static int api_imagefile_set_drive_model (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_drive_serial_number (api_imagefile_o *);
static int api_imagefile_set_drive_serial_number (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_acquisition_user (api_imagefile_o *);
static int api_imagefile_set_acquisition_user (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_acquisition_time (api_imagefile_o *);
static int api_imagefile_set_acquisition_time (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_acquisition_tool (api_imagefile_o *);
static int api_imagefile_set_acquisition_tool (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_acquisition_platform (api_imagefile_o *);
static int api_imagefile_set_acquisition_platform (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_get_hash_md5 (api_imagefile_o *);
static int api_imagefile_set_hash_md5 (api_imagefile_o *, PyObject *, void *);
static PyObject *api_imagefile_f_new_reader (api_imagefile_o *, PyObject *);
static PyObject *api_imagefile_f_new_writer (api_imagefile_o *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef api_imagefile_getsetters[] =
{
  {
    (char *) "is_available",
    (getter) api_imagefile_get_is_available,
    (setter) 0,
    (char *) "check if imagefile is available", NULL
  },
  {
    (char *) "is_valid",
    (getter) api_imagefile_get_is_valid,
    (setter) 0,
    (char *) "check if imagefile is valid", NULL
  },
  {
    (char *) "url",
    (getter) api_imagefile_get_url,
    (setter) 0,
    (char *) "imagefile URL", NULL
  },
  {
    (char *) "type",
    (getter) api_imagefile_get_type,
    (setter) 0,
    (char *) "imagefile type", NULL
  },
  {
    (char *) "size",
    (getter) api_imagefile_get_size,
    (setter) api_imagefile_set_size,
    (char *) "imagefile size in bytes", NULL
  },
  {
    (char *) "sectors",
    (getter) api_imagefile_get_sectors,
    (setter) api_imagefile_set_sectors,
    (char *) "number of sectors", NULL
  },
  {
    (char *) "sector_size",
    (getter) api_imagefile_get_sector_size,
    (setter) api_imagefile_set_sector_size,
    (char *) "sector size in bytes", NULL
  },
  {
    (char *) "segments",
    (getter) api_imagefile_get_segments,
    (setter) api_imagefile_set_segments,
    (char *) "number of file segments", NULL
  },
  {
    (char *) "segment_size",
    (getter) api_imagefile_get_segment_size,
    (setter) api_imagefile_set_segment_size,
    (char *) "segment size in bytes", NULL
  },
  {
    (char *) "drive_vendor",
    (getter) api_imagefile_get_drive_vendor,
    (setter) api_imagefile_set_drive_vendor,
    (char *) "drive vendor name", NULL
  },
  {
    (char *) "drive_model",
    (getter) api_imagefile_get_drive_model,
    (setter) api_imagefile_set_drive_model,
    (char *) "drive model", NULL
  },
  {
    (char *) "drive_serial_number",
    (getter) api_imagefile_get_drive_serial_number,
    (setter) api_imagefile_set_drive_serial_number,
    (char *) "drive serial number", NULL
  },
  {
    (char *) "acquisition_user",
    (getter) api_imagefile_get_acquisition_user,
    (setter) api_imagefile_set_acquisition_user,
    (char *) "acquisition user name", NULL
  },
  {
    (char *) "acquisition_time",
    (getter) api_imagefile_get_acquisition_time,
    (setter) api_imagefile_set_acquisition_time,
    (char *) "acquisition date and time", NULL
  },
  {
    (char *) "acquisition_tool",
    (getter) api_imagefile_get_acquisition_tool,
    (setter) api_imagefile_set_acquisition_tool,
    (char *) "acquisition tool", NULL
  },
  {
    (char *) "acquisition_platform",
    (getter) api_imagefile_get_acquisition_platform,
    (setter) api_imagefile_set_acquisition_platform,
    (char *) "acquisition platform", NULL
  },
  {
    (char *) "hash_md5",
    (getter) api_imagefile_get_hash_md5,
    (setter) api_imagefile_set_hash_md5,
    (char *) "MD5 hash", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef api_imagefile_methods[] =
{
  {
    (char *) "new_reader",
    (PyCFunction) api_imagefile_f_new_reader,
    METH_VARARGS,
    "create new reader"
  },
  {
    (char *) "new_writer",
    (PyCFunction) api_imagefile_f_new_writer,
    METH_VARARGS,
    "create new writer"
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject api_imagefile_t =
{
  PyObject_HEAD_INIT (0)
  0,                                          		// ob_size
  "api.imagefile",                      		// tp_name
  sizeof (api_imagefile_o),             		// tp_basicsize
  0,                                          		// tp_itemsize
  (destructor) api_imagefile_tp_dealloc,		// tp_dealloc
  0,                                          		// tp_print
  0,                                          		// tp_getattr
  0,                                          		// tp_setattr
  0,                                          		// tp_compare
  0,                                          		// tp_repr
  0,                                          		// tp_as_number
  0,                                          		// tp_as_sequence
  0,                                          		// tp_as_mapping
  0,                                          		// tp_hash
  0,                                          		// tp_call
  0,                                          		// tp_str
  0,                                          		// tp_getattro
  0,                                          		// tp_setattro
  0,                                          		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,   		// tp_flags
  "imagefile class",                          		// tp_doc
  0,                                          		// tp_traverse
  0,                                          		// tp_clear
  0,                                          		// tp_richcompare
  0,                                          		// tp_weaklistoffset
  0,                                          		// tp_iter
  0,                                          		// tp_iternext
  api_imagefile_methods,                		// tp_methods
  0,                                          		// tp_members
  api_imagefile_getsetters,             		// tp_getset
  0,                                          		// tp_base
  0,                                          		// tp_dict
  0,                                          		// tp_descr_get
  0,                                          		// tp_descr_set
  0,                                          		// tp_dictoffset
  0,                                          		// tp_init
  0,                                          		// tp_alloc
  api_imagefile_tp_new                  		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: tp_new
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_tp_new (PyTypeObject *type, PyObject *args, PyObject *kwds)
{
  const char *arg_url = nullptr;
  const char *arg_type = "autodetect";

  if (!PyArg_ParseTuple (args, "|ss", &arg_url, &arg_type))
    return nullptr;

  api_imagefile_o *self = (api_imagefile_o *) type->tp_alloc (type, 0);
  if (self != nullptr)
    {
      try
        {
          if (arg_url)
            self->obj = new mobius::imagefile::imagefile (arg_url, arg_type);
          else
            self->obj = new mobius::imagefile::imagefile ();
        }
      catch (const std::runtime_error& e)
        {
          Py_DECREF (self);
          PyErr_SetString (PyExc_IOError, e.what ());
          self = nullptr;
        }
    }

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
api_imagefile_tp_dealloc (api_imagefile_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: is_available getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_is_available (api_imagefile_o *self)
{
  return PyBool_FromLong (self->obj->is_available ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: is_valid getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_is_valid (api_imagefile_o *self)
{
  return PyBool_FromLong (self->obj->operator bool ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: url getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_url (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_url ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: type getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_type (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_type ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_size (api_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_size (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'size' attribute");
      return -1;
    }

  // set size
  std::uint64_t arg_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_size (arg_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: sectors getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_sectors (api_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_sectors ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: sectors setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_sectors (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'sectors' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'sectors' attribute");
      return -1;
    }

  // set sectors
  std::uint64_t arg_sectors = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_sectors (arg_sectors);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: sector_size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_sector_size (api_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_sector_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: sector_size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_sector_size (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'sector_size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'sector_size' attribute");
      return -1;
    }

  // set sector_size
  std::uint64_t arg_sector_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_sector_size (arg_sector_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: segments getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_segments (api_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_segments ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: segments setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_segments (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'segments' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'segments' attribute");
      return -1;
    }

  // set segments
  std::uint64_t arg_segments = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_segments (arg_segments);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: segment_size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_segment_size (api_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_segment_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: segment_size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_segment_size (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'segment_size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'segment_size' attribute");
      return -1;
    }

  // set segment_size
  std::uint64_t arg_segment_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_segment_size (arg_segment_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: drive_vendor getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_drive_vendor (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_vendor ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: drive_vendor setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_drive_vendor (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_vendor' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_vendor' attribute");
      return -1;
    }

  // set drive_model
  const char *arg_drive_vendor = PyString_AsString (value);
  self->obj->set_drive_vendor (arg_drive_vendor);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: drive_model getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_drive_model (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_model ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: drive_model setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_drive_model (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_model' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_model' attribute");
      return -1;
    }

  // set drive_model
  const char *arg_drive_model = PyString_AsString (value);
  self->obj->set_drive_model (arg_drive_model);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: drive_serial_number getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_drive_serial_number (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_serial_number ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: drive_serial_number setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_drive_serial_number (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_serial_number' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_serial_number' attribute");
      return -1;
    }

  // set drive_serial_number
  const char *arg_drive_serial_number = PyString_AsString (value);
  self->obj->set_drive_serial_number (arg_drive_serial_number);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: acquisition_user getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_acquisition_user (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_user ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: acquisition_user setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_acquisition_user (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_user' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_user' attribute");
      return -1;
    }

  // set acquisition_user
  const char *arg_acquisition_user = PyString_AsString (value);
  self->obj->set_acquisition_user (arg_acquisition_user);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: acquisition_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_acquisition_time (api_imagefile_o *self)
{
  return PyDateTime_from_datetime (self->obj->get_acquisition_time ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: acquisition_time setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_acquisition_time (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_time' attribute");
      return -1;
    }

  if (!check_PyDateTime (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_time' attribute");
      return -1;
    }

  // set acquisition_time
  self->obj->set_acquisition_time (datetime_from_PyDateTime (value));

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: acquisition_tool getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_acquisition_tool (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_tool ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: acquisition_tool setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_acquisition_tool (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_tool' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_tool' attribute");
      return -1;
    }

  // set acquisition_tool
  const char *arg_acquisition_tool = PyString_AsString (value);
  self->obj->set_acquisition_tool (arg_acquisition_tool);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: acquisition_platform getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_acquisition_platform (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_platform ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: acquisition_platform setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_acquisition_platform (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_platform' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_platform' attribute");
      return -1;
    }

  // set acquisition_platform
  const char *arg_acquisition_platform = PyString_AsString (value);
  self->obj->set_acquisition_platform (arg_acquisition_platform);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: hash_md5 getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_get_hash_md5 (api_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_hash_md5 ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: hash_md5 setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
api_imagefile_set_hash_md5 (api_imagefile_o *self, PyObject *value, void *closure)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'hash_md5' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'hash_md5' attribute");
      return -1;
    }

  // set hash_md5
  const char *arg_hash_md5 = PyString_AsString (value);
  self->obj->set_hash_md5 (arg_hash_md5);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: new_reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_f_new_reader (api_imagefile_o *self, PyObject *args)
{
  api_reader_o *ret = nullptr;

  try
    {
      ret = (api_reader_o *) _PyObject_New (&api_reader_t);

      if (ret != nullptr)
        ret->obj = new mobius::io::reader (self->obj->new_reader ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
    }

  return (PyObject *) ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.imagefile: new_writer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_imagefile_f_new_writer (api_imagefile_o *self, PyObject *args)
{
  api_writer_o *ret = nullptr;

  try
    {
      ret = (api_writer_o *) _PyObject_New (&api_writer_t);

      if (ret != nullptr)
        ret->obj = new mobius::io::writer (self->obj->new_writer ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
    }

  return (PyObject *) ret;
}
