// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "folder.h"
#include "file.h"
#include "folder_impl_null.h"
#include "local/folder_impl.h"
#include "local/get_current_folder.h"
#include "uri.h"
#include <mobius/exception.inc>
#include <stdexcept>
#include <config.h>

#ifdef HAVE_SMB_SUPPORT
#include "smb/folder_impl.h"
#endif

namespace mobius
{
namespace io
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Construct object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
folder::folder ()
{
  impl_ = std::make_shared <folder_impl_null> ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Construct object
//! \param impl Implementation object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
folder::folder (std::shared_ptr <folder_impl_base> impl)
  : impl_ (impl)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new file object
//! \return File object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
file
folder::new_file (const std::string& name) const
{
  return file (impl_->new_file (name));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get folder extension
//! \return Folder extension
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
folder::get_extension () const
{
  std::string extension;
  const std::string name = get_name ();
  auto pos = name.rfind ('.');

  if (pos != std::string::npos && pos != 0)
    extension = name.substr (pos + 1);
  
  return extension;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Copy folder
//! \param dst Folder object (destination)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
folder::copy (folder dst) const
{
  dst.create ();
  
  for (const auto& entry : get_children ())
    {
      if (entry.is_file ())
        {
          auto src_file = entry.get_file ();
          auto dst_file = dst.new_file (src_file.get_name ());
          src_file.copy (dst_file);
        }

      else if (entry.is_folder ())
        {
          auto src_folder = entry.get_folder ();
          auto dst_folder = dst.new_folder (src_folder.get_name ());
          src_folder.copy (dst_folder);
        }

      else
        throw std::invalid_argument (MOBIUS_EXCEPTION_MSG ("unhandled entry"));
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Move folder
//! \param dst Destination folder
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
folder::move (folder dst)
{
  if (!impl_->move (dst.impl_))
    {
      copy (dst);
      remove ();
    }

  impl_->reload ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create folder by path
//! \param path folder path
//! \return Folder object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
folder
new_folder_by_path (const std::string& path)
{
  return folder (std::make_shared <local::folder_impl> (path));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create folder by URL
//! \param url folder URL
//! \return Folder object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
folder
new_folder_by_url (const std::string& url)
{
  mobius::io::uri u (url);

  if (u.get_scheme () == "file")
    return new_folder_by_path (u.get_path ());

#ifdef HAVE_SMB_SUPPORT
  else if (u.get_scheme () == "smb")
    return folder (std::make_shared <smb::folder_impl> (url));
#endif

  else
    throw std::invalid_argument (MOBIUS_EXCEPTION_MSG ("unknown folder scheme"));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get current folder
//! \return Folder object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
folder
get_current_folder ()
{
  return local::get_current_folder ();
}

} // namespace io
} // namespace mobius
