#ifndef MOBIUS_MODEL_ITEM_H
#define MOBIUS_MODEL_ITEM_H

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <cstdint>
#include <map>
#include <memory>
#include <string>
#include <vector>
#include <mobius/bytearray.h>
#include <mobius/database/database.h>
#include <mobius/pod/data.h>
#include <mobius/pod/list.h>

namespace mobius
{
namespace model
{
class ant;
class application;
class Case;
class evidence;

class call;
class chat_message;
class cookie;
class password;
class password_hash;
class profile;
class text_autocomplete;
class text_search;
class bookmarked_url;
class visited_url;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief item class
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class item
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Datatypes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  using uid_type = std::int64_t;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Constructors
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  item () noexcept = default;
  item (const mobius::model::Case&, uid_type);
  item (item&&) noexcept = default;
  item (const item&) noexcept = default;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Operators
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  item& operator= (const item&) noexcept = default;
  item& operator= (item&&) noexcept = default;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Function prototypes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  uid_type get_uid () const;
  std::string get_category () const;
  Case get_case () const;

  int get_child_count () const;
  std::vector <item> get_children () const;
  item get_parent () const;
  item new_child (const std::string&, int = -1);
  void remove ();
  void move (int, const item&);

  bool has_attribute (const std::string&) const;
  std::string get_attribute (const std::string&) const;
  void set_attribute (const std::string&, const std::string&);
  void remove_attribute (const std::string&);
  std::map <std::string, std::string> get_attributes () const;
  void expand_masks ();

  mobius::pod::data get_metadata () const;
  void set_metadata (const mobius::pod::data&);

  void set_ant (const std::string&, const std::string&, const std::string&);
  void reset_ant (const std::string&);
  bool has_ant (const std::string&) const;
  void remove_ants ();
  std::vector <ant> get_ants () const;

  std::string get_data_path (const std::string&) const;
  std::string create_data_path (const std::string&) const;

  password new_password (const std::string&, const std::string&, const std::string&);
  std::vector <password> get_passwords () const;
  void remove_passwords ();
  std::int64_t count_passwords () const;

  password_hash new_password_hash (const std::string&, const std::string&, const std::string&);
  std::vector <password_hash> get_password_hashes () const;
  void remove_password_hashes ();
  std::int64_t count_password_hashes () const;

  profile new_profile (const std::string&, const std::string&);
  std::vector <profile> get_profiles () const;
  void remove_profiles_by_app_id (const std::string&);

  bookmarked_url new_bookmarked_url (const std::string&);
  std::vector <bookmarked_url> get_bookmarked_urls () const;
  void remove_bookmarked_urls ();
  std::int64_t count_bookmarked_urls () const;

  call new_call (const mobius::datetime::datetime&, const std::string&, const mobius::pod::list&, std::int64_t);
  std::vector <call> get_calls () const;
  void remove_calls ();
  std::int64_t count_calls () const;

  chat_message new_chat_message (const mobius::datetime::datetime&, const std::string&, const mobius::pod::list&, const mobius::pod::list&);
  std::vector <chat_message> get_chat_messages () const;
  void remove_chat_messages ();
  std::int64_t count_chat_messages () const;

  cookie new_cookie (const std::string&, const mobius::bytearray&, bool);
  std::vector <cookie> get_cookies () const;
  void remove_cookies ();
  std::int64_t count_cookies () const;

  text_autocomplete new_text_autocomplete (const std::string&, const std::string&);
  std::vector <text_autocomplete> get_text_autocompletes () const;
  void remove_text_autocompletes ();
  std::int64_t count_text_autocompletes () const;

  text_search new_text_search (const mobius::datetime::datetime&, const std::string&, const std::string&);
  std::vector <text_search> get_text_searches () const;
  void remove_text_searches ();
  std::int64_t count_text_searches () const;

  visited_url new_visited_url (const mobius::datetime::datetime&, const std::string&);
  std::vector <visited_url> get_visited_urls () const;
  void remove_visited_urls ();
  std::int64_t count_visited_urls () const;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Check if object is valid
  //! \return true/false
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  explicit operator bool () const noexcept
  {
    return bool (impl_);
  }

private:
  //! \brief implementation class forward declaration
  class impl;

  //! \brief implementation pointer
  std::shared_ptr <impl> impl_;
  
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Helper functions
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::database::database get_database () const;
  evidence _new_evidence (const std::string&);
  void _remove_evidences (const std::string&);
  std::int64_t _count_evidences (const std::string&) const;
  
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get evidences for a given type
  //! \param type Evidence type
  //! \return Vector of evidences of that type
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  template <typename T>
  std::vector <T> _get_evidences (const std::string& type) const
  {
    auto db = get_database ();

    auto stmt = db.new_statement (
                  "SELECT uid "
                    "FROM evidence "
                   "WHERE item_uid = ? "
                     "AND type = ?");

    stmt.bind (1, get_uid ());
    stmt.bind (2, type);

    std::vector <T> evidences;

    while (stmt.fetch_row ())
      {
        auto uid = stmt.get_column_int64 (0);
        evidences.emplace_back (T (evidence (*this, uid, type)));
      }

    return evidences;
  }
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Functions
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool operator== (const item&, const item&);
bool operator!= (const item&, const item&);

} // namespace model
} // namespace mobius

#endif
