// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief  C++ API mobius.ui module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include "ui.h"
#include "message_dialog.h"
#include <map>
#include <string>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Functions prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject* func_get_implementations (PyObject *, PyObject *);
PyObject* func_set_implementation (PyObject *, PyObject *);

namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Module constants
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const std::map <std::string, int> CONSTANTS_ =
{
  {"MESSAGE_DIALOG_TYPE_INFO", 1},
  {"MESSAGE_DIALOG_TYPE_WARNING", 2},
  {"MESSAGE_DIALOG_TYPE_QUESTION", 3},
  {"MESSAGE_DIALOG_TYPE_ERROR", 4},
  {"BUTTON_OK", 1},
  {"BUTTON_CANCEL", 2},
  {"BUTTON_YES", 3},
  {"BUTTON_NO", 4},
  {"BUTTON_OPEN", 5},
  {"BUTTON_SAVE", 6},
  {"BUTTON_CLOSE", 7},
  {"BUTTON_DISCARD", 8},
  {"BUTTON_APPLY", 9},
  {"BUTTON_RESET", 10},
  {"BUTTON_RESTORE_DEFAULTS", 11},
  {"BUTTON_HELP", 12},
  {"BUTTON_SAVE_ALL", 13},
  {"BUTTON_YES_TO_ALL", 14},
  {"BUTTON_NO_TO_ALL", 15},
  {"BUTTON_ABORT", 16},
  {"BUTTON_RETRY", 17},
  {"BUTTON_IGNORE", 18},
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Module methods
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef module_methods[] =
{
  {
    "get_implementations",
    func_get_implementations,
    METH_VARARGS,
    "Get UI implementations available"
  },
  {
    "set_implementation",
    func_set_implementation,
    METH_VARARGS,
    "Set UI implementation"
  },
  {
    nullptr,
    nullptr,
    0,
    nullptr
  } // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief module datatypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr struct
{
  const char *name;
  PyTypeObject *type;
} types[] =

{
  {"ui", &ui_ui_t},
  {"message_dialog", &ui_message_dialog_t},
};

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create mobius.ui module
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
new_ui_module ()
{
  // initialize module mobius.ui
  PyObject* module = Py_InitModule3 (
                       "mobius.ui",
                       module_methods,
                       "Mobius Forensic Toolkit UI module"
                     );

  // build types
  for (const auto& t : types)
    {
      if (PyType_Ready (t.type) < 0)
        return nullptr;
    }

  // add types
  for (const auto& t : types)
    {
      Py_INCREF (t.type);
      PyModule_AddObject (module, t.name, (PyObject *) t.type);
    }

  // add module constants
  for (auto& p : CONSTANTS_)
    PyModule_AddIntConstant (module, p.first.c_str (), p.second);

  // return module
  return module;
}
