# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import traceback
import pymobius.app.chromium
import pymobius.app.edge
import pymobius.app.gecko
import pymobius.app.internet_explorer
import mobius

ANT_ID = 'cookies'
ANT_NAME = 'Cookies'
ANT_VERSION = '1.3'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic cookie class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Cookie (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.is_encrypted = False
    self.is_deleted = False
    self.last_access_time = None
    self.expiration_time = None

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ant: Cookies
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.id = ANT_ID
    self.name = ANT_NAME
    self.version = ANT_VERSION
    self.__item = item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Run ant
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def run (self):
    if not self.__item.has_vfs ():
      return

    # retrieve evidence
    self.__entries = []

    self.__retrieve_chromium ()
    self.__retrieve_edge ()
    self.__retrieve_gecko ()
    self.__retrieve_internet_explorer ()

    self.__save_data ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Chromium based browsers
  # @todo Move Unicode convertion to profile class
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_chromium (self):
    try:
      model = pymobius.app.chromium.model (self.__item)

      for profile in model.get_profiles ():
        self.__retrieve_chromium_profile (profile)
    except Exception as e:
      mobius.core.logf ('WRN %s %s' % (str (e), traceback.format_exc ()))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Chromium profile
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_chromium_profile (self, profile):
    for entry in profile.get_cookies ():
      cookie = Cookie ()
      cookie.name = entry.name
      cookie.value = entry.value
      cookie.domain = entry.domain
      cookie.creation_time = entry.creation_time
      cookie.last_access_time = entry.last_access_time
      cookie.evidence_path = entry.evidence_path
      cookie.is_deleted = entry.is_deleted
      cookie.is_encrypted = entry.is_encrypted
      cookie.profile = profile
      self.__entries.append (cookie)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Microsoft Edge
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_edge (self):
    try:
      model = pymobius.app.edge.model (self.__item)

      for profile in model.get_profiles ():
        self.__retrieve_edge_profile (profile)
    except Exception as e:
      mobius.core.logf ('WRN %s %s' % (str (e), traceback.format_exc ()))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Microsoft Edge profile
  # @todo Move Unicode convertion to profile class
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_edge_profile (self, profile):
    for entry in profile.get_cookies ():
      cookie = Cookie ()
      cookie.name = entry.name
      cookie.value = entry.value
      cookie.domain = entry.domain
      cookie.creation_time = entry.creation_time
      cookie.last_access_time = entry.last_access_time
      cookie.expiration_time = entry.expiration_time
      cookie.evidence_path = entry.evidence_path
      cookie.is_deleted = entry.is_deleted
      cookie.profile = profile
      self.__entries.append (cookie)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Gecko based browsers
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_gecko (self):
    try:
      model = pymobius.app.gecko.model (self.__item)

      for profile in model.get_profiles ():
        self.__retrieve_gecko_profile (profile)
    except Exception as e:
      mobius.core.logf ('WRN %s %s' % (str (e), traceback.format_exc ()))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Gecko profile
  # @see http://doxygen.db48x.net/mozilla/html/interfacensIDownloadManager.html
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_gecko_profile (self, profile):
    for entry in profile.get_cookies ():
      cookie = Cookie ()
      cookie.name = entry.name
      cookie.value = entry.value
      cookie.domain = entry.domain
      cookie.creation_time = entry.creation_time
      cookie.last_access_time = entry.last_access_time
      cookie.expiration_time = entry.expiration_time
      cookie.evidence_path = entry.evidence_path
      cookie.is_deleted = entry.is_deleted
      cookie.profile = profile
      self.__entries.append (cookie)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Internet Explorer
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_internet_explorer (self):
    try:
      model = pymobius.app.internet_explorer.model (self.__item)

      for profile in model.get_profiles ():
        self.__retrieve_internet_explorer_profile (profile)
    except Exception as e:
      mobius.core.logf ('WRN %s %s' % (str (e), traceback.format_exc ()))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Internet Explorer profile
  # @todo Move Unicode convertion to profile class
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_internet_explorer_profile (self, profile):
    for entry in profile.get_cookies ():
      cookie = Cookie ()
      cookie.name = entry.name
      cookie.value = entry.value
      cookie.domain = entry.domain
      cookie.creation_time = entry.creation_time
      cookie.last_access_time = entry.last_access_time
      cookie.expiration_time = entry.expiration_time
      cookie.evidence_path = entry.evidence_path
      cookie.is_deleted = entry.is_deleted
      cookie.profile = profile
      self.__entries.append (cookie)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save data into model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __save_data (self):
    case = self.__item.case
    transaction = case.new_transaction ()

    # remove old data
    self.__item.remove_cookies ()

    # save cookies
    profiles = {}

    for c in self.__entries:
      cookie = self.__item.new_cookie (c.name, c.value, c.is_encrypted)
      cookie.domain = c.domain
      cookie.creation_time = c.creation_time
      cookie.last_access_time = c.last_access_time
      cookie.expiration_time = c.expiration_time
      cookie.evidence_path = c.evidence_path.replace ('/', '\\')

      if c.is_deleted != None:
        cookie.is_deleted = c.is_deleted

      # create profile, if necessary
      p = profiles.get (id (cookie.profile))
      if not p:
        app = case.new_application (c.profile.app_id, c.profile.app_name)

        p = self.__item.new_profile (c.profile.app_id, c.profile.path)
        p.id = c.profile.name
        p.username = c.profile.username

        if c.profile.creation_time:
          p.creation_time = c.profile.creation_time
        profiles[id (c.profile)] = p

      cookie.profile = p

    # set ant run
    self.__item.set_ant (ANT_ID, ANT_NAME, ANT_VERSION)
    transaction.commit ()
