// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \file C++ API <i>mobius.crypt</i> module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include "cipher_aes.h"
#include "cipher_des.h"
#include "cipher_rc4.h"
#include "cipher_rot13.h"
#include "cipher_zip.h"
#include "hash.h"
#include "hmac.h"
#include "api.h"
#include <mobius/crypt/crypt.h>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate_hash_nt function
//! \param self function object
//! \param args argument list
//! \return hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
func_generate_hash_nt (PyObject *self, PyObject *args)
{
  // parse input args
  const char * arg_password;

  if (!PyArg_ParseTuple (args, "s", &arg_password))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_from_bytearray (mobius::crypt::generate_hash_nt (arg_password));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate_hash_lm function
//! \param self function object
//! \param args argument list
//! \return hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
func_generate_hash_lm (PyObject *self, PyObject *args)
{
  // parse input args
  const char * arg_password;

  if (!PyArg_ParseTuple (args, "s", &arg_password))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_from_bytearray (mobius::crypt::generate_hash_lm (arg_password));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate_hash_msdcc1 function
//! \param self function object
//! \param args argument list
//! \return hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
func_generate_hash_msdcc1 (PyObject *self, PyObject *args)
{
  // parse input args
  const char * arg_password;
  const char * arg_username;

  if (!PyArg_ParseTuple (args, "ss", &arg_password, &arg_username))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_from_bytearray (mobius::crypt::generate_hash_msdcc1 (arg_password, arg_username));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate_hash_msdcc2 function
//! \param self function object
//! \param args argument list
//! \return hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
func_generate_hash_msdcc2 (PyObject *self, PyObject *args)
{
  // parse input args
  const char * arg_password;
  const char * arg_username;
  int arg_iterations = 10240;

  if (!PyArg_ParseTuple (args, "ss|i", &arg_password, &arg_username, &arg_iterations))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_from_bytearray (mobius::crypt::generate_hash_msdcc2 (arg_password, arg_username, arg_iterations));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief module methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef module_methods[] =
{
  {
    "generate_hash_nt",
    func_generate_hash_nt,
    METH_VARARGS,
    "generate NT hash"
  },
  {
    "generate_hash_lm",
    func_generate_hash_lm,
    METH_VARARGS,
    "generate LM hash"
  },
  {
    "generate_hash_msdcc1",
    func_generate_hash_msdcc1,
    METH_VARARGS,
    "generate MSDCCv1 hash"
  },
  {
    "generate_hash_msdcc2",
    func_generate_hash_msdcc2,
    METH_VARARGS,
    "generate MSDCCv2 hash"
  },
  {
    NULL,
    NULL,
    0,
    NULL
  } // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief module datatypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr struct
{
  const char *name;
  PyTypeObject *type;
} types[] =

{
  {"cipher_aes", &crypt_cipher_aes_t},
  {"cipher_des", &crypt_cipher_des_t},
  {"cipher_rc4", &crypt_cipher_rc4_t},
  {"cipher_rot13", &crypt_cipher_rot13_t},
  {"cipher_zip", &crypt_cipher_zip_t},
  {"hash", &crypt_hash_t},
  {"hmac", &crypt_hmac_t},
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create mobius.crypt module
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
new_crypt_module ()
{
  // initialize module mobius.crypt
  PyObject* module = Py_InitModule3 (
                       "crypt",
                       module_methods,
                       "Mobius Forensic Toolkit mobius.crypt module"
                     );

  // build types
  for (const auto& t : types)
    {
      if (PyType_Ready (t.type) < 0)
        return nullptr;
    }

  // add types
  for (const auto& t : types)
    {
      Py_INCREF (t.type);
      PyModule_AddObject (module, t.name, (PyObject *) t.type);
    }

  // return module
  return module;
}
