# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2018 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import CSearchWnd
import CXMLElement
import mobius
import pymobius.forensics.p2p

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief generic data holder
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class dataholder (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief retrieve data from Searches.dat
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve (item, model, data_folder, username):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # try to open Searches.dat file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  f = data_folder.get_child_by_name ('Searches.dat', False)
  if not f:
    return

  reader = f.new_reader ()
  if not reader:
    return

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # decode file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  decoder = mobius.decoder.mfc_decoder (reader)
  count = decoder.get_count ()
  searches = []

  while count == 1:
    csearch_wnd = CSearchWnd.decode (decoder)
    searches.append (csearch_wnd)
    count = decoder.get_count ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # process data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  d_search = {}
  for csearch_wnd in searches:

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # generate search history
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    for search in csearch_wnd.searches:
      search.app_id = 'shareaza'
      search.app = 'Shareaza'
      search.username = username
      search.text = search.qs.search_string
      search.timestamp = None
      search.first_hit_time = None
      search.last_hit_time = None
      search.id = search.qs.guid
      search.pxml = search.qs.pxml
      search.files = []

      search.metadata = []
      search.metadata.append (('Application', 'Shareaza'))
      search.metadata.append (('User name', username))
      search.metadata.append (('ID', search.qs.guid))
      search.metadata.append (('Min. size', search.qs.min_size))
      search.metadata.append (('Max. size', search.qs.max_size))

      model.searches.append (search)
      d_search[search.id] = search

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # generate remote files list
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    for match_file in csearch_wnd.match_list.match_files:
      for q in match_file.query_hits:
        search = d_search[q.search_id]

        rf = dataholder ()
        rf.username = username
        rf.app_id = 'shareaza'
        rf.app = 'Shareaza'
        rf.country = ''
        rf.timestamp = match_file.last_seen_time
        rf.size = match_file.size
        rf.str_size = match_file.str_size
        rf.hash_sha1 = match_file.hash_sha1
        rf.hash_tiger = match_file.hash_tiger
        rf.hash_ed2k = match_file.hash_ed2k
        rf.hash_bth = match_file.hash_bth
        rf.hash_md5 = match_file.hash_md5
        rf.preview = match_file.preview
        rf.name = q.name
        rf.url = q.url
        rf.protocol_id = q.protocol_id
        rf.client_id = q.client_id
        rf.comments = q.comments
        rf.pxml = q.pxml

        # hashes
        rf.hashes = []

        for t in ('sha1', 'tiger', 'ed2k', 'bth', 'md5'):
          h = dataholder ()
          h.type = t
          h.value = getattr (match_file, 'hash_%s' % t, None)
          rf.hashes.append (h)

        # tags
        rf.tags = set ()

        # peer data
        peer = dataholder ()
        peer.ip = q.ip
        peer.port = q.port
        peer.nickname = q.nick
        peer.country = ''
        peer.application = q.peer_app
        rf.peer = peer

        # metadata
        rf.metadata = []
        rf.metadata.append (('Search Term', search.text))
        rf.metadata.append (('Comments', rf.comments))
        rf.metadata.append (('Client ID', rf.client_id))
        rf.metadata += CXMLElement.get_metadata_from_pxml (rf.pxml)
        model.remote_files.append (rf)

        if not search.timestamp or search.timestamp > rf.timestamp:
          search.timestamp = rf.timestamp
          search.first_hit_time = rf.timestamp

        if not search.last_hit_time or search.last_hit_time < rf.timestamp:
          search.last_hit_time = rf.timestamp

        search.files.append (rf)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # normalize data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  for search in model.searches:
    search.metadata.append (('Hits', len (search.files)))

    if len (search.files) > 0:
      search.metadata.append (('First Hit Date/Time', search.first_hit_time))
      search.metadata.append (('Last Hit Date/Time', search.last_hit_time))

    search.metadata += CXMLElement.get_metadata_from_pxml (search.pxml)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # add file to ignored kff
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  pymobius.forensics.p2p.set_handled (item, f)
