// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief C++ API module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "reader.h"
#include "api.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new object from C++ object
//! \param obj C++ object
//! \return new object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
PyMobius_IO_Reader_from_cpp (mobius::io::reader obj)
{
  PyObject *ret = nullptr;

  if (obj)
    {
      ret = _PyObject_New (&io_reader_t);

      if (ret)
        ((io_reader_o *) ret)->obj = new mobius::io::reader (obj);
    }
  else
    {
      Py_INCREF (Py_None);
      ret = Py_None;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief is_seekable getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_is_seekable (io_reader_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyBool_FromLong (self->obj->is_seekable ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief is_rewindable getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_is_rewindable (io_reader_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyBool_FromLong (self->obj->is_rewindable ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_size (io_reader_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyInt_FromLong (self->obj->get_size ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef tp_getset[] =
{
  {
    (char *) "is_seekable",
    (getter) tp_getter_is_seekable,
    (setter) 0,
    (char *) "check if reader is seekable", NULL
  },
  {
    (char *) "is_rewindable",
    (getter) tp_getter_is_rewindable,
    (setter) 0,
    (char *) "check if reader is rewindable", NULL
  },
  {
    (char *) "size",
    (getter) tp_getter_size,
    (setter) 0,
    (char *) "data size in bytes", NULL
  },
  {NULL, NULL, NULL, NULL, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief read method
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_read (io_reader_o *self, PyObject *args)
{
  PyObject *ret = nullptr;

  try
    {
      // parse input args
      std::uint64_t arg_size = self->obj->get_size () - self->obj->tell ();

      if (!PyArg_ParseTuple (args, "|K", &arg_size))
        return nullptr;

      // build value
      ret = PyString_from_bytearray (self->obj->read (arg_size));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief tell method
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_tell (io_reader_o *self, PyObject *)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyInt_FromLong (self->obj->tell ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief seek method
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_seek (io_reader_o *self, PyObject *args)
{
  // parse input args
  std::int64_t arg_offset;
  int arg_whence = 0;

  if (!PyArg_ParseTuple (args, "L|i", &arg_offset, &arg_whence))
    return nullptr;

  mobius::io::reader::whence_type w;

  if (arg_whence == 0)
    w = mobius::io::reader::whence_type::beginning;

  else if (arg_whence == 1)
    w = mobius::io::reader::whence_type::current;

  else if (arg_whence == 2)
    w = mobius::io::reader::whence_type::end;

  else
    {
      PyErr_SetString (PyExc_ValueError, "invalid 'whence' argument");
      return nullptr;
    }

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      self->obj->seek (arg_offset, w);
      Py_INCREF (Py_None);
      ret = Py_None;
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief rewind method
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_rewind (io_reader_o *self, PyObject *)
{
  PyObject *ret = nullptr;

  try
    {
      self->obj->rewind ();
      Py_INCREF (Py_None);
      ret = Py_None;
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief skip method
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_skip (io_reader_o *self, PyObject *args)
{
  // parse input args
  std::int64_t arg_size;

  if (!PyArg_ParseTuple (args, "L", &arg_size))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      self->obj->skip (arg_size);
      Py_INCREF (Py_None);
      ret = Py_None;
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef tp_methods[] =
{
  {
    (char *) "read",
    (PyCFunction) tp_f_read,
    METH_VARARGS,
    "read bytes from reader"
  },
  {
    (char *) "tell",
    (PyCFunction) tp_f_tell,
    METH_VARARGS,
    "get current read position"
  },
  {
    (char *) "seek",
    (PyCFunction) tp_f_seek,
    METH_VARARGS,
    "set current read position"
  },
  {
    (char *) "rewind",
    (PyCFunction) tp_f_rewind,
    METH_VARARGS,
    "set current read position to the beginning of data"
  },
  {
    (char *) "skip",
    (PyCFunction) tp_f_skip,
    METH_VARARGS,
    "set read position n bytes ahead"
  },
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief tp_alloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
io_reader_o *
io_reader_tp_alloc ()
{
  return (io_reader_o *) io_reader_t.tp_alloc (&io_reader_t, 0);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
tp_dealloc (io_reader_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject io_reader_t =
{
  PyVarObject_HEAD_INIT (NULL, 0)
  "mobius.io.reader",               	           	// tp_name
  sizeof (io_reader_o),              	          	// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) tp_dealloc,  	                   	// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "reader for mobius.io.file",       	          	// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  tp_methods,                                  		// tp_methods
  0,                                       		// tp_members
  tp_getset,                	                   	// tp_getset
  0,                             	              	// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  0,                                    		// tp_new
  0,                                       		// tp_free
  0,                                       		// tp_is_gc
  0,                                       		// tp_bases
  0,                                       		// tp_mro
  0,                                       		// tp_cache
  0,                                       		// tp_subclasses
  0,                                       		// tp_weaklist
  0,                                       		// tp_del
  0,                                       		// tp_version_tag
};
