/******************************************************************************
 *
 * Copyright (C) 2003, 2004, 2005 Alex Karev.
 *
 * This file is part of Modb library.
 *
 * Modb library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * Modb library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Modb library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 ******************************************************************************/
#include <time.h>
#include <assert.h>
#include <errno.h>
#include <pthread.h>
#include <ctype.h>

#include "modb.h"

#define LMASK8 0x0F
#define LMASK16 0x00FF 

#define RUN_DOZA_STAT

#define MODB_ASCII_START_BYTE_NUMBER		5
#define MODB_RTU_START_BYTE_NUMBER		2

#define TO_STRING(e) #e

#define N_ELS(a) (sizeof (a) / sizeof (a)[0])

#define HEX_TO_ASCII_H(byte) ( (((byte) >> 4) < 0x0A) ? 		\
	(((byte) >> 4) + 0x30) : (((byte) >> 4) + 0x37) )
#define HEX_TO_ASCII_L(byte) ( (((byte) & LMASK8 ) < 0x0A) ? 		\
	(((byte) & LMASK8) + 0x30) : (((byte) & LMASK8) + 0x37) )

#define ASCII_TO_HEX(hb,lb) (uint8_t) ( ((((hb) < 0x3A) ? ((hb) - 0x30) : ((hb) - 0x37)) << 4) | \
	(((lb) < 0x3A) ? ((lb) - 0x30) : ((lb) - 0x37)) ) 
	
#define PUSH(a,v,id) \
	(void) ((a)[(id)++] = (uint8_t) (v),assert((id) < MODB_RTU_MAX_PACKET_SIZE))

#define PUSHA(a,v,id) \
	do { 								\
		((a)[(id)++] = HEX_TO_ASCII_H(v), 			\
		       	assert((id) < MODB_ASCII_MAX_PACKET_SIZE));	\
		((a)[(id)++] = HEX_TO_ASCII_L(v), 			\
			assert((id) < MODB_ASCII_MAX_PACKET_SIZE));	\
	} while(0)	

#define LRC(p,n,lrc)						\
	do {							\
		uint8_t *pb = p;				\
		volatile register int iter = (n);		\
		(lrc) = 0;					\
		while(iter-- > 0) {				\
			(lrc) += *(pb)++;			\
		}						\
		(lrc) = (uint8_t) (-((int8_t)(lrc)));		\
	} while(0)

/* 
 * Oh this crazy, crazy word!
 * They marshall hex message first, 
 * then they count LRC for hex data,
 * then they convert it to ascii,
 * then they send it in ascii. 
 * 
 * Or even worth, they convert it to ascii while sending byte by byte. 
 * Oh no, enough!
 */
#define LRC1(p,n,lrc)							\
	do {								\
		uint8_t *pb = p;					\
		int m = (n);						\
		(lrc) = 0;						\
		while(m) {						\
			(lrc) += (uint8_t) ASCII_TO_HEX((pb)[0],(pb)[1]);	\
			pb += 2; m -= 2;				\
		}							\
		(lrc) = (uint8_t) (-((int8_t)(lrc)));			\
	} while(0)

/* Table of CRC values for high.order byte */
static uint8_t auchCRCHi[] = {
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,
	0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,
	0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01,
	0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81,
	0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0,
	0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01,
	0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,
	0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,
	0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01,
	0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,
	0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0,
	0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01,
	0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81, 0x40, 0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41,
	0x00, 0xC1, 0x81, 0x40, 0x01, 0xC0, 0x80, 0x41, 0x01, 0xC0, 0x80, 0x41, 0x00, 0xC1, 0x81,
	0x40
};

static uint8_t auchCRCLo[] = {
	0x00, 0xC0, 0xC1, 0x01, 0xC3, 0x03, 0x02, 0xC2, 0xC6, 0x06, 0x07, 0xC7, 0x05, 0xC5, 0xC4,
	0x04, 0xCC, 0x0C, 0x0D, 0xCD, 0x0F, 0xCF, 0xCE, 0x0E, 0x0A, 0xCA, 0xCB, 0x0B, 0xC9, 0x09,
	0x08, 0xC8, 0xD8, 0x18, 0x19, 0xD9, 0x1B, 0xDB, 0xDA, 0x1A, 0x1E, 0xDE, 0xDF, 0x1F, 0xDD,
	0x1D, 0x1C, 0xDC, 0x14, 0xD4, 0xD5, 0x15, 0xD7, 0x17, 0x16, 0xD6, 0xD2, 0x12, 0x13, 0xD3,
	0x11, 0xD1, 0xD0, 0x10, 0xF0, 0x30, 0x31, 0xF1, 0x33, 0xF3, 0xF2, 0x32, 0x36, 0xF6, 0xF7,
	0x37, 0xF5, 0x35, 0x34, 0xF4, 0x3C, 0xFC, 0xFD, 0x3D, 0xFF, 0x3F, 0x3E, 0xFE, 0xFA, 0x3A,
	0x3B, 0xFB, 0x39, 0xF9, 0xF8, 0x38, 0x28, 0xE8, 0xE9, 0x29, 0xEB, 0x2B, 0x2A, 0xEA, 0xEE,
	0x2E, 0x2F, 0xEF, 0x2D, 0xED, 0xEC, 0x2C, 0xE4, 0x24, 0x25, 0xE5, 0x27, 0xE7, 0xE6, 0x26,
	0x22, 0xE2, 0xE3, 0x23, 0xE1, 0x21, 0x20, 0xE0, 0xA0, 0x60, 0x61, 0xA1, 0x63, 0xA3, 0xA2,
	0x62, 0x66, 0xA6, 0xA7, 0x67, 0xA5, 0x65, 0x64, 0xA4, 0x6C, 0xAC, 0xAD, 0x6D, 0xAF, 0x6F,
	0x6E, 0xAE, 0xAA, 0x6A, 0x6B, 0xAB, 0x69, 0xA9, 0xA8, 0x68, 0x78, 0xB8, 0xB9, 0x79, 0xBB,
	0x7B, 0x7A, 0xBA, 0xBE, 0x7E, 0x7F, 0xBF, 0x7D, 0xBD, 0xBC, 0x7C, 0xB4, 0x74, 0x75, 0xB5,
	0x77, 0xB7, 0xB6, 0x76, 0x72, 0xB2, 0xB3, 0x73, 0xB1, 0x71, 0x70, 0xB0, 0x50, 0x90, 0x91,
	0x51, 0x93, 0x53, 0x52, 0x92, 0x96, 0x56, 0x57, 0x97, 0x55, 0x95, 0x94, 0x54, 0x9C, 0x5C,
	0x5D, 0x9D, 0x5F, 0x9F, 0x9E, 0x5E, 0x5A, 0x9A, 0x9B, 0x5B, 0x99, 0x59, 0x58, 0x98, 0x88,
	0x48, 0x49, 0x89, 0x4B, 0x8B, 0x8A, 0x4A, 0x4E, 0x8E, 0x8F, 0x4F, 0x8D, 0x4D, 0x4C, 0x8C,
	0x44, 0x84, 0x85, 0x45, 0x87, 0x47, 0x46, 0x86, 0x82, 0x42, 0x43, 0x83, 0x41, 0x81, 0x80,
	0x40
};

/* The function returns the CRC as a unsigned short type */
#define CRC16(b,len,crc) \
	do { \
		(crc) = 0;		\
		int usDataLen = (len);											\
		uint8_t *puchMsg = (b);											\
		uint8_t uchCRCHi = 0xFF ; /* high byte of CRC initialized */	\
		uint8_t uchCRCLo = 0xFF ; /* low byte of CRC initialized */		\
		unsigned int uIndex ; /* will index into CRC lookup table */	\
		while (usDataLen--) { \
			uIndex = uchCRCHi ^ *puchMsg++ ; /* calculate the CRC */	\
			uchCRCHi = uchCRCLo ^ auchCRCHi[uIndex] ;					\
			uchCRCLo = auchCRCLo[uIndex] ;								\
		}	\
		(crc) = ( uchCRCHi << 8) | uchCRCLo;		\
	} while(0)



enum {
	RESERVED0,
        ILLEGAL_FUNCTION,
        ILLEGAL_DATA_ADDRESS,
        ILLEGAL_DATA_VALUE,
        SLAVE_DEVICE_FAILURE,
        ACKNOWLEDGE,
        SLAVE_DEVICE_BUSY,
        RESERVED1,
        MEMORY_PARITY_ERROR,
        RESERVED2,
        GATEWAY_PATH_UNAVAILABLE,
        GATEWAY_TARGET_DEVICE_FAILED_TO_RESPOND,
} Excepions;

struct exceptions {
        char *desc;
} except[] = 
{
        { TO_STRING("Unknown") },
        { TO_STRING(ILLEGAL_FUNCTION) },
        { TO_STRING(ILLEGAL_DATA_ADDRESS) },
        { TO_STRING(ILLEGAL_DATA_VALUE) },
        { TO_STRING(SLAVE_DEVICE_FAILURE) },
        { TO_STRING(ACKNOWLEDGE) },
        { TO_STRING(SLAVE_DEVICE_BUSY) },
        { TO_STRING(RESERVED1) },
        { TO_STRING(MEMORY_PARITY_ERROR) },
        { TO_STRING(RESERVED2) },
        { TO_STRING(GATEWAY_PATH_UNAVAILABLE) },
        { TO_STRING(GATEWAY_TARGET_DEVICE_FAILED_TO_RESPOND) }
};


volatile int exit_simulator;
volatile int print_stat = 0;


static int modb_ascii_snd_rcv(Port *p, uint8_t dsta, uint8_t fun,
		uint8_t *buffer, size_t n, size_t len);
static int modb_rtu_snd_rcv(Port *p, uint8_t dsta, uint8_t fun,
		uint8_t *buffer, size_t n, size_t len);

static int (*modb_snd_rcv[])(Port *p, uint8_t dsta, uint8_t fun,
		uint8_t *buffer, size_t n, size_t len ) = 
{
	modb_ascii_snd_rcv,
	modb_rtu_snd_rcv
};

static int modb_rtu_get_next_message(Port *p, 
		uint8_t address, uint8_t *buffer, size_t n);
static int modb_ascii_get_next_message(Port *p, 
		uint8_t address, uint8_t *buffer, size_t n);

static int (*modb_get_next_message[])(Port *p, 
		uint8_t address, uint8_t *buffer, size_t n) = 
{
	modb_ascii_get_next_message,
	modb_rtu_get_next_message
};


static int modb_ascii_send_response(Port *p, uint8_t *buffer, size_t size, int len);
static int modb_rtu_send_response(Port *p, uint8_t *buffer, size_t size, int len);

static int (*modb_send_response[])(Port *p, uint8_t *buffer, size_t size, int len) =
{
	modb_ascii_send_response,
	modb_rtu_send_response
};

static inline int modb_bin_to_ascii( uint8_t *bina, size_t binl, 
		uint8_t *asciia, size_t asciil );

static inline int modb_ascii_to_bin( uint8_t *asciia, size_t asciil, 
		uint8_t *bina, size_t binl );

static inline int modb_handle_exception(uint8_t *frame, size_t len);
static int modb_prepare_exception( int id, uint8_t *in, size_t len);

#if 0
static unsigned char LRC_fun(unsigned char *auchMsg, unsigned short usDataLen);

unsigned char 
LRC_fun(auchMsg, usDataLen) /* the function returns the LRC as a type unsigned char */
	unsigned char *auchMsg ; /* message to calculate LRC upon */
	unsigned short usDataLen ; /* quantity of bytes in message */
{
	unsigned char uchLRC = 0 ; /* LRC char initialized */
	while (usDataLen--) /* pass through message buffer */
		uchLRC += *auchMsg++ ; /* add buffer byte without carry */
	return ((unsigned char)(-((char)uchLRC))) ; /* return twos complement */
}
#endif

int
modb_handle_exception(uint8_t *frame, size_t len)
{
	fprintf(stderr, "exception[%02x]: %s\n", 
				frame[ 2 ],
				(frame[ 2 ] < N_ELS(except)) ? except[ frame[ 2 ]  ].desc : "Unknown");

	return (0);
}

int
modb_prepare_exception( int id, uint8_t *in, size_t len)
{
#if 1
	fprintf(stderr, "exception: [%s]\n", except[id].desc);
#endif
	in[1] = in[1] | 0x80;
	in[2] = (uint8_t) id;

	return (3);
}

/** Dummy registers array for master/slave tests */
static uint16_t regs[] =
{
	0xF1F2, 0x01FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0x02FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0x03FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0x04FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,

	0xF1F2, 0x05FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0x06FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0x07FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0x08FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,

	0xF1F2, 0xA9FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0xB1FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0xC2FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0xD3FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,

	0xF1F2, 0xE4FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0xF5FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0x16FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4,
	0xF1F2, 0x27FF, 0xFF00, 0xF3F4,	0xF1F2, 0x00FF, 0xFF00, 0xF3F4
}; /* 512 bytes */

static uint16_t regs_zero[] =
{
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,

	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,

	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,

	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,	0x0000, 0x0000, 0x0000, 0x0000
}; /* 512 bytes */


int
modb_read_regs(uint8_t *buffer, int len)
{
	int i;
	uint16_t
		address,
		count,
		*pregs;
	uint8_t *pb = buffer + 3;
	

	address	= buffer[2];
	address	= (address << 8) | buffer[3];

	count	= buffer[4];
	count	= (count<< 8) | buffer[5];
	
#if 1
	fprintf(stderr, "Request to read %hu regs from address %04x\n", count, address);
#endif 

	/*
	if (count > N_ELS(regs_zero)) return (-ILLEGAL_DATA_VALUE);
	*/

	if ((count == 0x0000) && (count > 0x007D) ) return (-ILLEGAL_DATA_VALUE);

#if 0
	if ((address + count) > N_ELS(regs_zero)) return (-ILLEGAL_DATA_ADDRESS);
	pregs =	regs_zero + address;
#else 

	pregs = regs_zero;
#endif

	buffer[2] = count << 1;
	for(i = 0; (i < count) && (i < len - 2 - 1); i++) {
		pb[i << 1]		= (uint8_t) (pregs[i] >> 8);
		pb[(i << 1) + 1]	= (uint8_t) (pregs[i] & LMASK16);
	}

	return ((i << 1) + 3 /* addr, func, bytes */);
}

int
modb_write_regs(uint8_t *buffer, int len)
{
	int i;
	uint16_t
		address,
		count,
		bytes,
		*pregs;
	uint8_t *pb;
	

	address	= buffer[2];
	address	= (address << 8) | buffer[3];

	count	= buffer[4];
	count	= (count << 8) | buffer[5];

	bytes	= buffer[6];

#if 1 /** Debug */
	fprintf(stderr, "Request to write %hu regs starting from address %04x\n", count, address);
#endif 
	
	if ( ((count == 0x0000) && (count > 0x007D)) ||
			(bytes != count << 1)) 
		return (-ILLEGAL_DATA_VALUE);

	/*
	if ((address + count) > N_ELS(regs)) return (-ILLEGAL_DATA_ADDRESS);
	pregs	= regs + address;
	*/
 
	pb		= buffer + 7;
	pregs = regs_zero;
	for(i = 0; i < count; i++) {
		pregs[i] = ( (uint16_t) pb[ i<<1 ] << 8) | pb[ (i << 1) + 1];
	}

	{
		fputs("Registers DUMP:\n", stderr);
		DUMP_BUFFER_BY_WORD(regs_zero, count * 2);
	}

	return (6); /* address, function id, 2 * address, 2 * quantity */
}

int 
modb_ascii_send_response(Port *p, uint8_t *buffer, size_t size, int len)
{
	int ln;
	uint8_t frame[MODB_ASCII_MAX_PACKET_SIZE];
	uint8_t LRC;

	if (len > size) {
		errno=EFBIG;
		return (-1);
	}
	
	LRC(buffer,len,LRC);
#if 0
	DUMP_BUFFER(buffer,len);
#endif
	memcpy(p->out,buffer,len);
	p->fill_out = len;
	
	/* Converting to ascii */
	ln =  modb_bin_to_ascii(buffer, len, frame, sizeof frame);

	/* Add LRC and stop chars to the end  of frame */
	PUSHA(frame,LRC,ln); 
	PUSH(frame,MODB_STOP_CHAR1,ln);
	PUSH(frame,MODB_STOP_CHAR2,ln);

	
	return rs_send(p, frame, ln);
}

int 
modb_rtu_send_response(Port *p, uint8_t *buffer, size_t size, int len)
{
	uint16_t CRC16;
	CRC16(buffer,len,CRC16);

	PUSH(buffer,CRC16 >> 8,len);
	PUSH(buffer,CRC16 & LMASK16,len);
	
	memcpy(p->out,buffer,len);
	p->fill_out = len;

	return rs_send(p, buffer, len);
}


int
modb_doza_read_test(Port *p, uint8_t dsta, 
                    uint16_t from, uint16_t quantity, unsigned int mode)
{
	uint16_t cmp_regs[ MODB_MAX_REGISTERS_TO_WRITE ];
	int ret, count = 0;

		
	do {
			if ((ret = modb_test_fun_rhr(p, 
					dsta, 
					from,
					quantity,
					cmp_regs, 
					sizeof cmp_regs, 
					mode )) == -1) {
				if (errno != EINTR) {
					/*
					fprintf(stderr, "Packets recieved before LCR mistmach: %d\n", count);
					count = 0;
					*/
				}
			} else {
				count++;
			}
			
#if 0
			fputs("get:\n", stderr);
			DUMP_BUFFER_BY_WORD(cmp_regs, ret);
#endif
			usleep(100000L);
			if (print_stat) {
				rs_print_stat(p, stderr);
				print_stat = 0;
			}
	} while(!exit_simulator);


	return (0);
}


int
modb_master_test(Port *p, uint8_t dsta1, uint8_t dsta2, 
                    uint16_t from, uint16_t quantity, unsigned int mode)
{
	uint16_t cmp_regs[2][MODB_MAX_REGISTERS_TO_WRITE + 1];
	int ret, i, id, c;
	uint16_t tmpreg;
	char tstr[32];

#if 1
	if (dsta1 > dsta2) {
		uint8_t tmp;

		tmp = dsta1;
		dsta2 = dsta1;
		dsta1 = tmp;
	}
#endif
	
	do {

		time_t start;
		for(id = dsta1; id <= dsta2; id++) {
			for (c = 1; c <= quantity; c++) {
			
				time(&start);
				strncpy(tstr, ctime(&start), sizeof tstr);
				tstr[ strlen(tstr) - 1 ] = '\0';

				fprintf(stdout, "%s: Testing i/o for device %d. processing %d register%s ", 
						tstr, id, c, (c > 1) ? "s" : "" );

				for(i = 0; i < c; i++) {
					cmp_regs[0][i] += 1;
				}

				if ((ret = modb_test_fun_wmr(p, 
						id, 
						from, 
						cmp_regs[0], 
						c, 
						mode )) == -1) {
					if (errno != EINTR)
						fputs("Error running write test!\n",stderr);
				}

				if ((ret = modb_test_fun_rhr(p, 
						id, 
						from,
						c,
						cmp_regs[1], 
						sizeof cmp_regs, 
						mode )) == -1) {
					if (errno != EINTR)
						fputs("Error running write test!\n",stderr);
				}

				for(i = 0; i < c; i++) {

					tmpreg = ((cmp_regs[1][i] << 8) & 0xFF00) |
							((cmp_regs[1][i] >> 8) & LMASK16);

					if (cmp_regs[0][i] != tmpreg) break;
				}

				if (i == c) { 
					fputs("\tok", stdout);
				} else {
					fputs("\tfailed", stdout);
				}

				(void) fputc('\n', stdout), (void) fflush(stdout);
			}
		}
	} while(!exit_simulator);

	return (0);
}


int
modb_slave_simulator(Port *p, uint8_t address, unsigned int mode)
{
	uint8_t buffer[MODB_RTU_MAX_PACKET_SIZE];
	int 
		ret,
		len = 0;

	do {
	
		mode = (mode < MODB_MODES_NUMBER) ? (mode) : MODB_ASCII_MODE;
		
		if ((ret = modb_get_next_message[mode](p, address, buffer, sizeof buffer) == -1)) {
			if  (errno != EINTR)
				perror("modb_get_next_message");
			continue;
		}

		fprintf(stderr, "get function %02x\n", buffer[1] );

		switch(buffer[1]) {
			case MODB_FUN_RHR:
				len = modb_read_regs(buffer, sizeof buffer);
				break;
			case MODB_FUN_WMR:
				len = modb_write_regs(buffer, sizeof buffer);
				break;
			default:
				len = modb_prepare_exception( ILLEGAL_FUNCTION, buffer, sizeof buffer);
				break;
		}

	
		if (buffer[0] == 0x00) continue;
		if (len < 0) { /* exception while processing */
			len = modb_prepare_exception( -len, buffer, sizeof buffer);
		}
		
		if ((ret = modb_send_response[mode](p, buffer, sizeof buffer, len)) == -1) {
			perror("modb_send_response");
		}

	} while (!exit_simulator);

	return (ret);
}

int
modb_rtu_get_next_message(Port *p, uint8_t address, uint8_t *buffer, size_t n)
{
	uint16_t CRC16;
	int 
		wevent,
		r;

	do {
		if ((wevent = rs_wait_for_incoming_event(p->fd, 2, 0)) > 0) {
		/* Get response */
			
		} else if (!wevent) {	/* timeout */
#if 0
			fputs("TIMEOUT!!!\n",stderr);
#endif
			continue;
		} else {		/* error   */

			if (errno==EINTR) break;
#if 0
			fputs("ERROR!!!\n",stderr);
#endif
			return (-1);
		}

		if ((r = rs_recv(p, buffer, MODB_RTU_MAX_PACKET_SIZE)) == -1) {
			perror("rs_recv");
			return (-1);
		} else {
#if 0
			fprintf(stderr, "Get %d bytes\n", r);
#endif
		}

#if 0
		fputs("in (DUMP):\n", stderr);
		DUMP_BUFFER(frame, r + 1);
#endif

		/* Check for CRC */
		CRC16(buffer,r,CRC16);

		memcpy(p->in,buffer,r);
		p->fill_in = r;

#if 0
		fputs("in (DUMP):\n", stderr);
		DUMP_BUFFER(buffer,r);
#endif

		if (CRC16) { /* Incorrect CRC */
#if 1
			fputs("Incorrect CRC16\n", stderr);
#endif
			errno=EPROTO;
			return (-1);
		}

		/* Check for address, first byte */
		if (buffer[0] != address) {
#if 1
			fputs("Incorrect response address\n", stderr);
#endif
			continue;
		}

		break;
	} while (!exit_simulator);

	return (0);
}

int
modb_ascii_get_next_message(Port *p, uint8_t address, uint8_t *buffer, size_t n)
{
	int 
		wevent,
		c,
		ln,
		r;
	uint8_t LRC, retLRC; /* Longitudinal Redundancy Checking byte */
	uint8_t frame[MODB_ASCII_MAX_PACKET_SIZE] = {0}; 

	do {
		if ((wevent = rs_wait_for_incoming_event(p->fd, MODB_SEC_TO_WAIT_FOR_RESPONSE, 0)) > 0) {
			 /* Get response */
			

		c = '\0';
		if (read(p->fd, &c, 1) == -1) return (-1);
#if 0
		fprintf(stderr, "get byte ASCII [%c] hex [%08x] MODB_START_CHAR ASCII [%c] hex[%08x]\n", 
				c, c,
				MODB_START_CHAR, MODB_START_CHAR);
#endif

		if (c == MODB_START_CHAR) {
#if 0
			fputs("Get MODB_START_CHAR!!!\n",stderr);
#endif
		} else {
			continue;
		}
			
		} else if (!wevent) {	/* timeout */
#if 0
			fputs("TIMEOUT!!!\n",stderr);
#endif
			continue;
		} else {		/* error   */

#if 0
			fputs("ERROR!!!\n",stderr);
#endif

			if (errno=EINTR) {

#if 0
				fputs("INCOMING SIGNAL!!!\n",stderr);
#endif
				
			}
			return (-1);
		}

		/* buffer[0] already contain ':', and already read first byte, 
		 * so we pass it here */
		if ((r = rs_recv(p, frame + 1, MODB_ASCII_MAX_PACKET_SIZE)) == -1) {

			if (errno=EINTR) {
#if 0
				fputs("ERROR - SIGNAL?!", stderr);
#endif
			}
		
			perror("rs_recv");
			return (-1);
		} else {
#if 0
			fprintf(stderr, "Get %d bytes\n", r);
#endif
		}

#if 0
		fputs("in (DUMP):\n", stderr);
		DUMP_BUFFER(frame, r + 1);
#endif

		ln = modb_ascii_to_bin(frame + 1, r - 2, buffer, MODB_RTU_MAX_PACKET_SIZE);
	
		/* Check for LRC */
		LRC(buffer,ln - 1,LRC);
		retLRC = buffer[ln - 1];

		memcpy(p->in,buffer,ln);
		p->fill_in = ln;
#if 0
		fputs("in (DUMP):\n", stderr);
		DUMP_BUFFER(buffer,ln);
#endif

		if (LRC != retLRC) { /* Incorrect LRC */
			p->errChksum++;
#if 1
			fputs("Incorrect LRC\n", stderr);
#endif
			errno=EPROTO;
			return (-1);
		}

#if 0
		/* Check for address, first byte */
		if (buffer[0] != address) {

#if 0
			fputs("Incorrect response address\n", stderr);
#endif
			continue;
		}
#endif

		break;
	} while (!exit_simulator);

	return(0);
}


int
modb_test_fun_rhr(Port *p, uint8_t dsta, uint16_t sb, uint16_t w, 
		uint16_t *a, size_t asize, unsigned int mode)
{
	uint8_t frame[ MODB_RTU_MAX_PACKET_SIZE ] = { 0 };
	int 
		len		= MODB_RTU_START_BYTE_NUMBER,
		ret		= 0;

	PUSH(frame,(sb >> 8),len);		/* start address hight byte 		*/
	PUSH(frame,(sb & LMASK16),len);	/* start address low byte			*/
	PUSH(frame,(w >> 8),len);		/* query words number hight byte	*/
	PUSH(frame,(w  & LMASK16),len);	/* query words number low byte		*/

	mode = (mode < MODB_MODES_NUMBER) ? (mode) : MODB_ASCII_MODE;
	if ((ret = modb_snd_rcv[mode](p, dsta, MODB_FUN_RHR, 
					frame, sizeof frame, len)) == -1) { 
		return (-1);
	}

	if (!ret) { /* Handle exception */
		(void) modb_handle_exception(frame, sizeof frame);
		return (0);
	}
	
	if (frame[2] > asize) { 
		errno=ERANGE;
	   	return (-1);
	}
	
	memcpy(a, &frame[3], frame[2]);
	
	return ( frame[2] );
}

int
modb_test_fun_wmr(Port *p, uint8_t dsta, uint16_t sb,
		uint16_t *regs, uint16_t w, unsigned int mode)
{
	uint8_t frame[MODB_RTU_MAX_PACKET_SIZE] = { 0 };
	int 
		len		= MODB_RTU_START_BYTE_NUMBER,
		ret		= 0, 
		i, bytes;

	PUSH(frame,(sb >> 8),len);		/* start address hight byte				*/
	PUSH(frame,(sb & LMASK16),len);	/* start address low byte				*/
	PUSH(frame,(w >> 8),len);		/* number of regs to write (hight byte)	*/
	PUSH(frame,(w &  LMASK16),len);	/* number of regs to write (low byte)	*/

	bytes = w * 2;					/* Why they really need it?				*/
	PUSH(frame,bytes,len);			/* bytes count							*/

	for(i = 0; i < w; i++) {
		PUSH(frame, (regs[i] >> 8),len);	
		PUSH(frame, (regs[i] & LMASK16),len);	
	}
	
	mode = (mode < MODB_MODES_NUMBER) ? (mode) : MODB_ASCII_MODE;
	if ((ret = modb_snd_rcv[mode](p, dsta, MODB_FUN_WMR, 
					frame, sizeof frame, len)) == -1) { 
		return (-1);
	}

    if (!ret) { /* Handle exception */
		(void) modb_handle_exception(frame, sizeof frame);
		return (0);
   }
 

	return (ret);
}

int
modb_test_fun_rsi(Port *p, uint8_t dsta, unsigned int mode)
{
	uint8_t frame[MODB_RTU_MAX_PACKET_SIZE] = { 0 };
	int 
		len		= MODB_RTU_START_BYTE_NUMBER,
		ret		= 0;

	mode = (mode < MODB_MODES_NUMBER) ? (mode) : MODB_ASCII_MODE ;
	if ((ret = modb_snd_rcv[mode](p, dsta, MODB_FUN_RSI, 
					frame, sizeof frame, len)) == -1) { 
		return (-1);
	}

    if (!ret) { /* Handle exception */
		(void) modb_handle_exception(frame, sizeof frame);
		return (0);
	}
	
	fprintf(stderr, "F[%02Xh] Reporting slave device %d ID,"
					" bytes count %d\n",
			frame[1], 
			frame[0],
			frame[2]);
	
	fputs( "Dump of slave ID\n", stderr );
	DUMP_BUFFER(&frame[3], frame[2]);

	return (ret);
}



int
modb_bin_to_ascii(uint8_t *bina, size_t binl, uint8_t *asciia, size_t asciil )
{
	int n = binl;
	int len = 0;
	uint8_t *p;

	p = bina;

	PUSH(asciia,MODB_START_CHAR,len);

	while(n--) {
		PUSHA(asciia,*p,len);
		p++;
	}

	return (len);
}

int
modb_ascii_to_bin(uint8_t *asciia, size_t asciil, uint8_t *bina, size_t binl ) 
{
	int m = asciil;
	uint8_t *pa = asciia;
	int len = 0;

	while(m > 0) {
		if (!isascii(pa[0]) || !isascii(pa[1]) ) return (-1);
		PUSH(bina,ASCII_TO_HEX(pa[0],pa[1]),len);
		pa += 2, m -= 2;										
	}														

	return (len);
}

int
modb_ascii_snd_rcv(Port *p, uint8_t dsta, uint8_t fun,
		uint8_t *buffer, size_t n, size_t len)
{
	int 
		ret, 
		wevent,
		retries		= MODB_RETRIES,
		c,
		towait		= MODB_SEC_TO_WAIT_FOR_RESPONSE,
		r,
		ln = 0;
	uint8_t frame[MODB_ASCII_MAX_PACKET_SIZE] = {0}; 
	uint8_t LRC, retLRC; /* Longitudinal Redundancy Checking byte */
	
	/** Filling packet */
	PUSH(buffer,dsta,ln);	/* destination address  */
	PUSH(buffer,fun,ln);	/* function number	*/

	/* Converting to ascii */
	ln =  modb_bin_to_ascii(buffer, len, frame, sizeof frame);

#if 1 /* Print out modbus message */
	fprintf(stderr,"out (ASCII): %s\n", frame);
	fputs("out (DUMP):\n", stderr);
	DUMP_BUFFER(buffer, len);
#endif

	LRC(buffer,len,LRC);
	memcpy(p->out, buffer, len);
	p->fill_out = len;

	/* Add LRC and stop chars to the end */
	PUSHA(frame,LRC,ln); 
	PUSH(frame,MODB_STOP_CHAR1,ln);
	PUSH(frame,MODB_STOP_CHAR2,ln);

#if 0
	fprintf(stderr,"out (ASCII): %s\n", frame);
	fputs("out (DUMP):\n", stderr);
	DUMP_BUFFER(frame, ln);
#endif


	if ((ret = rs_send(p, frame, ln)) == -1) {
		return (-1);
	}
	p->pts_snd++;

	while (retries--) { 
		if ((wevent = rs_wait_for_incoming_event(p->fd, towait, 0)) > 0) {
			 /* Get response */
#if 0
			fputs("Get DATA!!!\n",stderr);
#endif
			c = '\0';
			if (read(p->fd, &c, 1) == -1) return (-1);
			if (c == MODB_START_CHAR) {
#if 0
				fputs("Get MODB_START_CHAR!!!\n",stderr);
#endif
				towait = 1;
				break;
			} else {
#if 0
				fprintf(stderr, "wrong char \'%c\' [%02x]\n", c, c);
#endif
				if ( c != 0x7F )
					p->errStartCH++;
				continue;
			}
			
		} else if (!wevent) {	/* timeout */
			p->errTimeout++;
#if 0
			fputs("TIMEOUT!!!\n",stderr);
#endif
			errno = ETIMEDOUT;
			return (-1);
		} else {		/* error   */
#if 0
			fputs("ERROR!!!\n",stderr);
#endif
			if (errno=EINTR) {
				continue;
			}
			
			return (-1);
		}
	}
	
	/* 
	 * buffer[0] already contain ':', and already we read first byte, 
	 * so we pass it. We get also 0x0D here.
	 *
	 */
	retries = 10;
	r = rs_recv(p, frame + 1, MODB_ASCII_MAX_PACKET_SIZE);
	if (r < 0)
		return (-1);
	
	p->pts_rcvd++;

#if 1
	fputs("in (DUMP):\n", stderr);
	DUMP_BUFFER(frame, r);
#endif
	
	/* Perform simple check here */
#if 0
	if (r < MODB_ASCII_MIN_RESPONSE_SIZE) return (-1);
#endif

#if 0 /** FIXME: are we really need it? */
	/* Check for stop bytes */
	if (	( frame[ r     ] != MODB_STOP_CHAR2) || 
		( frame[ r - 1 ] != MODB_STOP_CHAR1) ) {
#if 0
		fprintf(stderr, "%02x %02x\n", frame[r - 1], frame[r]); 
#endif
		errno=EPROTO;
		return (-1);
	}
#endif
#if 0
	fprintf(stderr, "%02x %02x\n", frame[r - 1], frame[r]); 
#endif

	if (frame[r - 1] != 0x0D ) {
		errno=EPROTO;
		p->errIncmt++;
		return (-1);
	}
	
	ln = modb_ascii_to_bin(frame + 1, r - 3, buffer, MODB_RTU_MAX_PACKET_SIZE);
	
	if (ln == -1) {
		errno=EPROTO;
		p->errNoASCII++;
	   	return (-1);
	}

		
	/* Check for LRC */
	LRC(buffer,ln - 1,LRC);		/* ln-1 --- without LCR byte */
	retLRC = buffer[ln - 1];

	memcpy(p->in, buffer, ln);
	p->fill_in = ln;

#if 0
	fprintf(stderr, "f[%2d]:%02x\n", r-1, buffer[r-1] );
	fprintf(stderr, "f[%2d]:%02x\n", r-2, buffer[r-2] );
#endif
#if 0
	fprintf(stderr, "Bytes recieved %d\n", ln);
	DUMP_BUFFER(buffer, ln);
#endif

	if (LRC != retLRC) { /* Incorrect LRC */
		p->errChksum++;
		// rs_print_stat(p, stderr);
#if 0
		LRC_flag = 1;
//		DUMP_BUFFER(buffer, r);
		//fputs("Incorrect LRC\n", stderr);
#endif

		errno=EPROTO;
		return (-1);
	}

	/* Check for address, first byte */
	if (buffer[0] != dsta) {
#if 0
		fputs("Incorrect response address\n", stderr);
#endif
		errno=EADDRNOTAVAIL;
		return (-1);
	}

	/* Check for function number */
	if ( !(buffer[1] & fun) ) {
#if 0
		fprintf(stderr, "Incorrect function number out: %02x in: %02x\n", 
				fun, buffer[1] & fun);
#endif
		errno=ENOSYS;
		return (-1);
	}
	
#if 0
	fputs("Incoming message:\n",stderr);
	DUMP_BUFFER(buffer,ln);
#endif

	if ( buffer[1] & 0x80 ) { /* exception */
		return (0);
	}
	
	return (ln);
}

int 
modb_rtu_snd_rcv(Port *p, uint8_t dsta, uint8_t fun,
		uint8_t *buffer, size_t n, size_t len)
{
	int 
		ret, 
		wevent,
		towait		= MODB_SEC_TO_WAIT_FOR_RESPONSE,
		r,
		ln = 0;
	uint16_t 
		CRC		= 0;

	/* Filling packet */
	PUSH(buffer,dsta,ln);	/* destination address  */
	PUSH(buffer,fun,ln);	/* function number	*/

#if 0 /* Print out modbus message */
	fprintf(stderr,"out (ASCII): %s\n", frame);
	fputs("out (DUMP):\n", stderr);
	DUMP_BUFFER(buffer, len);
#endif

	CRC16(buffer,len,CRC);

	memcpy(p->out, buffer, len);
	p->fill_out = len;

	PUSH(buffer,CRC >> 8,len);
	PUSH(buffer,CRC & LMASK16,len);


#if 0
	fputs("out (DUMP):\n", stderr);
	DUMP_BUFFER(buffer, len + 2);
#endif

	if ((ret = rs_send(p, buffer, len)) == -1) {
		return (-1);
	}

	if ((wevent = rs_wait_for_incoming_event(p->fd, towait, 0)) > 0) {
			 /* Get response */
#if 0
			fputs("Get DATA!!!\n",stderr);
#endif
	} else if (!wevent) {	/* timeout */
#if 0
			fputs("TIMEOUT!!!\n",stderr);
#endif
			errno = ETIMEDOUT;
			return (-1);
	} else {		/* error   */
#if 0
			fputs("ERROR!!!\n",stderr);
#endif
			if (errno=EINTR) {
#if 0
				fputs("INCOMING SIGNAL!!!\n",stderr);
#endif
			}
			return (-1);
	}
	
	if ((r = rs_recv(p, buffer, MODB_RTU_MAX_PACKET_SIZE)) == -1) {

		if (errno=EINTR) {
#if 0	
			fputs("SIGNAL?!", stderr);
#endif
		}
		
		perror("rs_recv");
		return (-1);
	} else {
#if 0
		fprintf(stderr, "Get %d bytes\n", r);
#endif
	}


#if 0
	fputs("in (DUMP):\n", stderr);
	DUMP_BUFFER(buffer, r);
#endif
	
	/* Perform simple check here */
#if 0
	if (r < MODB_ASCII_MIN_RESPONSE_SIZE) return (-1);
#endif

	/* Check for CRC */
	CRC16(buffer,r,CRC);
	/* Unused
	retCRC  = buffer[ r - 2 ] << 8;
	retCRC |= buffer[ r - 1 ];
	*/
	memcpy(p->in, buffer, r);
	p->fill_in = r;


	if (CRC) { /* Incorrect CRC */
		p->errChksum++;
#if 0
		fputs("Incorrect CRC16\n", stderr);
#endif
		errno=EPROTO;
		return (-1);
	}

	/* Check for address, first byte */
	if (buffer[0] != dsta) {
#if 0
		fputs("Incorrect response address\n", stderr);
#endif
		errno=EADDRNOTAVAIL;
		return (-1);
	}

	/* Check for function number */
	if ( !(buffer[1] & fun) ) {
#if 0
		fprintf(stderr, "Incorrect function number out: %02x in: %02x\n", 
				fun, buffer[1] & fun);
#endif
		errno=ENOSYS;
		return (-1);
	}
	
#if 0
	fputs("Incoming message:\n",stderr);
	DUMP_BUFFER(buffer,ln);
#endif

	if ( buffer[1] & 0x80 ) { /* exception */
		return (0);
	}
	
	return (r);
}


