/******************************************************************************
 * Copyright (C) 2003, 2004, 2005 Alex Karev.
 *
 * This file is part of Modb library.
 *
 * Modb library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * Modb library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Modb library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 ****** vim: set ts=4 sw=4: **************************************************/
/*
 *
 * 	Read comments in rs485.h
 *
 */
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/wait.h>
#include <sys/sysinfo.h>

#include "rs.h"

#define CONTROL_RTS

#define MODB_RTU_MAX_PACKET_SIZE		256

#define TO_STRING(s) #s

#define N_ELS(a) (sizeof (a) / sizeof (a)[0])

#define BUSYLOOP( magic ) { \
		int j; \
		for ( j = 0; j < magic; j++ )	\
			;; \
	}

#define DropRTS( fd )	\
	{	\
		unsigned int arg;	\
		(void) ioctl(fd, TIOCMGET, &arg);	\
		arg &=  ~TIOCM_RTS /* ~TIOCM_DTR */;	\
		(void) ioctl(fd, TIOCMSET, &arg);	\
	}

#define RaiseRTS( fd )	\
	{	\
		unsigned int arg;	\
		(void) ioctl(fd, TIOCMGET, &arg);	\
		arg |= TIOCM_RTS /* TIOCM_DTR */;	\
		(void) ioctl(fd, TIOCMSET, &arg);	\
	}


#define WAIT_TEMT( fd ) \
{ \
	int count = TO_WAIT_TEMT; \
	\
	while(count--) { \
		if ( rs_xemty(fd) ) { \
			break; \
		} else { \
			BUSYLOOP(20); \
		} \
	} \
} 


#define MAGIC_NUM 'S'
#define IOCTL_3544_MODE_SET _IOW(MAGIC_NUM, 5, sizeof(unsigned short))
#define IOCTL_3544_MODE_GET _IOR(MAGIC_NUM, 6, sizeof(unsigned short))


#define ISSET_REBOOT(fl) fl & FL_REBOOT


static const int  RETRIES 	= 8;
static const long RTO 		= 50 /* ms */ * 1000;

/* INEUM protocol specific 
 * TODO: ??? */
// static unsigned  SLAVE_MODULE  =  0x01;		// must be unique for each slave !!!!!
												// read from file NETADRES.BIN
												// must be from 1 to 32
#ifdef INEUM_PROT												
static unsigned  SLAVE_MODULE  =  1;			// must be unique for each slave !!!!!
#endif

#define MOXA_MOD 		NULL
#define ADLINK_MOD		"ps3544"

enum {
	CONF_BAUD_FLAG,
	CONF_DATA,
	CONF_PARITY,
	CONF_STOPB,
	CONF_ARRAY_SIZE
};

int predef_port_cfg[][CONF_ARRAY_SIZE] = 
{
	{B9600,  CS7, 0, 2 },
	{B19200, CS7, 0, 2 },
	{B38400, CS7, 0, 2 },
	{B9600,  CS8, 1, 1 },
	{B19200, CS8, 1, 1 },
	{B38400, CS8, 1, 1 },
	{B9600,  CS8, 2, 1 },
	{B19200, CS8, 2, 1 },
	{B38400, CS8, 2, 1 }
};

typedef struct {
	char name[128];
	char path[256];
	const char *module;
	int irq;
	unsigned int port;
} port_state;

static const char *const dev_path = "/dev";

static const port_state rsps[] = {
/** Standart serial interfaces */
	{"COM1",				"ttyS0", NULL,   4, 	0x3F8},
	{"COM2",				"ttyS1", NULL,   3, 	0x2F8},
	{"COM3",				"ttyS2", NULL,  12, 	0x3E8},
	{"COM4",				"ttyS3", NULL,  11, 	0x2E8},
/** Octagon two-wire RS-485 */
	{"Octa6225 RS485 COM2", "ttyS01", NULL,  3, 	0x2F8},
	{"Octa6225 RS485 COM4", "ttyS03", NULL, 11, 	0x2E8},
/** INEUM most popular RS-485 board */
	{"DUMMY PCL-743b CH1",	"ttyS2", NULL,  5,	0x3E8},
	{"DUMMY PCL-743b CH2", 	"ttyS3", NULL,  7, 	0x2E8},
/** MOXA CT-114 (and probably other RS-485 boards) */
	{"MOXA CH1",            "ttyM0", MOXA_MOD},
	{"MOXA CH2",            "ttyM1", MOXA_MOD},
	{"MOXA CH3",            "ttyM2", MOXA_MOD},
	{"MOXA CH4",            "ttyM3", MOXA_MOD},
/** Adlink cPCI-3544 (and probably other RS-485 boards) */
	{"ADLINK CH1",          "ttyL0", ADLINK_MOD},
	{"ADLINK CH2",          "ttyL1", ADLINK_MOD},
	{"ADLINK CH3",          "ttyL2", ADLINK_MOD},
	{"ADLINK CH4",          "ttyL3", ADLINK_MOD}
/** Fastwel CPB902 board ports */
	{"CPB902 CH3",		"ttyS3", NULL, 9, 0x100},
	{"CPB902 CH4",		"ttyS4", NULL, 9, 0x108},
	{"CPB902 CH5",		"ttyS5", NULL, 9, 0x110},
	{"CPB902 CH6",		"ttyS6", NULL, 9, 0x118}
};


#if 0
static unsigned int rs_xemty( int fd );
#endif
static void rs_delay_ms(long ms);
static int set_low_latency( int fd, int on );
static int rs_set_irq(int fd, int irq);
static int rs_set_port(int fd, unsigned int port);

static int rs_get_response(Port *p, unsigned char *frame, int *len);
#if 0
static int rs_load_module(const char *name);
#endif

#if 0
static void print_warning(const char *fmt, ...);
	
static void 
print_warning(const char *fmt, ...)
{
	va_list vl;

	va_start(vl, fmt);
	vfprintf(stderr, fmt, vl);
	va_end(vl);
}

#endif

unsigned short 
rs_get_network_mode_3544(int fd)
{
	unsigned short arg = 0;
	int res;
	
	res = ioctl(fd, IOCTL_3544_MODE_GET, &arg);
	if (res) 
		return (-1);

	return (arg);
}

int
rs_switch_modbus_mode(Port *p, int mode)
{
 	tcflush(p->fd, TCIOFLUSH);
        tcgetattr( p->fd, &p->oldt );

	memset(&p->newt, 0, sizeof p->newt);

	switch(mode) {
		case RS_FLAG_ASCII_MODBUS:
			p->newt.c_lflag |=	NOFLSH | ICANON | ICRNL;
			p->newt.c_cflag = (p->newt.c_cflag & ~CSIZE) | CS7;

			p->newt.c_cc[VEOL]	= 0x0D;

			set_low_latency(p->fd, 1);

		break;
		case RS_FLAG_RTU_MODBUS:
			p->newt.c_lflag = 0;
			p->newt.c_cflag = (p->newt.c_cflag & ~CSIZE) | CS8;

			memset(p->newt.c_cc, 0, NCCS);

			p->newt.c_cc[VTIME]	= 2;
			p->newt.c_cc[VMIN]	= MODB_RTU_MAX_PACKET_SIZE - 1; 

			set_low_latency(p->fd, 1);
		break;

		case RS_FLAG_INEUM:
			p->newt.c_lflag = 0;
			p->newt.c_cflag = (p->newt.c_cflag & ~CSIZE) | CS8;

			memset(p->newt.c_cc, 0, NCCS);

			p->newt.c_cc[VTIME]	= 1;
			p->newt.c_cc[VMIN]	= 4; 

			set_low_latency(p->fd, 1);

		break;
			
		default:
			break;
	}

	tcsetattr(p->fd, TCSANOW, &p->newt);
 	tcflush(p->fd, TCIOFLUSH);

	DropRTS( p->fd ); /* FIXME: where it must really be, after or before tcsetattr */

	return (0);
}

int
rs_set_network_mode_3544(int fd, unsigned short mode)
{
	unsigned short arg = mode;
	int res;
	
	res = ioctl(fd, IOCTL_3544_MODE_SET, &arg);

	return (res);
}

#define MOD_EXTENSION ".o"

#if 0
int
rs_load_module(const char *name) 

	int ps, tmp;

	/* FIXME: C99 extension, try to avoid at any cost */
	char module[ strlen(name) + strlen(MOD_EXTENSION) + 1 ]; 
	fprintf(stderr, "Loading driver: %s\n", name);

   	/* Fork and try to install module */
	ps = fork();

	if (ps == 0) { /* Child */
		
#if 0 /* Don't use it, only for historical reasons here */
		fputs("Trying modprobe...\n", stderr);

		if ( execl("modporbe", "modprobe", name, "2>/dev/null", "1>&2", NULL) ) {
		}
#endif
	
		/* add .o extension */
		strncpy(module, name, sizeof module);
		strncat(module, MOD_EXTENSION, sizeof module);
#if 0
		fprintf(stderr, "module: [%s]\n", module);
#endif

		fputs("Trying insmod...\n", stderr);
		if ( execl("insmod", "insmod", module, "2>/dev/null", "1>&2", NULL) ) {
		 	exit(2);
		}
	}
 
  	waitpid(ps, &tmp, 0); /* waiting for kid */

	fprintf(stderr, "Driver install...\t%s\n", (!tmp) ? "ok" : "failed" );
 
	return (tmp);
}

#endif

void 
rs_print_port_list(FILE *fp)
{
	int i;

	for(i = 0; i < N_ELS(rsps); i++) {
		fprintf(fp, "  %2d %s\n", 
				i + 1, 
				rsps[i].name);
	}
}

void 
rs_print_port_predef_conf(FILE *fp)
{
	int i,
		data,
		baud,
		parity = '\0';

	fputc('\n', stdout);	
	for(i = 0; i < N_ELS(predef_port_cfg); i++) {
		
		switch(predef_port_cfg[i][CONF_BAUD_FLAG]) {
			case B9600:
				baud = 9600;
				break;
			case B19200:
				baud = 19200;
				break;
			case B38400:
				baud = 38400;
				break;
			case B57600:
				baud = 57600;
			case B115200:
				baud = 115200;
			default:
				baud = 0;
		}
		
		switch(predef_port_cfg[i][CONF_DATA]) {
			case CS5:
				data = 5;
				break;
			case CS6:
				data = 6;
				break;
			case CS7:
				data = 7;
				break;
			case CS8:
				data = 8;
				break;
			default:
				data = 0;
				break;
		}

		switch(predef_port_cfg[i][CONF_PARITY]) {
			case 0:
				parity = 'N';
				break;
			case 1:
				parity = 'O';
				break;
			case 2:
				parity = 'E';
				break;
			default:
				parity = 'U';
				break;
		}
		
		fprintf(fp, "  %2d %6d %2d%c%d\n",
				i + 1,
				baud, 
				data,
			  	parity,
			   	predef_port_cfg[i][CONF_STOPB]);
	}

}

int
rs_device_array_size(void)
{
	return N_ELS(rsps);
}

int
rs_conf_array_size(void) 
{
	return N_ELS(predef_port_cfg);
}

int
rs_get_predef_baud(int id)
{
	return predef_port_cfg[id][CONF_BAUD_FLAG];
}

int
rs_get_predef_data(int id)
{
	return predef_port_cfg[id][CONF_DATA];
}


int
rs_get_predef_parity(int id)
{
	return predef_port_cfg[id][CONF_PARITY];
}

int
rs_get_predef_stopb(int id)
{
	return predef_port_cfg[id][CONF_STOPB];
}


void
rs_delay_ms(long ms) 
{
		struct timespec ts;

		memset(&ts, 0, sizeof(struct timespec ));

		if ((ts.tv_nsec = ms * 1000000L) > 999999999L)
			ts.tv_nsec = 999999999L;

		while ( nanosleep(&ts, &ts) == -1 ) {
				if (errno != EINTR) break;
		}
}

int 
rs_wait_for_incoming_event(int fd, int rto_s, int rto_u)
{
	struct timeval to;
	fd_set rfd;
	
	memset(&to, 0, sizeof ( struct timeval ) );

	FD_ZERO(&rfd);
	FD_SET(fd, &rfd);
	to.tv_sec = rto_s;
	to.tv_usec = rto_u;
	
	/* fputs("select\n", stderr); */
	return select( fd + 1, &rfd, NULL, NULL, (rto_s || rto_u) ? &to : NULL );
}

int 
rs_get_response(Port *p, unsigned char *frame, int *len)
{
	register int n = 0;
	int l;
	
	/* 
	 * select returns when it's some data available on fd, 
	 * but it's possible that driver get only few bytes, 
	 * so it's better for us to wait for something really valuable
	 */
	rs_delay_ms(5); 	

	if ( ( n = read(p->fd, frame, SERIAL_BUFF_SIZE) ) == -1 ) {
		p->errCrit++;
		return (0);
	}
	
	p->bts_rcvd += n;
#if 0
	{
		int i;
		fprintf(stderr, "%s", "rin1:  ");
		for ( i = 0 ; i < n ; i++ ) {
			fprintf(stderr, "%02x%c", frame[ i ], (i == (n - 1)) ? '\n' : ' ');
		}
	}
#endif
	
	if ( n < SERIAL_MIN_OUT_FRAME ) {
		p->errIncmt++;
		return (0);
	}

	l = frame[2] & 0x1F;

	if ( l == n - SERIAL_MIN_OUT_FRAME ) {
		if ( Gen_Chksum(frame, n - 1) == frame[n - 1] ) {
			*len = n;
			return (1);
		} else {
			p->errChksum++;
		}
	} else {
		p->errLen++;
	}
	
	*len = 0;
	return (0);
}

/**
 * rs_send -	sends bytes array to port, which is represented by 
 *				Port structure. It's added mostly for modbus test with 
 *				hope that it would be helpful someware else.
 *
 *	TODO: Add retries
 */
int 
rs_send( Port *p, unsigned char *frame, size_t len)
{
	int res = 0;
	

	/* Added from rs_snd_rcv, look for newer version
	 * and read comments there */

	/* Just for case. Do we really need it? */
	/* removed 25.11.04: */
	/* Added again 08.12.04
	 * We need to clear out garbage/noise from io buffers before send/recieve session.
	 * It seems important at hight boud rates (from 19200 and upper).
	 */
	tcflush( p->fd, TCIOFLUSH );
	RaiseRTS(p->fd); 




	if ( (res = write(p->fd, frame, len)) == -1 ) {
		/* Wait for all output to be transmitted 
		 * (is it really usefull in case of error?) */
#ifdef CONTROL_RTS
		tcdrain(p->fd);
		DropRTS(p->fd); 
#endif
		p->errSnd++;
		return (-1);
	}
	
#if 0 /** Not needed with patched kernel */
#ifdef CONTROL_RTS
	tcdrain(p->fd);
	DropRTS(p->fd);	
/*	tcflush(p->fd, TCIFLUSH ); */  /* clear incoming queue */
#endif
#endif

	p->bts_snd += res;
	return (res);
}

/**
 * rs_recv -	reads bytes array from port, which is represented by 
 *				Port structure. I's added mostly for modbus test with 
 *				hope that it would be helpful someware else.
 *
 *	TODO: Add retries
 */
int 
rs_recv( Port *p, unsigned char *frame, size_t len)
{
	/* From rs_get_response!!! Unused here
	 * select returns when it's some data available on fd, 
	 * but it's possible that driver get only few bytes, 
	 * so it's better for us to wait for something really valuable
	 */
	/* rs_delay_ms(5); 	*/

	if ( ( len = read(p->fd, frame, len) ) == -1 ) {
		if (errno!=EINTR)
			p->errRcv++;
		return (-1);
	}
	
	p->bts_rcvd += len;

	return (len);
}

int 
rs_snd_rcv( Port *p, unsigned char *frame, unsigned int *len)
{
	register int count = RETRIES;
	unsigned char *sendb = frame;
	int n = *len;
	int res = 0;
	

	/** Sanity checks */
	if (frame == NULL || len == NULL || p == NULL ) {
		errno = EFAULT;
		return (-1);
	}

	if ( *len > SERIAL_BUFF_SIZE - 1 || *len == 0 ) { /* One byte reserved for chksum */
		errno = ERANGE;
		return (-1);
	}

	/* Just for case. Do we really need it? */
	/* removed 25.11.04: */
	/* Added again 08.12.04
	 * We need to clear out garbage/noise from io buffers before send/recieve session.
	 * It seems important at hight boud rates (from 19200).
	 */
	tcflush( p->fd, TCIOFLUSH );

	sendb[n] = Gen_Chksum(sendb, n);
	n++;

#if 0
	{
		int i;
		fprintf(stderr, "%s", "out: ");
		for ( i = 0 ; i < n ; i++ ) {
			fprintf(stderr, "%02x%c", sendb[ i ], (i == (n - 1)) ? '\n' : ' ');
		}
	}
#endif

	while (count--) {
		
#ifdef CONTROL_RTS
		/** Removed for Adlink 3544 **/
		RaiseRTS(p->fd); 		
#endif

		if ( (res = write(p->fd, sendb, n)) == -1 ) {
#ifdef CONTROL_RTS
			tcdrain(p->fd);
			DropRTS(p->fd);
#endif
			p->errSnd++;
			return (-1);
		}


		/** 
		 * Whait while output buffer(s) being transmited. Not really needed
		 * when using blocking write. Just for case here. */
		/** Removed for Adlink 3544 */
#ifdef CONTROL_RTS
		tcdrain(p->fd);
		/** Removed for Adlink 3544 */
		DropRTS(p->fd);
#endif

		*len = 0;
		p->bts_snd += res;
	
		if ( ( res = rs_wait_for_incoming_event(p->fd, 0, RTO) ) == -1 && errno == EINTR ) { /* signal, try again FIXME: not working */
			count++;
			continue;
		} else if ( res && rs_get_response(p, frame, len)) 
			break;
		
		p->errTimeout++;
		tcflush(p->fd, TCIFLUSH ); /* clear incoming queue */
	}
	
	return *len;
}

#ifdef INEUM_PROT
/* 
 * changes:
 *
 *	14.08.03
 *
 *	1. This dummy wait for xmiter is very time consuming, try to avoid it on 
 *	slow machins tcdrain seems first not to relable, but after testing it 
 *	show best results for sequence of frames from host. And by the way - it 
 *	is more standart way. But on resonable fast machines it may be more 
 *	sofisticated way to use dummy wait loops
 *
 *	3. result == len statement was changed to len < result
 *
 *	25.08.03
 *
 *	Adjustment of port parameters, as result now working stable with Win32 
 *	client version and don't working with Dos version.
 *
 *	12.11.03
 *
 *	Changed function prototype.
 *
 */
int 
rs_rcv_snd( Port *p) 
{
    unsigned char len, buffer[ SERIAL_BUFF_SIZE ], chksum;
    int result;
	int flags = 0;

    if ( p == NULL ) return -1;

    result = read( p->fd, buffer, SERIAL_BUFF_SIZE );
	p->bts_rcvd += result;

#if 0
	{
		int i;
		fprintf(stderr, "%s", "rin:  ");
		for ( i = 0 ; i < result ; i++ ) {
			fprintf(stderr, "%02x%c", buffer[ i ], (i == (result - 1)) ? '\n' : ' ');
		}
	}
#endif

    /*	Three cases:
     *
     *	N - result.
     * 
     *	1. N >= SERIAL_MIN_IN_FRAME
     *	    We must check if the lenth and chksum is correct.
     *	    
     *	2. 0 < N < SERIAL_MIN_IN_FRAME
     *	    Incomplete header. 
     *	    Must read remaining chars, or just try to read.
     *	    
     *	3. N == -1
     *	    Read error - critical, we must return with error status.
     *
     */
    if ( result >= SERIAL_MIN_IN_FRAME ) { /* BINGO ! we got something useful ? */
	    len = ( buffer[2] & 0x1F ) + 4;
		if ( len <= result ) {
			chksum = Gen_Chksum( buffer, len - 1 );
			if ( chksum == buffer[ len - 1 ] ) { /* evrything is ok */
		    	p->pts_rcvd++;
				if ( p->proc_req != NULL && p->proc_req( buffer, &result, &flags, p) ) {
					
					rs_delay_ms(5);
					
					RaiseRTS( p->fd ); 
					if ( ( result = write( p->fd, buffer, result ) ) == -1 ) {
						tcdrain(p->fd);
						DropRTS( p->fd );
						p->errSnd++;
						return -1;
					}
					tcdrain(p->fd);
					DropRTS( p->fd );

					p->pts_snd++;
					p->bts_snd += result;
					
					/* check for global flags */
					if ( ISSET_REBOOT(flags) ) {
						/* Reset */
						(void) sync();
						(void) sync();
						(void) reboot( RB_AUTOBOOT );
					}
				}
			} else 	p->errChksum++; /* chksum == buffer [ len - 1 ] */
		} else p->errIncmt++; /* len == result */
	} else p->errCrit++;

    return result;
}

int
rs_proc_request(unsigned char *frame, int *len, int *flags, void *arg)
{
	int Command, 
		Address, 
		SeqNum, 
		Module_Address;

	unsigned char *XmitBuffer = frame;
	Port *p = arg;

	unsigned long stat = 0,
		XmitBufferCount = 0;

	if ( frame[0] == SLAVE_MODULE ) {

	Module_Address = frame[ 1 ];
	SeqNum	= frame[ 2 ] >> 5;
	Command = frame[ 3 ];
	Address = frame[ 4 ];

	XmitBuffer[0] = 0x00; /* RS NETWORK ADDRESS	HOST = 0 */
	XmitBuffer[1] = SLAVE_MODULE; /* TODO: ??? */

	/* TODO: dummy command parser - emulate few KP commands !!! */
	switch ( Command ) {
		case SET_COMMAND_BVCI:
			XmitBuffer[2] = (SeqNum << 5) + 1; /* size of answer only, without header and chksum, AND SeqNum */
			XmitBuffer[3] = STATUS_SET_COMMAND_BVCI;
			*flags |= FL_REBOOT;
			XmitBufferCount = 5;
			break;
		case READ_ID:
//			rs_delay_ms( 10 /* ms */ ); /* Only for test, must be removed */
			XmitBufferCount = 5;
			XmitBuffer [2] =  (SeqNum << 5) + 1;
			XmitBuffer [3] = STATUS_READ_ID;
			break;
		case READ_STAT:
			XmitBufferCount = 9;
			XmitBuffer [2] =  (SeqNum << 5) + 5; //5 data bytes
			XmitBuffer [3] = STATUS_READ_STAT;

			if (Address == 1 ) stat = p->errIncmt; /* FrameBadCount; */
			if (Address == 2 ) stat = p->errChksum;  /* ChksumBadCount; */
			if (Address == 3 ) stat = Address;  /* SegNumBadCount; */
			if (Address == 4 ) stat = p->errNoHost; /* NoHostCadrCount; */
			if (Address == 5 ) stat = Address;  /* NoEnoughRoomCount; */
			if (Address == 6 ) stat = p->pts_rcvd;  /* PacketCount; */
			if (Address == 7 ) stat = Address;  /* AsyncPacketsRcvd; */
			if (Address == 8 ) stat = Address;  /* AsyncPacketsSend; */
			if (Address == 9 ) stat = Address;  /* AnalogError; */
			if (Address == 10) stat = Address;  /* AnalogErrorLast; */
			if (Address == 11) stat = Address;  /* NoEnoughRoomCount2; */
			if (Address == 12) stat = Address;  /* PacketCount2; */
			if (Address == 13) stat = Address;  /* AsyncPacketsRcvd2; */
			if (Address == 14) stat = Address;  /* AsyncPacketsSend2; */
			
			XmitBuffer [7] = stat;
			XmitBuffer [6] = stat >> 8;
			XmitBuffer [5] = stat >> 16;
			XmitBuffer [4] = stat >> 24;
		break;
		 case READ_DISCRET_32:
                             XmitBufferCount = 9;
                             XmitBuffer [2] =  (SeqNum << 5) + 5; //5 data bytes
                             XmitBuffer [3] = STATUS_READ_DISCRET_32;
                             XmitBuffer [7] = 0xAA;
                             XmitBuffer [6] = 0x55;
                             XmitBuffer [5] = 0xAA;
                             XmitBuffer [4] = 0x55;
       break;

		case READ_DISCRET: /* Used in WinCC test*/
			XmitBufferCount = 9;
			XmitBuffer [2] =  (SeqNum << 5) + 5; //4 data bytes
			XmitBuffer [3] = STATUS_READ_DISCRET;
#if 0
			{

				static int imit_flag = 0;
			imit_flag = (imit_flag > 15) ? 0 : imit_flag + 1;
			msem_rs_mutex_lock(sigsh->msem.sid);
			XmitBuffer [4] = 0xAA >> imit_flag;
			XmitBuffer [5] = 0x55 >> imit_flag;
			XmitBuffer [6] = 0xAA >> imit_flag;
			XmitBuffer [7] = 0x55 >> imit_flag;
			msem_rs_mutex_unlock(sigsh->msem.sid);
			}
#else
			XmitBuffer [4] = 0x13;
			XmitBuffer [5] = 0x10;
			XmitBuffer [6] = 0x13;
			XmitBuffer [7] = 0x10;
#endif
			/* II original 
			XmitBuffer [4] = Module_Discret [Address] [1];
			XmitBuffer [5] = Module_Discret [Address] [2];
			XmitBuffer [6] = Module_Discret [Address] [3];
			XmitBuffer [7] = Module_Discret [Address] [4];
			*/
		break;

                               case READ_DISCRET_32_1:
                             XmitBufferCount = 9;
                             XmitBuffer [2] =  (SeqNum << 5) + 5; //4 data bytes
                             XmitBuffer [3] = STATUS_READ_DISCRET_32_1;
							 XmitBuffer [4] = 0xFF;
                             XmitBuffer [5] = 0xFF;
                             XmitBuffer [6] = 0xFF;
                             XmitBuffer [7] = 0xFF;
	/*
                             XmitBuffer [4] = Module_Discret [1] [1];
                             XmitBuffer [5] = Module_Discret [1] [2];
                             XmitBuffer [6] = Module_Discret [1] [3];
                             XmitBuffer [7] = Module_Discret [1] [4];
							*/	 
                               break;

                               case READ_DISCRET_32_2:
                             XmitBufferCount = 9+4;
                             XmitBuffer [2] =  (SeqNum << 5) + 5 + 4 ; //8
                             XmitBuffer [3] = STATUS_READ_DISCRET_32_2;
							 XmitBuffer [4] = 0xFF;
                             XmitBuffer [5] = 0xFF;
                             XmitBuffer [6] = 0xFF;
                             XmitBuffer [7] = 0xFF;
                             XmitBuffer [8] = 0xFF;
                             XmitBuffer [9] = 0xFF;
                             XmitBuffer [10] = 0xFF;
                             XmitBuffer [11] = 0xFF;

							 /*
                             XmitBuffer [4] = Module_Discret [1] [1];
                             XmitBuffer [5] = Module_Discret [1] [2];
                             XmitBuffer [6] = Module_Discret [1] [3];
                             XmitBuffer [7] = Module_Discret [1] [4];
                             XmitBuffer [8] =  Module_Discret [2] [1];
                             XmitBuffer [9] =  Module_Discret [2] [2];
                             XmitBuffer [10] = Module_Discret [2] [3];
                             XmitBuffer [11] = Module_Discret [2] [4];
							 */
                               break;


        case READ_DISCRET_32_4:
                             XmitBufferCount = 9+12;
                             XmitBuffer [2] = (SeqNum << 5) + 5 + 12 ; //16
                             XmitBuffer [3] = STATUS_READ_DISCRET_32_4;

							 XmitBuffer [4] = 0xFF;
                             XmitBuffer [5] = 0xFF;
                             XmitBuffer [6] = 0xFF;
                             XmitBuffer [7] = 0xFF;
                             XmitBuffer [8] = 0xFF;
                             XmitBuffer [9] = 0xFF;
                             XmitBuffer [10] = 0xFF;
                             XmitBuffer [11] = 0xFF;
                             XmitBuffer [12] = 0xFF;
                             XmitBuffer [13] = 0xFF;
                             XmitBuffer [14] = 0xFF;
                             XmitBuffer [15] = 0xFF;
                             XmitBuffer [16] = 0xFF;
                             XmitBuffer [17] = 0xFF;
                             XmitBuffer [18] = 0xFF;
                             XmitBuffer [19] = 0xFF;
/* original
                             XmitBuffer [4] = Module_Discret [1] [1];
                             XmitBuffer [5] = Module_Discret [1] [2];
                             XmitBuffer [6] = Module_Discret [1] [3];
                             XmitBuffer [7] = Module_Discret [1] [4];
                             XmitBuffer [8] =  Module_Discret [2] [1];
                             XmitBuffer [9] =  Module_Discret [2] [2];
                             XmitBuffer [10] = Module_Discret [2] [3];
                             XmitBuffer [11] = Module_Discret [2] [4];
                             XmitBuffer [12] =  Module_Discret [3] [1];
                             XmitBuffer [13] =  Module_Discret [3] [2];
                             XmitBuffer [14] =  Module_Discret [3] [3];
                             XmitBuffer [15] =  Module_Discret [3] [4];
                             XmitBuffer [16] =  Module_Discret [4] [1];
                             XmitBuffer [17] =  Module_Discret [4] [2];
                             XmitBuffer [18] =  Module_Discret [4] [3];
                             XmitBuffer [19] =  Module_Discret [4] [4];
							 */
        break;
		case READ_ANALOG:
			XmitBufferCount = 7;
			XmitBuffer [2] =  (SeqNum << 5) + 3; // 2 data
			XmitBuffer [3] = STATUS_READ_ANALOG;
#if 0
			{
			unsigned short val;
			if (Address == 0) break;
			msem_rs_mutex_lock(sigsh->msem.sid);
			val = getAnalogVal(sigsh, 0, Address - 1  );
			msem_rs_mutex_unlock(sigsh->msem.sid);
			XmitBuffer [4] = val;
			XmitBuffer [5] = val >> 8;
			}
#else
			XmitBuffer [4] = 0xFF;
			XmitBuffer [5] = 0xFF;
#endif

/*	Original
			XmitBuffer [4] = Module_Analog  [Address] [1];
			XmitBuffer [5] = Module_Analog  [Address] [2];
*/
		break;
		case READ_ANALOG_8:
                             XmitBufferCount = 7+14;
                             XmitBuffer [2] =  (SeqNum << 5) + 3 + 14; // 2 data
                             XmitBuffer [3] = STATUS_READ_ANALOG_8;
#if 0
							 { 
								 unsigned char *ptr =  (unsigned char *) getAnalogValPtr(sigsh, 0 );
							 msem_rs_mutex_lock(sigsh->msem.sid);
							 XmitBuffer [4] = *ptr;
                             XmitBuffer [5] = *(ptr + 1);
                             XmitBuffer [6] = *(ptr + 2);
                             XmitBuffer [7] = *(ptr + 3);
                             XmitBuffer [8] = *(ptr + 4);
                             XmitBuffer [9] = *(ptr + 5);
                             XmitBuffer [10] = *(ptr + 6);
                             XmitBuffer [11] = *(ptr + 7);
                             XmitBuffer [12] = *(ptr + 8);
                             XmitBuffer [13] = *(ptr + 9);
                             XmitBuffer [14] = *(ptr + 10);
                             XmitBuffer [15] = *(ptr + 11);
                             XmitBuffer [16] = *(ptr + 12);
                             XmitBuffer [17] = *(ptr + 13);
                             XmitBuffer [18] = *(ptr + 14);
                             XmitBuffer [19] = *(ptr + 15);
							 msem_rs_mutex_unlock(sigsh->msem.sid);
							 }
#else
						 	XmitBuffer [4] = 0xFF;
                             XmitBuffer [5] =0xFF;
                             XmitBuffer [6] =0xFF;
                             XmitBuffer [7] =0xFF;
                             XmitBuffer [8] =0xFF;
                             XmitBuffer [9] =0xFF;
                             XmitBuffer [10] =0xFF;
                             XmitBuffer [11] =0xFF;
                             XmitBuffer [12] =0xFF;
                             XmitBuffer [13] =0xFF;
                             XmitBuffer [14] =0xFF;
                             XmitBuffer [15] =0xFF;
                             XmitBuffer [16] =0xFF;
                             XmitBuffer [17] =0xFF;
                             XmitBuffer [18] =0xFF;
                             XmitBuffer [19] =0xFF;
#endif
	
/* II original
                             XmitBuffer [4] = Module_Analog [1] [1];
                             XmitBuffer [5] = Module_Analog [1] [2];
                             XmitBuffer [6] = Module_Analog [2] [1];
                             XmitBuffer [7] = Module_Analog [2] [2];
                             XmitBuffer [8] = Module_Analog [3] [1];
                             XmitBuffer [9] = Module_Analog [3] [2];
                             XmitBuffer [10] = Module_Analog [4] [1];
                             XmitBuffer [11] = Module_Analog [4] [2];
                             XmitBuffer [12] = Module_Analog [5] [1];
                             XmitBuffer [13] = Module_Analog [5] [2];
                             XmitBuffer [14] = Module_Analog [6] [1];
                             XmitBuffer [15] = Module_Analog [6] [2];
                             XmitBuffer [16] = Module_Analog [7] [1];
                             XmitBuffer [17] = Module_Analog [7] [2];
                             XmitBuffer [18] = Module_Analog [8] [1];
                             XmitBuffer [19] = Module_Analog [8] [2];
							 */
		break;
 
		default:
		XmitBufferCount = 0;
			break;
		}
	} else {
		p->errNoHost++;
	}

	if ( XmitBufferCount != 0 ) 
		XmitBuffer[ XmitBufferCount - 1 ] = Gen_Chksum( XmitBuffer, XmitBufferCount - 1 );

	*len = XmitBufferCount;

#if 0
	{
		int i;
		fprintf( stderr, "%s", "out: ");
		for ( i = 0 ; i < XmitBufferCount ; i++ ) {
			fprintf( stderr, "%02x%c", XmitBuffer[ i ], (i == (XmitBufferCount - 1)) ? '\n' : ' ');
		}
	}
#endif
	return 1;
}
#endif /** INEUM_PROT */

int
rs_init_port(Port *p, int num) 
{
   	const port_state *pps;
	
	pps = &rsps[ num ];

	p->num = num;
	/** Fill port structure. We are copying device filename to path field  */
	strncpy(p->path, pps->path, 
			(sizeof( p->path) >  strlen(pps->path)) ? strlen(pps->path) : sizeof( p->path) );
	
	/** If no module needed we are done */
	if (pps->module	== NULL) return (0);
	
#if 0 /** Commented out, because we don't deside from where to load modules */
	return rs_load_module(pps->module);
#endif 

	return (0);

}

#if 0
void 
dummy_my_sigtest(int arg) 
{
	fprintf(stderr, "Hello!!!\n");
	fflush(stderr);
	sleep(1);
	fprintf(stderr, "Hello!!!\n");
	//exit(233);
}
#endif

/* TODO: errors check */
int 
rs_open(Port *p, int num, int baud, int bytesize, int parity, int stopb, int flags)
{
	const port_state *pps;
	char  port_name[PATH_LENGTH] = {0};
	int   path_str_len = strlen(dev_path);

    if (p == NULL) return (-1);
    if ( num + 1 > sizeof rsps / sizeof rsps[ 0 ] || num < 0 ) return (-1);

	/* Clear Port structure, just for case... */
	memset(p, 0, sizeof (Port ) );
	pps = &rsps[ num ];
	
	/** Set `dev` prefix */
	if ( path_str_len > (sizeof (port_name) - 1) ) { /** Can't supress path, sanity check */
		return (-1);
	}

	strncpy(port_name, dev_path, path_str_len);

	if ( dev_path[ path_str_len - 1 ] != '/') { /** not ending with slash, append it */
		port_name[ path_str_len ] = '/';
		port_name[ path_str_len + 1] = '\0';
	} else {
		port_name[ path_str_len ] = '\0';
	}

	/** Append tty file names */
	strncat( port_name, pps->path, strlen(pps->path) );

	/** Fill port structure and try to load driver module if needed */
	if (rs_init_port(p, num)) {
		return (-1);
	}

#if 0
	fprintf( stderr, "Openning: %s...\n", port_name);
#endif

 	if ( ( p->fd = open( port_name, O_RDWR | O_NOCTTY | O_SYNC ) ) == -1 ) {
		return (-1);
    	}
	
   	/** Set port. Warning! Setting wrong port number can hang computer */
	if (pps->port) {
		if (rs_set_port(p->fd, pps->port)) { /* errno already set by ioctl */
			return (-1);
		}
	}
 

	/** Setup irq, if any */
	if ( pps->irq ) {
		if (rs_set_irq(p->fd, pps->irq)) { /* errno already set by ioctl */
			return (-1);
		}
#if 0
	fprintf(stderr, "Set IRQ num to %hd\n", pps->irq ); 
#endif
	}
	
	
	/** Setting port papameters */
	tcgetattr( p->fd, &p->oldt );
	memset( &p->newt, 0, sizeof( p->newt ) );

	switch(parity) {
		case 0: /* none */
			parity = 0;
			break;
		case 1: /* odd  */
			parity = PARENB | PARODD;
			break;
		case 2: /* even */
			parity = PARENB;
			break;
		default:
			parity = 0;
			break;
	}
	
	switch(stopb) {
		case 0:
			break;
		case 1:
			stopb = 0;
			break;
		case 2:
			stopb = CSTOPB;
			break;
		default:
			stopb = 0;
			break;
	}

	
	/* The |= operator useless for now, but in future can help to avoid silly bugs */
	p->newt.c_cflag &= ~( CSIZE | PARENB ); /* Useless if we fill structure with zeroes  */
	p->newt.c_cflag |= baud |/* CSIZE |*/ bytesize | parity | stopb | CLOCAL | CREAD;

    p->newt.c_iflag |= ( (parity) ? 0 : IGNPAR ); /* FIXME!!! */
    p->newt.c_oflag |= OPOST;
    p->newt.c_lflag &= ~( ECHO | ECHONL | ISIG ); /* Useless if we fill structure with zeroes  */

    /* 
     *	c_cc[VTIME] - time in ?deciseconds? to 
     *	wait before return from read.
     * 
     *	c_cc[VMIN] - bytes to recieve before return from read 
     *	current header size is SERIAL_MIN_IN_FRAME, so we must recieve at least 
     *	SERIAL_MIN_IN_FRAME bytes.
     */
	if (flags & RS_FLAG_ASCII_MODBUS) {

#if 1
		p->newt.c_lflag |=	ICANON | ICRNL;

		p->newt.c_cc[VEOL]	= 0x0A;
#endif
	} else {
		p->newt.c_cc[VTIME]	= 1;
		p->newt.c_cc[VMIN]	= SERIAL_MIN_IN_FRAME; 
	}


	set_low_latency( p->fd, 1);

        tcsetattr(p->fd, TCSANOW, &p->newt);
 	tcflush(p->fd, TCIOFLUSH);


#ifdef CONTROL_RTS
	DropRTS( p->fd ); /* FIXME: where it must really be, after or before tcsetattr */
#endif


    return p->fd;
}

int
rs_close(Port *p)
{
    if ( p == NULL ) return -1;
    
    tcsetattr( p->fd , TCSANOW, &p->oldt);
    return close( p->fd );
}
/*
 * changes:
 * 	14.08.03 - lsrtry now not used
 */
void
rs_print_stat(Port *p, FILE *fp)
{
	struct sysinfo sysi;
	
	if ( (p->fill_out > 0) && (p->fill_out < sizeof p->out)) {
		fputs("Last outcoming message:\n", fp);
		DUMP_BUFFER(p->out, p->fill_out);
	}

	if ( (p->fill_in > 0) && (p->fill_in < sizeof p->in)) {
		fputs("Last incoming message:\n", fp);
		DUMP_BUFFER(p->in, p->fill_in);
	}

	fprintf(fp, 
	       "\nSend      packets: %6d, %6d %s, errSnd: %4d\n",
		   p->pts_snd,  
		   (p->bts_snd > 1024) ? (p->bts_snd / 1024) : (p->bts_snd), 
		   (p->bts_snd > 1024) ? "kb" : "b",
		   p->errSnd );
	
  	fprintf(fp, 
		   "Recieved  packets: %6d, %6d %s, errRcv: %4d\n",
		   p->pts_rcvd, 
		   (p->bts_rcvd > 1024) ? (p->bts_rcvd / 1024) : (p->bts_rcvd), 
		   (p->bts_rcvd > 1024) ? "kb" : "b",
		   p->errRcv );
 
	fprintf( fp, 
		"errChksum: %4d errCmd:    %4d errTimeout: 	%4d  errNoASCII: %4d\n" \
		"errLen:    %4d errIncmt:  %4d errCrit:     %4d  errNoHost:  %4d\n" \
		"errStartCH: %4d\n",
		   p->errChksum, p->errCmd, p->errTimeout, p->errNoASCII,
		   p->errLen, p->errIncmt, p->errCrit, p->errNoHost,
		   p->errStartCH );
	
	memset(&sysi, 0, sizeof sysi);
	if (sysinfo(&sysi)) {
		fprintf(fp, "sysinfo: %s\n", strerror(errno));
	} else {
		fprintf(fp, "\nTime since boot:     %0ld sec\n", 
				sysi.uptime);
		fprintf(fp, "Load avarage:        %04lu %04lu %04lu\n", 
				sysi.loads[0], sysi.loads[1], sysi.loads[2] );
		fprintf(fp, "Number of processes: %0hd\n", 
				sysi.procs );
		fprintf(fp, "Free ram:            %0lu byte%c\n", 
				sysi.freeram, (sysi.freeram > 1) ? 's' : ' ');
	}

}

/* Swith ON/OFF RTS */
#if 0 /* Used dummy macros instead */
void
RaiseRTS(int fd)
{
	unsigned int arg;
	
	if (ioctl(fd, TIOCMGET, &arg) == -1) {
/*		return -1;*/
		return;
	}

	arg |= TIOCM_RTS;

	if (ioctl(fd, TIOCMSET, &arg) == -1) {
		return;
/*		return -1;*/
	}

/*	return 1;*/
} 

void
DropRTS(int fd)
{
	unsigned int arg;
	
	if (ioctl(fd, TIOCMGET, &arg) == -1) {
/*	    return -1;*/
		return;
	}
	
	arg &= ~TIOCM_RTS;

	if (ioctl(fd, TIOCMSET, &arg) == -1) {
		return;
/*	    return -1; */
	}

/*	return 1; */
}
#endif 

int 
set_low_latency( int fd, int on )
{
	struct serial_struct ss;

	if (ioctl(fd, TIOCGSERIAL, &ss) == -1) {
	    return (-1);
	}
	
	if (on) {
		ss.flags |= ASYNC_LOW_LATENCY;
	} else {
		ss.flags &= ~ASYNC_LOW_LATENCY;
	}

	if ( ioctl(fd, TIOCSSERIAL, &ss) == -1 ) {
	    return (-1);
	}

	return (0);
}

unsigned char
Gen_Chksum (unsigned char *Buffer, unsigned long Len)
{
    unsigned char x = 0;
    unsigned long i;

    for (i=0; i<Len; i++) {
        x += Buffer [i];
    }

    return (0x100 - x);
}

#if 0
static unsigned int
rs_xemty( int fd ) 
{
	unsigned int lsrarg = 0;

	if (	ioctl(fd, TIOCSERGETLSR, &lsrarg) == -1) {
		return 0;
	}

	return lsrarg;
}
#endif

int 
rs_set_irq(int fd, int irq)
{
	struct serial_struct ss = {0};

	if ( ioctl(fd, TIOCGSERIAL, &ss) == -1 )
		return (1);
	
	ss.irq = irq;
	
	if ( ioctl(fd, TIOCSSERIAL, &ss) == -1 )
		return (1);

	return (0);
}

int 
rs_set_port(int fd, unsigned int port)
{
	struct serial_struct ss = {0};

	if ( ioctl(fd, TIOCGSERIAL, &ss) == -1 )
		return (1);
	
	ss.port = port;
	
	if ( ioctl(fd, TIOCSSERIAL, &ss) == -1 )
		return (1);

	return (0);
}

