/* Apache interface for mod_caml programs.
 * Copyright (C) 2003 Merjis Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: apache_c.c,v 1.8 2004/02/21 16:49:28 rwmj Exp $
 */

#include "config.h"

#include <stdio.h>
#include <sys/stat.h>

#include <caml/alloc.h>
#include <caml/callback.h>
#include <caml/fail.h>
#include <caml/memory.h>
#include <caml/mlvalues.h>

#include <httpd.h>
#include <http_config.h>
#include <http_protocol.h>
#include <http_request.h>

#if APACHE2
#include <apr_strings.h>
#include <apr_file_info.h>
#include <apr_time.h>
#endif

#include "wrappers.h"

#if APACHE2
#define ap_table_get apr_table_get
#define ap_table_set apr_table_set
#define ap_table_unset apr_table_unset
#define ap_pstrdup apr_pstrdup
#define ap_palloc apr_palloc
#endif

extern module caml_module;

static void
raise_http_error (int i)
{
  raise_with_arg (*caml_named_value ("mod_caml_http_error"), Val_int (i));
}

static void
possible_http_error (int i)
{
  if (i != OK && i != DECLINED && i != DONE) raise_http_error (i);
}

/*----- Tables. -----*/

CAMLprim value
mod_caml_table_get (value tv, value str)
{
  CAMLparam2 (tv, str);
  table *t = Table_val (tv);
  const char *res = ap_table_get (t, String_val (str));
  if (res)
    CAMLreturn (copy_string (res));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_table_set (value tv, value key, value val)
{
  CAMLparam3 (tv, key, val);
  table *t = Table_val (tv);
  ap_table_set (t, String_val (key), String_val (val));
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_table_unset (value tv, value key)
{
  CAMLparam2 (tv, key);
  table *t = Table_val (tv);
  ap_table_unset (t, String_val (key));
  CAMLreturn (Val_unit);
}

/*----- Server structure. -----*/

CAMLprim value
mod_caml_server_hostname (value sv)
{
  CAMLparam1 (sv);
  server_rec *s = Server_rec_val (sv);
  if (s->server_hostname)
    CAMLreturn (copy_string (s->server_hostname));
  else
    raise_not_found ();
}

/*----- Connection structure. -----*/

/* nothing at the moment ... */

/*----- Request structure. -----*/

CAMLprim value
mod_caml_request_connection (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_conn_rec (r->connection));
}

CAMLprim value
mod_caml_request_server (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_server_rec (r->server));
}

CAMLprim value
mod_caml_request_next (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv), *rr = r->next;
  if (rr)
    CAMLreturn (Val_request_rec (rr));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_prev (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv), *rr = r->prev;
  if (rr)
    CAMLreturn (Val_request_rec (rr));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_main (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv), *rr = r->main;
  if (rr)
    CAMLreturn (Val_request_rec (rr));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_the_request (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  if (r->the_request)
    CAMLreturn (copy_string (r->the_request));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_assbackwards (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_bool (r->assbackwards));
}

CAMLprim value
mod_caml_request_header_only (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_bool (r->header_only));
}

CAMLprim value
mod_caml_request_protocol (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  if (r->protocol)
    CAMLreturn (copy_string (r->protocol));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_proto_num (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_int (r->proto_num));
}

CAMLprim value
mod_caml_request_hostname (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  if (r->hostname)
    CAMLreturn (copy_string (r->hostname));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_request_time (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (copy_double ((double) r->request_time));
}

CAMLprim value
mod_caml_request_method (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (copy_string (r->method));
}

CAMLprim value
mod_caml_request_method_number (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_int (r->method_number));
}

CAMLprim value
mod_caml_request_headers_in (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_table (r->headers_in));
}

CAMLprim value
mod_caml_request_headers_out (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_table (r->headers_out));
}

CAMLprim value
mod_caml_request_err_headers_out (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_table (r->err_headers_out));
}

CAMLprim value
mod_caml_request_subprocess_env (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_table (r->subprocess_env));
}

CAMLprim value
mod_caml_request_notes (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_table (r->notes));
}

CAMLprim value
mod_caml_request_content_type (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  if (r->content_type)
    CAMLreturn (copy_string (r->content_type));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_set_content_type (value rv, value str)
{
  CAMLparam2 (rv, str);
  request_rec *r = Request_rec_val (rv);
  r->content_type = ap_pstrdup (r->pool, String_val (str));
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_user (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
#if APACHE2
  if (r->user)
    CAMLreturn (copy_string (r->user));
#else
  if (r->connection->user)
    CAMLreturn (copy_string (r->connection->user));
#endif
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_uri (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  if (r->uri)
    CAMLreturn (copy_string (r->uri));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_set_uri (value rv, value str)
{
  CAMLparam2 (rv, str);
  request_rec *r = Request_rec_val (rv);
  r->uri = ap_pstrdup (r->pool, String_val (str));
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_filename (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  if (r->filename)
    CAMLreturn (copy_string (r->filename));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_set_filename (value rv, value str)
{
  CAMLparam2 (rv, str);
  request_rec *r = Request_rec_val (rv);
  r->filename = ap_pstrdup (r->pool, String_val (str));
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_path_info (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  if (r->path_info)
    CAMLreturn (copy_string (r->path_info));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_set_path_info (value rv, value str)
{
  CAMLparam2 (rv, str);
  request_rec *r = Request_rec_val (rv);
  r->path_info = ap_pstrdup (r->pool, String_val (str));
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_args (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  if (r->args)
    CAMLreturn (copy_string (r->args));
  else
    raise_not_found ();
}

CAMLprim value
mod_caml_request_set_args (value rv, value str)
{
  CAMLparam2 (rv, str);
  request_rec *r = Request_rec_val (rv);
  r->args = ap_pstrdup (r->pool, String_val (str));
  CAMLreturn (Val_unit);
}

#if APACHE2
static int file_kind_table[] = {
  APR_REG, APR_DIR, APR_CHR, APR_BLK, APR_LNK, APR_PIPE, APR_SOCK
};
#else
static int file_kind_table[] = {
  S_IFREG, S_IFDIR, S_IFCHR, S_IFBLK, S_IFLNK, S_IFIFO, S_IFSOCK
};
#endif

static value
cst_to_constr (int n, int *tbl, int size, int deflt)
{
  int i;
  for (i = 0; i < size; i++)
    if (n == tbl[i]) return Val_int(i);
  return Val_int(deflt);
}

CAMLprim value
mod_caml_request_finfo (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLlocal5 (v, sb, atime, mtime, ctime);

#if APACHE2
  if (r->finfo.filetype != APR_NOFILE) /* Some statbuf */
    {
      atime = (r->finfo.valid & APR_FINFO_ATIME) ?
	copy_double ((double) apr_time_sec (r->finfo.atime)) :
	copy_double (0.);
      mtime = (r->finfo.valid & APR_FINFO_MTIME) ?
	copy_double ((double) apr_time_sec (r->finfo.mtime)) :
	copy_double (0.);
      ctime = (r->finfo.valid & APR_FINFO_CTIME) ?
	copy_double ((double) apr_time_sec (r->finfo.ctime)) :
	copy_double (0.);

      sb = alloc_small (12, 0);
      Field (sb, 0) = Val_int (r->finfo.device);
      Field (sb, 1) = Val_int (r->finfo.inode);
      Field (sb, 2) =
	cst_to_constr (r->finfo.filetype, file_kind_table,
		       sizeof (file_kind_table) / sizeof (int), 0);
      Field (sb, 3) = Val_int (r->finfo.protection);
      Field (sb, 4) = Val_int (r->finfo.nlink);
      Field (sb, 5) = Val_int (r->finfo.user);
      Field (sb, 6) = Val_int (r->finfo.group);
      Field (sb, 7) = Val_int (0); /* XXX rdev? */
      Field (sb, 8) = Val_int (r->finfo.size); /* XXX 64 bit file offsets */

      Field (sb, 9) = atime;
      Field (sb, 10) = mtime;
      Field (sb, 11) = ctime;

      v = alloc (1, 0);		/* The "Some" block. */
      Field (v, 0) = sb;
    }
else
    v = Val_int (0);		/* None. */

#else /* not APACHE2 */

  if (r->finfo.st_mode)		/* Some statbuf */
    {
      /* This code copied and modified from otherlibs/unix/stat.c. */
      atime = copy_double ((double) r->finfo.st_atime);
      mtime = copy_double ((double) r->finfo.st_mtime);
      ctime = copy_double ((double) r->finfo.st_ctime);

      sb = alloc_small (12, 0);
      Field (sb, 0) = Val_int (r->finfo.st_dev);
      Field (sb, 1) = Val_int (r->finfo.st_ino);
      Field (sb, 2) =
	cst_to_constr (r->finfo.st_mode & S_IFMT, file_kind_table,
		       sizeof (file_kind_table) / sizeof (int), 0);
      Field (sb, 3) = Val_int (r->finfo.st_mode & 07777);
      Field (sb, 4) = Val_int (r->finfo.st_nlink);
      Field (sb, 5) = Val_int (r->finfo.st_uid);
      Field (sb, 6) = Val_int (r->finfo.st_gid);
      Field (sb, 7) = Val_int (r->finfo.st_rdev);
      Field (sb, 8) = Val_int (r->finfo.st_size); /* XXX 64 bit file offsets */
      Field (sb, 9) = atime;
      Field (sb, 10) = mtime;
      Field (sb, 11) = ctime;

      v = alloc (1, 0);		/* The "Some" block. */
      Field (v, 0) = sb;
    }
  else
    v = Val_int (0);		/* None. */
#endif /* not APACHE2 */

  CAMLreturn (v);
}

CAMLprim value
mod_caml_request_send_http_header (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
#if APACHE2
  /* XXX do nothing in Apache 2.x? */
#else
  ap_send_http_header (r);
#endif
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_setup_client_block (value rv, value rp)
{
  CAMLparam2 (rv, rp);
  request_rec *r = Request_rec_val (rv);
  int i = ap_setup_client_block (r, Int_val (rp));
  possible_http_error (i);
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_should_client_block (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  CAMLreturn (Val_bool (ap_should_client_block (r)));
}

CAMLprim value
mod_caml_request_get_client_block (value rv)
{
  CAMLparam1 (rv);
  CAMLlocal1 (str);
  request_rec *r = Request_rec_val (rv);
  const int huge_string_len = 8192; /* Same as Apache's HUGE_STRING_LEN. */
  char buffer [huge_string_len];
  int i;

  str = Val_unit;
  i = ap_get_client_block (r, buffer, huge_string_len);
  if (i == -1) raise_http_error (HTTP_INTERNAL_SERVER_ERROR);

  str = alloc_string (i);
  memcpy (String_val (str), buffer, i);

  CAMLreturn (str);
}

CAMLprim value
mod_caml_request_discard_request_body (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  int i = ap_discard_request_body (r);
  possible_http_error (i);
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_note_auth_failure (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  ap_note_auth_failure (r);
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_note_basic_auth_failure (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  ap_note_basic_auth_failure (r);
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_note_digest_auth_failure (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  ap_note_digest_auth_failure (r);
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_get_basic_auth_pw (value rv)
{
  CAMLparam1 (rv);
  request_rec *r = Request_rec_val (rv);
  const char *pw = 0;
  int i = ap_get_basic_auth_pw (r, &pw);
  possible_http_error (i);
  if (i == DECLINED) pw = 0;	/* XXX */
  CAMLreturn (Val_optstring (pw));
}

CAMLprim value
mod_caml_request_internal_redirect (value new_uri, value rv)
{
  CAMLparam2 (new_uri, rv);
  request_rec *r = Request_rec_val (rv);
  ap_internal_redirect (String_val (new_uri), r);
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_internal_redirect_handler (value new_uri, value rv)
{
  CAMLparam2 (new_uri, rv);
  request_rec *r = Request_rec_val (rv);
  ap_internal_redirect_handler (String_val (new_uri), r);
  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_print_char (value rv, value cv)
{
  CAMLparam2 (rv, cv);
  request_rec *r = Request_rec_val (rv);
  int c = Int_val (cv);
  int i = ap_rputc (c, r);

  if (i == -1)
    failwith ("mod_caml_print_char: failed"); /* XXX */

  CAMLreturn (Val_unit);
}

CAMLprim value
mod_caml_request_print_string (value rv, value strv)
{
  CAMLparam2 (rv, strv);
  request_rec *r = Request_rec_val (rv);
  const char *str = String_val (strv);
  int n = string_length (strv);
  int i = ap_rwrite (str, n, r);

  if (i < n)
    failwith ("mod_caml_print_string: failed"); /* XXX */

  CAMLreturn (Val_unit);
}

static int
run_cleanup (void *fv)
{
  value f = *(value *) fv;

  callback (f, Val_unit);
  remove_global_root ((value *) fv);
  return OK;
}

CAMLprim value
mod_caml_request_register_cleanup (value rv, value f)
{
  CAMLparam2 (rv, f);
  request_rec *r = Request_rec_val (rv);
  value *v = (value *) ap_palloc (r->pool, sizeof (value));

  *v = f;
  register_global_root (v);

#if APACHE2
  apr_pool_cleanup_register (r->pool, v, run_cleanup, apr_pool_cleanup_null);
#else
  ap_register_cleanup (r->pool, v,
		       (void (*)(void *)) run_cleanup, ap_null_cleanup);
#endif

  CAMLreturn (Val_unit);
}

/*----- Miscellaneous functions. -----*/

CAMLprim value
mod_caml_get_server_config (value rv)
{
  CAMLparam1 (rv);
  CAMLlocal1 (config);
  request_rec *r = Request_rec_val (rv);
  config =
    *(value *) ap_get_module_config (r->server->module_config, &caml_module);
  CAMLreturn (config);
}

CAMLprim value
mod_caml_get_dir_config (value rv)
{
  CAMLparam1 (rv);
  CAMLlocal1 (config);
  request_rec *r = Request_rec_val (rv);
  config = *(value *) ap_get_module_config (r->per_dir_config, &caml_module);
  CAMLreturn (config);
}
