(* ML part of mod_caml.
 * Copyright (C) 2003 Merjis Ltd.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: mod_caml.ml,v 1.5 2004/01/26 15:33:55 rwmj Exp $
 *)

(* Note a peculiarity of Apache: it runs all initialization TWICE. See this
 * page for an explanation of this feature:
 * http://thingy.kcilink.com/modperlguide/config/
 *   Apache_Restarts_Twice_On_Start.html
 * The upshot is that the configuration code here will be called twice.
 *)

open Mod_caml_config
open Apache

type server_config_t = {
    mutable translate_handler : handler_t option;
  }

type dir_config_t = {
    mutable location : string option;
    mutable check_user_id : handler_t option;
    mutable auth_checker : handler_t option;
    mutable access_checker : handler_t option;
    mutable type_checker : handler_t option;
    mutable fixer_upper : handler_t option;
    mutable logger : handler_t option;
    mutable header_parser : handler_t option;
    mutable post_read_request : handler_t option;
    mutable ocaml_bytecode_handler : handler_t option;
  }

(* Not very useful - gets called later than you would expect. *)
let module_init () =
  ()

(*----- Initialize Dynlink library. -----*)

let () =
  try
    Dynlink.init ();
    Dynlink.allow_unsafe_modules true
  with
    Dynlink.Error (e) ->
      failwith (Dynlink.error_message e)

(*----- Configuration. -----*)

let create_dir_config dirname =
  { location = dirname;
    check_user_id = None;
    auth_checker = None;
    access_checker = None;
    type_checker = None;
    fixer_upper = None;
    logger = None;
    header_parser = None;
    post_read_request = None;
    ocaml_bytecode_handler = None }

let merge_dir_config base add =
  let merged = base in
  if add.location <> None then merged.location <- add.location;
  if add.check_user_id <> None then
    merged.check_user_id <- add.check_user_id;
  if add.auth_checker <> None then
    merged.auth_checker <- add.auth_checker;
  if add.access_checker <> None then
    merged.access_checker <- add.access_checker;
  if add.type_checker <> None then
    merged.type_checker <- add.type_checker;
  if add.fixer_upper <> None then
    merged.fixer_upper <- add.fixer_upper;
  if add.logger <> None then
    merged.logger <- add.logger;
  if add.header_parser <> None then
    merged.header_parser <- add.header_parser;
  if add.post_read_request <> None then
    merged.post_read_request <- add.post_read_request;
  if add.ocaml_bytecode_handler <> None then
    merged.ocaml_bytecode_handler <- add.ocaml_bytecode_handler;
  merged

let create_server_config s =
  { translate_handler = None; }

let merge_server_config base add =
  let merged = base in
  if add.translate_handler <> None then
    merged.translate_handler <- add.translate_handler;
  merged

external get_server_config : Request.t -> server_config_t
    = "mod_caml_get_server_config"

external get_dir_config : Request.t -> dir_config_t
    = "mod_caml_get_dir_config"

(*----- Handlers. -----*)

let translate_handler r =
  let config = get_server_config r in
  match config.translate_handler with
    Some handler -> handler r
  | None -> DECLINED

let check_user_id r =
  let config = get_dir_config r in
  match config.check_user_id with
    Some handler -> handler r
  | None -> DECLINED

let auth_checker r =
  let config = get_dir_config r in
  match config.auth_checker with
    Some handler -> handler r
  | None -> DECLINED

let access_checker r =
  let config = get_dir_config r in
  match config.access_checker with
    Some handler -> handler r
  | None -> DECLINED

let type_checker r =
  let config = get_dir_config r in
  match config.type_checker with
    Some handler -> handler r
  | None -> DECLINED

let fixer_upper r =
  let config = get_dir_config r in
  match config.fixer_upper with
    Some handler -> handler r
  | None -> DECLINED

let logger r =
  let config = get_dir_config r in
  match config.logger with
    Some handler -> handler r
  | None -> DECLINED

let header_parser r =
  let config = get_dir_config r in
  match config.header_parser with
    Some handler -> handler r
  | None -> DECLINED

let post_read_request r =
  let config = get_dir_config r in
  match config.post_read_request with
    Some handler -> handler r
  | None -> DECLINED

let ocaml_bytecode_handler r =
  let config = get_dir_config r in
  match config.ocaml_bytecode_handler with
    Some handler -> handler r
  | None -> DECLINED

(*----- Handler registration. -----*)

let reg_table = Hashtbl.create 16

let reg_module_name = ref None

(* Register the module's handler. *)
let register_handler handler name =
  match !reg_module_name with
    None -> failwith ("Mod_caml.register_handler call outside module " ^
		      "initialization")
  | Some module_name ->
      Hashtbl.replace reg_table (module_name ^ "." ^ name) handler

(*----- Commands. -----*)

let cmd_load filename =
  reg_module_name :=
    Some (String.capitalize (Filename.chop_extension
			       (Filename.basename filename)));
  try
    Dynlink.loadfile filename;
  with
    Dynlink.Error (e) ->
      failwith (Dynlink.error_message e)

let cmd_translate_handler sconfig name =
  sconfig.translate_handler <- Some (Hashtbl.find reg_table name)

let cmd_check_user_id_handler dconfig name =
  dconfig.check_user_id <- Some (Hashtbl.find reg_table name)

let cmd_auth_checker_handler dconfig name =
  dconfig.auth_checker <- Some (Hashtbl.find reg_table name)

let cmd_access_checker_handler dconfig name =
  dconfig.access_checker <- Some (Hashtbl.find reg_table name)

let cmd_type_checker_handler dconfig name =
  dconfig.type_checker <- Some (Hashtbl.find reg_table name)

let cmd_fixer_upper_handler dconfig name =
  dconfig.fixer_upper <- Some (Hashtbl.find reg_table name)

let cmd_logger_handler dconfig name =
  dconfig.logger <- Some (Hashtbl.find reg_table name)

let cmd_header_parser_handler dconfig name =
  dconfig.header_parser <- Some (Hashtbl.find reg_table name)

let cmd_post_read_request_handler dconfig name =
  dconfig.post_read_request <- Some (Hashtbl.find reg_table name)

let cmd_handler dconfig name =
  dconfig.ocaml_bytecode_handler <- Some (Hashtbl.find reg_table name)

(*----- Register functions. -----*)

let () =
  Callback.register "mod_caml_create_dir_config" create_dir_config;
  Callback.register "mod_caml_merge_dir_config" merge_dir_config;
  Callback.register "mod_caml_create_server_config" create_server_config;
  Callback.register "mod_caml_merge_server_config" merge_server_config;
  Callback.register "mod_caml_module_init" module_init;

  Callback.register "mod_caml_translate_handler" translate_handler;
  Callback.register "mod_caml_check_user_id" check_user_id;
  Callback.register "mod_caml_auth_checker" auth_checker;
  Callback.register "mod_caml_access_checker" access_checker;
  Callback.register "mod_caml_type_checker" type_checker;
  Callback.register "mod_caml_fixer_upper" fixer_upper;
  Callback.register "mod_caml_logger" logger;
  Callback.register "mod_caml_header_parser" header_parser;
  Callback.register "mod_caml_post_read_request" post_read_request;
  Callback.register "mod_caml_ocaml_bytecode_handler" ocaml_bytecode_handler;

  Callback.register "mod_caml_cmd_load" cmd_load;
  Callback.register "mod_caml_cmd_translate_handler" cmd_translate_handler;
  Callback.register "mod_caml_cmd_check_user_id_handler"
    cmd_check_user_id_handler;
  Callback.register "mod_caml_cmd_auth_checker_handler"
    cmd_auth_checker_handler;
  Callback.register "mod_caml_cmd_access_checker_handler"
    cmd_access_checker_handler;
  Callback.register "mod_caml_cmd_type_checker_handler"
    cmd_type_checker_handler;
  Callback.register "mod_caml_cmd_fixer_upper_handler"
    cmd_fixer_upper_handler;
  Callback.register "mod_caml_cmd_logger_handler" cmd_logger_handler;
  Callback.register "mod_caml_cmd_header_parser_handler"
    cmd_header_parser_handler;
  Callback.register "mod_caml_cmd_post_read_request_handler"
    cmd_post_read_request_handler;
  Callback.register "mod_caml_cmd_handler" cmd_handler;

  ()
