(**
  * Library for escaping and decoding.
  *
  * Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
  *
  * Copyright (C) 1997 Daniel de Rauglaudre, INRIA
  *
  * Copyright (C) 1998 Jean-Christophe FILLIATRE
  * 
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Library General Public
  * License as published by the Free Software Foundation; either
  * version 2 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  * Library General Public License for more details.
  *
  * You should have received a copy of the GNU Library General Public
  * License along with this library; if not, write to the Free
  * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  *
  * $Id: cgi_escape.mli,v 1.1 2004/02/26 12:10:51 rwmj Exp $
  *)

type escape_t = EscapeNone | EscapeUrl | EscapeHtml | EscapeHtmlTag
              | EscapeHtmlTextarea
(** Types of escaping performed by the escape function. *)

val escape : escape_t -> string -> string
(** Perform text escaping on the given string.
 *
 * [escape EscapeNone] leaves the string unchanged.
 *
 * [escape EscapeUrl] does URL-encoding, the same as [encode].
 *
 * [escape EscapeHtml] escapes [& < > "] with the appropriate
 * HTML escape sequence. It also turns end of lines into [<br>].
 *
 * [escape EscapeHtmlTag] escapes [& < > "] with the appropriate
 * HTML escape sequence. It also turns end of lines into [&#13;&#10].
 * This function is suitable when placing text between the quotes
 * in an HTML tag, eg. [<input value="text">].
 *
 * [escape EscapeHtmlTextarea] escapes [& < > "] with the appropriate
 * HTML escape sequence. It leaves everything else unchanged. This
 * is a suitable function to use when placing text in an HTML
 * [<textarea>...</textarea>].
 *
 * " <- keeps OCaml happy.
 *)

val escape_url : string -> string
(** Same as [escape EscapeUrl] and [encode]. *)
val escape_html : string -> string
(** Same as [escape EscapeHtml] *)
val escape_html_tag : string -> string
(** Same as [escape EscapeHtmlTag] *)
val escape_html_textarea : string -> string
(** Same as [escape EscapeHtmlTextarea] *)

val decode : string -> string
(** Decode URL-encoded string. Note that this is already done for you by
 * the functions in the [cgi] class.
 *)
val encode : string -> string
(** Encode URL-encoded string. *)
