(**
  * Library for parsing templates.
  *
  * Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Library General Public
  * License as published by the Free Software Foundation; either
  * version 2 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  * Library General Public License for more details.
  *
  * You should have received a copy of the GNU Library General Public
  * License along with this library; if not, write to the Free
  * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  *
  * $Id: template.mli,v 1.1 2004/02/26 12:10:51 rwmj Exp $
  *)

type table_row_t = (string * var_t) list
and var_t = VarString of string
            | VarTable of table_row_t list
	    | VarConditional of bool
(** Variables are either simple string ::tags::, tables or
  conditionals.  Tables are rows with each row being an assoc-list
  of name -> variable.  Calling the [template#set] method gives a variable a
  simple string value (ie. a VarString).  Calling [template#table]
  sets up the value of a whole table (ie. a VarTable).  Calling
  [template#conditional] sets up a conditional value (as used in
  ::if(...):: statements.  In future we may extend this type to provide
  things like closures (to allow for delayed evaluation for values
  which are expensive to compute).  *)

class template :
  string ->
object
  method set : string -> string -> unit
    (** Set a variable in the template. *)

  method table : string -> (string * var_t) list list -> unit
    (** Set a table in the template. *)

  method conditional : string -> bool -> unit
    (** Set a conditional in the template. *)

  method to_string : string
    (** Return the template as a string. *)

  method to_channel : out_channel -> unit
    (** Write the template to a channel. *)

  method source : string
    (** Return the original source code for the template. *)
end

val template : string -> template
  (** Compile the template from a named file. *)

val template_from_string : string -> template
  (** Compile the template from a literal string. *)

val template_from_channel : in_channel -> template
  (** Compile the template from a channel. *)
