(*
 * ocamlcgi - Objective Caml library for writing CGIs
 * Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
 * Copyright (C) 1997 Daniel de Rauglaudre, INRIA
 * Copyright (C) 1998 Jean-Christophe FILLIATRE
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: cgi.ml,v 1.15 2004/09/27 11:56:41 rwmj Exp $
 *)

open Apache
open Cgi_escape

(* Maximum size for POSTed requests. *)
let post_max = ref max_int

let set_post_max i =
  post_max := i

let post_max () =
  !post_max

let may f v = match v with None -> () | Some v -> f v

(* Split a string at the given separator character into multiple substrings. *)
let split separator text =
  let len = String.length text in
  let rec loop pos =
    if pos < len then
      try
       	let last = String.index_from text pos separator in
 	let str = String.sub text pos (last-pos) in
	  str::(loop (succ last))
      with Not_found ->
 	if pos < len then [String.sub text pos (len-pos)]
 	else []
    else []
  in
  loop 0

module Cookie =
  struct

    class cookie ~name ~value ~expires ~domain ~path ~secure =
    object (self)
      val mutable name = name
      val mutable value = value
      val mutable expires = expires
      val mutable domain = domain
      val mutable path = path
      val mutable secure = secure

      method name = name
      method value = value
      method expires = expires
      method domain = domain
      method path = path
      method secure = secure

      method set_name v = name <- v
      method set_value v = value <- v
      method set_expires v = expires <- v
      method set_domain v = domain <- v
      method set_path v = path <- v
      method set_secure v = secure <- v

      method as_string =
	let buff = Buffer.create 128 in
	Buffer.add_string buff name;
	Buffer.add_char buff '=';
	Buffer.add_string buff value;
	if expires <> "" then (
	  Buffer.add_string buff "; expires=";
	  Buffer.add_string buff expires
	);
	if path <> "" then (
	  Buffer.add_string buff "; path=";
	  Buffer.add_string buff path
	);
	if domain <> "" then (
	  Buffer.add_string buff "; domain=";
	  Buffer.add_string buff domain
	);
	if secure then
	  Buffer.add_string buff "; secure";
	Buffer.contents buff
    end

    let cookie ~name ~value ?(expires = "") ?(domain = "") ?(path = "")
      ?(secure = false) () =
      new cookie ~name ~value ~expires ~domain ~path ~secure

    let split_re = Pcre.regexp "; ?"

    let parse header =
      let cookies = Pcre.split ~rex:split_re header in
      List.map
	(fun s ->
	   let name, value =
	     try
	       let i = String.index s '=' in
	       String.sub s 0 i,
	       decode (String.sub s (i+1) (String.length s - i - 1))
	     with
		 Not_found -> s, "" in
	   cookie ~name ~value ()) cookies
  end

(* Read the full body of a POST request. *)
let get_post_body r =
  (* http://www.auburn.edu/docs/apache/misc/client_block_api.html *)
  Request.setup_client_block r Request.REQUEST_CHUNKED_ERROR;

  if Request.should_client_block r then (
    let buf = Buffer.create 8192 in
    let rec loop () =
      let s = Request.get_client_block r in
      if s <> "" then (
	Buffer.add_string buf s;
	if Buffer.length buf > post_max () then
	  raise (HttpError cHTTP_REQUEST_ENTITY_TOO_LARGE);
	loop ()
      )
    in
    loop ();
    Buffer.contents buf
  )
  else ""

(* Parsing of the CGI arguments. *)
module Cgi_args =
struct
  let parse s =
    let assocs = split '&' s in
    let one_assoc s =
      try
	let i = String.index s '=' in
	String.sub s 0 i, 
	decode (String.sub s (succ i) (String.length s - i - 1))
      with
	| Not_found -> s, ""
    in
    List.map one_assoc assocs
end

let parse_args r = 
  let req_method = Request.method_number r in
  let s = 
    match req_method with
	M_GET ->
	  Request.discard_request_body r;
	  (try Request.args r
	   with
	       Not_found -> "")
      | M_POST ->
	  get_post_body r
      | _ -> raise (HttpError cHTTP_METHOD_NOT_ALLOWED)
  in
  Cgi_args.parse s

(* multipart_args: parsing of the CGI arguments for multipart/form-data
   encoding *)

(* XXX RWMJ's note: slowly converting this file to use Pcre. *)
let boundary_re1 =
  Str.regexp_case_fold "boundary=\"\\([^\"]+\\)\""
let boundary_re2 =
  Str.regexp_case_fold "boundary=\\([^ \t\r\n]+\\)"
let name_re1 =
  Str.regexp_case_fold "name=\"\\([^\"]+\\)\""
let name_re2 =
  Str.regexp_case_fold "name=\\([^ \t\r\n;:]+\\)"
let filename_re1 =
  Str.regexp_case_fold "filename=\"\\([^\"]*\\)\""
let filename_re2 =
  Str.regexp_case_fold "filename=\\([^ \t\r\n;:]+\\)"
let content_type_re1 =
  Str.regexp_case_fold "Content-type:[ \t]*\"\\([^\"]+\\)\""
let content_type_re2 =
  Str.regexp_case_fold "Content-type:[ \t]*\\([^ \t\r\n;:]+\\)"
let separator_re =
  Str.regexp "\r\n\r\n"

let match_string re1 re2 str =
  try
    ignore(Str.search_forward re1 str 0); Str.matched_group 1 str
  with Not_found ->
    ignore(Str.search_forward re2 str 0); Str.matched_group 1 str

(* Extract field name and value from a chunk.  Raise Not_found if not
   a valid chunk. *)

type upload_data = {
  upload_value: string;
  upload_filename: string;
  upload_content_type: string
}

let extract_field chunk =
  let pos_separator = Str.search_forward separator_re chunk 0 in
  let header = String.sub chunk 0 pos_separator in
  let field_name = match_string name_re1 name_re2 header in
  let field_filename =
    try match_string filename_re1 filename_re2 header
    with Not_found -> "" in
  let field_content_type =
    try match_string content_type_re1 content_type_re2 header
    with Not_found -> "" in
  let beg_value = pos_separator + 4 in
  (* Chop final \r\n that browsers insist on putting *)
  let end_value =
    let len = String.length chunk in
    if len >= beg_value && String.sub chunk (len - 2) 2 = "\r\n"
    then len - 2
    else len in
  let field_value =
    String.sub chunk beg_value (end_value - beg_value) in
  (field_name, { upload_filename = field_filename;
                 upload_content_type = field_content_type;
                 upload_value = field_value })

(* Same, for a list of chunks *)

let rec extract_fields accu = function
  | [] -> 
      accu
  | chunk :: rem ->
      extract_fields
	(try extract_field chunk :: accu with Not_found -> accu)
	rem

(* Return true if the string starts with the prefix. *)
let string_starts_with s pref =
  String.length s >= String.length pref &&
  String.sub s 0 (String.length pref) = pref

(* Parse a multipart body. *)
let parse_multipart_args r =
  (* Determine boundary delimiter *)
  let content_type = Table.get (Request.headers_in r) "Content-Type" in
  let boundary =
    try
      match_string boundary_re1 boundary_re2 content_type
    with Not_found ->
      failwith ("Cgi: no boundary provided in " ^ content_type) in
  (* Extract the fields. *)
  extract_fields []
    (Str.split (Str.regexp_string ("--" ^ boundary))
       (get_post_body r))

let downconvert_upload_data fields =
  List.map (fun (name, field) -> name, field.upload_value) fields

let get_params r =
  let is_multipart =
    try
      let req_method = Request.method_number r
      and content_type = Table.get (Request.headers_in r) "Content-Type" in
      req_method = M_POST &&
      string_starts_with content_type "multipart/form-data"
    with
	Not_found -> false in

  if not is_multipart then
    parse_args r, [], false
  else (
    let uploads = parse_multipart_args r in
    let params = downconvert_upload_data uploads in
    params, uploads, true
  )

let get_cookies r =
  try
    let header = Table.get (Request.headers_in r) "Cookie" in
    Cookie.parse header
  with
      Not_found -> []

class cgi r =
  let params, uploads, is_multipart = get_params r in
  let cookies = get_cookies r in

  object (self)

    method private send_cookies ?cookie ?cookies is_redirect =
      let set_header =
	Table.add (if is_redirect then
		     Request.err_headers_out r
		   else
		     Request.headers_out r) "Set-Cookie" in
      may (fun cookie -> set_header cookie#as_string)
	(cookie : Cookie.cookie option);
      may (List.iter (fun cookie -> set_header cookie#as_string))
	(cookies : Cookie.cookie list option);

    method header ?(content_type = "text/html") ?cookie ?cookies () =
      Request.set_content_type r content_type;
      self#send_cookies ?cookie ?cookies false;
      Request.send_http_header r

    method template ?content_type ?cookie ?cookies
        (template : Template.template) =
      self#header ?content_type ?cookie ?cookies ();
      print_string r template#to_string

    method redirect ?cookie ?cookies url =
      self#send_cookies ?cookie ?cookies true;
      Table.set (Request.headers_out r) "Location" url;
      raise (HttpError cREDIRECT);
      ()

    (* Note that eventually we'll add some optional arguments to this method *)
    (* See apache/src/main/util_script.c and mod_cgi.c *)
    method url () =
      Request.uri r			(* Catch Not_found ? XXX *)

    method param name =
      snd (List.find (fun (this_name, _) -> name = this_name) params)

    method param_all name =
      List.map snd
	(List.filter (fun (this_name, _) -> name = this_name) params)

    method param_exists name =
      try
	ignore (self#param name); true
      with
	  Not_found -> false

    method param_true name =
      try
	let str = self#param name in str <> "" && str <> "0"
      with
	  Not_found -> false

    method params =
      params

    method is_multipart =
      is_multipart

    method upload name =
      snd (List.find (fun (this_name, _) -> name = this_name) uploads)

    method upload_all name =
      List.map snd
	(List.filter (fun (this_name, _) -> name = this_name) uploads)

    method cookie name =
      List.find (fun cookie -> cookie#name = name) cookies

    method cookies =
      cookies

    method request =
      r
  end

module StdPages = struct
  type button = {
    label : string;
    link : string;
    method_ : string option;
    params : (string * string) list;
  }

  let default_template =
    Template.template_from_string
      "<html>
<head>
<title>::title_html::</title>
<link rel=\"stylesheet\" href=\"/css/default.css\" type=\"text/css\">
</head>
<body bgcolor=\"#ffffff\">
::if(has_title)::
<h1>::title_html::</h1>
::end::
::if(has_icon)::
<img style=\"float: left; clear:both; margin:10px;\"
     alt=\"::icon_alt_html_tag::\"
     src=\"::icon_html_tag::\">
::end::
<p>::message_html::</p>
<hr>
<table><tr>
::if(has_back_button)::
<td><form><input type=\"button\" value=\"&lt;&lt; Go Back\"
                 onclick=\"history.go (-1)\"></form></td>
::end::
::if(has_close_button)::
<td><form><input type=\"button\" value=\"Close Window\"
                 onclick=\"top.close ()\"></form></td>
::end::
::table(buttons)::
<td><form method=\"::method_html_tag::\" action=\"::action_html_tag::\">
::table(params)::
<input type=\"hidden\" name=\"::name_html_tag::\" value=\"::value_html_tag::\">
::end::
<input type=\"submit\" value=\"::name_html_tag::\"></form></td>
::end::
</body>
</html>"

  let dialog ?cookie ?cookies
      ?(template = default_template) ?title ?icon ?icon_alt
      ?(back_button = true) ?(close_button = false) ?buttons
      (q : cgi) message =
    template#conditional "has_title" (title <> None);
    may (fun title -> template#set "title" title) title;
    template#conditional "has_icon" (icon <> None);
    may (fun icon -> template#set "icon" icon) icon;
    template#set "icon_alt" " ";
    may (fun icon_alt -> template#set "icon_alt" icon_alt) icon_alt;
    template#set "message" message;
    template#conditional "has_back_button" back_button;
    template#conditional "has_close_button" close_button;
    template#table "buttons" [];
    may
      (fun buttons ->
	 let buttons =
	   List.map
	     (fun button ->
		let params =
		  List.map
		    (fun (name, value) ->
		       [ "name", Template.VarString name;
			 "value", Template.VarString value;
		       ]) button.params in
		[ "method", Template.VarString (match button.method_ with
						    None -> "GET"
						  | Some m -> m);
		  "action", Template.VarString button.link;
		  "params", Template.VarTable params;
		  "name", Template.VarString button.label;
		]
	     ) buttons in
	 template#table "buttons" buttons
      ) buttons;
    q#template ?cookie ?cookies template

  let error ?cookie ?cookies ?template
      ?(title = "There was an error")
      ?(icon = "/caml-icons/error.png")
      ?(icon_alt = "Error")
      ?back_button ?close_button q message =
    dialog ?cookie ?cookies ?template ~title ~icon ?back_button ?close_button
      q message

  let ok ?cookie ?cookies ?template
      ?(title = "That operation was carried out successfully")
      ?(icon = "/caml-icons/ok.png")
      ?(icon_alt = "OK")
      ?(back_button = false)
      ?close_button ?buttons q message =
    dialog ?cookie ?cookies ?template ~title ~icon
      ~back_button ?close_button ?buttons
      q message

  end

module Sendmail =
struct

  open Unix

  exception Sendmail_failure of string

  let sendmail = ref "/usr/sbin/sendmail"
  let sendmail_args = ref "-t -i"
  let send () =
    let cmd = !sendmail ^ " " ^ !sendmail_args in
    open_process_out cmd

  let close chan =
    let status = close_process_out chan in
    match status with
	WEXITED 0 -> ()
      | WEXITED n ->
	  raise (Sendmail_failure (!sendmail ^ ": non-zero exit status: " ^
				   string_of_int n))
      | WSIGNALED n ->
	  raise (Sendmail_failure (!sendmail ^ ": killed by signal " ^
				   string_of_int n))
      | WSTOPPED n ->
	  raise (Sendmail_failure (!sendmail ^ ": stopped by signal " ^
				   string_of_int n))

  (* NB. 'to' is a reserved word. *)
  let send_mail ?subject ?to_addr ?cc ?bcc ?from ?content_type ?headers
    ~body () =
    let headers =
      match headers with
	  None -> []
	| Some hs -> hs in
    let headers =
      match subject with
	  None -> headers
	| Some s -> ("Subject", s) :: headers in
    let headers =
      match to_addr with
	  None -> headers
	| Some ss -> ("To", String.concat ", " ss) :: headers in
    let headers =
      match cc with
	  None -> headers
	| Some ss -> ("Cc", String.concat ", " ss) :: headers in
    let headers =
      match bcc with
	  None -> headers
	| Some ss -> ("Bcc", String.concat ", " ss) :: headers in
    let headers =
      match from with
	  None -> headers
	| Some s -> ("From", s) :: headers in
    let headers =
      match content_type with
	  None -> headers
	| Some s -> ("Content-Type", s) :: headers in

    (* Open a connection. *)
    let chan = send () in

    (* Send the headers. *)
    List.iter
      (fun (k, v) ->
	 output_string chan k;
	 output_string chan ": ";
	 output_string chan (Pcre.replace ~pat:"\n" ~templ:"\n\t" v);
	 output_char chan '\n'
      ) headers;

    (* Send an extra blank line. *)
    output_char chan '\n';

    (* Send the body. *)
    output_string chan body;

    (* Close connection. *)
    close chan

  end

(* Generic expires headers. *)
module Expires =
struct
  let short_weekday = function
    | 0 -> "Sun" | 1 -> "Mon" | 2 -> "Tue" | 3 -> "Wed"
    | 4 -> "Thu" | 5 -> "Fri" | 6 -> "Sat" | 7 -> "Sun"
    | _ -> invalid_arg "short_weekday"

  let short_month = function
    | 1 -> "Jan" | 2 -> "Feb" | 3 -> "Mar" | 4 -> "Apr"
    | 5 -> "May" | 6 -> "Jun" | 7 -> "Jul" | 8 -> "Aug"
    | 9 -> "Sep" | 10 -> "Oct" | 11 -> "Nov" | 12 -> "Dec"
    | _ -> invalid_arg "short_month"

  let make offset =
    let t = Unix.time () in
    let tm = Unix.gmtime (t +. float offset) in
    Printf.sprintf "%s, %02d %s %04d %02d:%02d:%02d GMT"
      (short_weekday tm.Unix.tm_wday)
      tm.Unix.tm_mday
      (short_month (tm.Unix.tm_mon + 1))
      (tm.Unix.tm_year + 1900)
      tm.Unix.tm_hour
      tm.Unix.tm_min
      tm.Unix.tm_sec

  let past () = make ((-5) * 60)
  let short () = make (5 * 60)
  let medium () = make 86400
  let long () = make (365 * 2 * 86400)

end

(* Generate a suitable random number for use in random cookies, session IDs,
 * etc.  These numbers are supposed to be very hard to predict.  Current
 * implementation requires /dev/urandom which may not exist on all platforms.
 *)
let random_sessionid () =
  let chan = open_in_bin "/dev/urandom" in
  let len = 16 in
  let sessionid = String.create (len * 2) in
  for i = 0 to len-1 do
    let hex = Printf.sprintf "%02x" (input_byte chan) in
    String.blit hex 0 sessionid (i*2) 2
  done;
  close_in chan;
  sessionid
