(** Apache interface for mod_caml programs.
  * Copyright (C) 2003 Merjis Ltd.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Library General Public
  * License as published by the Free Software Foundation; either
  * version 2 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  * Library General Public License for more details.
  *
  * You should have received a copy of the GNU Library General Public
  * License along with this library; if not, write to the Free
  * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  *
  * $Id: apache.mli,v 1.13 2004/11/05 11:39:20 rwmj Exp $
  *)

type result_type = OK | DECLINED | DONE

type method_type = M_GET
                 | M_PUT
		 | M_POST
		 | M_DELETE
		 | M_CONNECT
		 | M_OPTIONS
		 | M_TRACE
		 | M_PATCH
		 | M_PROPFIND
		 | M_PROPPATCH
		 | M_MKCOL
		 | M_COPY
		 | M_MOVE
		 | M_LOCK
		 | M_UNLOCK
		 | M_INVALID
(** Types of request methods. See [Request.method] and
 * [Request.method_number]. [M_HEAD] = [M_GET].
 *)

exception HttpError of int
(** Most Apache API functions which, in C, might return an integer indicating
 * an error, have been changed to throw this exception which contains the
 * error number or status number.
 *)

(* HTTP status codes: *)
val cHTTP_CONTINUE : int
val cHTTP_SWITCHING_PROTOCOLS : int
val cHTTP_PROCESSING : int
val cHTTP_OK : int
val cHTTP_CREATED : int
val cHTTP_ACCEPTED : int
val cHTTP_NON_AUTHORITATIVE : int
val cHTTP_NO_CONTENT : int
val cHTTP_RESET_CONTENT : int
val cHTTP_PARTIAL_CONTENT : int
val cHTTP_MULTI_STATUS : int
val cHTTP_MULTIPLE_CHOICES : int
val cHTTP_MOVED_PERMANENTLY : int
val cHTTP_MOVED_TEMPORARILY : int
val cHTTP_SEE_OTHER : int
val cHTTP_NOT_MODIFIED : int
val cHTTP_USE_PROXY : int
val cHTTP_TEMPORARY_REDIRECT : int
val cHTTP_BAD_REQUEST : int
val cHTTP_UNAUTHORIZED : int
val cHTTP_PAYMENT_REQUIRED : int
val cHTTP_FORBIDDEN : int
val cHTTP_NOT_FOUND : int
val cHTTP_METHOD_NOT_ALLOWED : int
val cHTTP_NOT_ACCEPTABLE : int
val cHTTP_PROXY_AUTHENTICATION_REQUIRED : int
val cHTTP_REQUEST_TIME_OUT : int
val cHTTP_CONFLICT : int
val cHTTP_GONE : int
val cHTTP_LENGTH_REQUIRED : int
val cHTTP_PRECONDITION_FAILED : int
val cHTTP_REQUEST_ENTITY_TOO_LARGE : int
val cHTTP_REQUEST_URI_TOO_LARGE : int
val cHTTP_UNSUPPORTED_MEDIA_TYPE : int
val cHTTP_RANGE_NOT_SATISFIABLE : int
val cHTTP_EXPECTATION_FAILED : int
val cHTTP_UNPROCESSABLE_ENTITY : int
val cHTTP_LOCKED : int
val cHTTP_FAILED_DEPENDENCY : int
val cHTTP_INTERNAL_SERVER_ERROR : int
val cHTTP_NOT_IMPLEMENTED : int
val cHTTP_BAD_GATEWAY : int
val cHTTP_SERVICE_UNAVAILABLE : int
val cHTTP_GATEWAY_TIME_OUT : int
val cHTTP_VERSION_NOT_SUPPORTED : int
val cHTTP_VARIANT_ALSO_VARIES : int
val cHTTP_INSUFFICIENT_STORAGE : int
val cHTTP_NOT_EXTENDED : int
val cDOCUMENT_FOLLOWS : int
val cPARTIAL_CONTENT : int
val cMULTIPLE_CHOICES : int
val cMOVED : int
val cREDIRECT : int
val cUSE_LOCAL_COPY : int
val cBAD_REQUEST : int
val cAUTH_REQUIRED : int
val cFORBIDDEN : int
val cNOT_FOUND : int
val cMETHOD_NOT_ALLOWED : int
val cNOT_ACCEPTABLE : int
val cLENGTH_REQUIRED : int
val cPRECONDITION_FAILED : int
val cSERVER_ERROR : int
val cNOT_IMPLEMENTED : int
val cBAD_GATEWAY : int
val cVARIANT_ALSO_VARIES : int

val is_http_info : int -> bool
(** Return true if status code is informational 1xx. *)
val is_http_success : int -> bool
(** Return true if status code is success 2xx. *)
val is_http_redirect : int -> bool
(** Return true if status code is redirect 3xx. *)
val is_http_error : int -> bool
(** Return true if status code is error 4xx or 5xx. *)
val is_http_client_error : int -> bool
(** Return true if status code is client error 4xx. *)
val is_http_server_error : int -> bool
(** Return true if status code is server error 5xx. *)

module Table :
  sig
    type t
    (** Apache [table] structure. *)

    external get : t -> string -> string = "mod_caml_table_get"
    (** [Table.get tbl key] returns the corresponding entry in the table.
     * The [key] is matched case-insensitively. Throws [Not_found] otherwise.
     *)
    external set : t -> string -> string -> unit = "mod_caml_table_set"
    (** [Table.set tbl key value] sets the [(key, value)] pair in the
      * table [tbl].
      *)
    external unset : t -> string -> unit = "mod_caml_table_unset"
    (** [Table.unset tbl key] removes the key from the table [tbl].
      *)
    external add : t -> string -> string -> unit = "mod_caml_table_add"
  end

module Server :
  sig
    type t
    (** Apache [server_rec] structure. *)

    external hostname : t -> string = "mod_caml_server_hostname"
    (** [server_rec] [hostname] field. Throws [Not_found] if NULL. *)
  end

module Connection :
  sig
    type t
    (** Apache [conn_rec] structure. *)

    external remote_ip : t -> string = "mod_caml_connection_remote_ip"
    (** [conn_rec] [remote_ip] field. Throws [Not_found] if NULL. *)

  end

module Request :
  sig
    type t
    (** Apache [request_rec] structure. *)

    type read_policy = REQUEST_NO_BODY
                     | REQUEST_CHUNKED_ERROR
		     | REQUEST_CHUNKED_DECHUNK
		     | REQUEST_CHUNKED_PASS
    (** For [Request.setup_client_block] call. *)

    external connection : t -> Connection.t = "mod_caml_request_connection"
    (** [request_rec] [connection] field. *)
    external server : t -> Server.t = "mod_caml_request_server"
    (** [request_rec] [server] field. *)
    external next : t -> t = "mod_caml_request_next"
    (** [request_rec] [next] field. Raises [Not_found] if NULL. *)
    external prev : t -> t = "mod_caml_request_prev"
    (** [request_rec] [prev] field. Raises [Not_found] if NULL. *)
    external main : t -> t = "mod_caml_request_main"
    (** [request_rec] [main] field. Raises [Not_found] if NULL. *)
    external the_request : t -> string = "mod_caml_request_the_request"
    (** [request_rec] [the_request] field. Throws [Not_found] if NULL. *)
    external assbackwards : t -> bool
	= "mod_caml_request_assbackwards"
    (** [request_rec] [assbackwards] field. *)

    external header_only : t -> bool = "mod_caml_request_header_only"
    (** [request_rec] [header_only] field. *)
    external protocol : t -> string = "mod_caml_request_protocol"
    (** [request_rec] [protocol] field. Throws [Not_found] if NULL. *)
    external proto_num : t -> int = "mod_caml_request_proto_num"
    (** [request_rec] [proto_num] field. *)
    external hostname : t -> string = "mod_caml_request_hostname"
    (** [request_rec] [hostname] field. Throws [Not_found] if NULL. *)
    external request_time : t -> float = "mod_caml_request_request_time"
    (** [request_rec] [request_time] field. *)
    external status_line : t -> string = "mod_caml_request_status_line"
    (** [request_rec] [status_line] field. Throws [Not_found] if NULL. *)
    external set_status_line : t -> string -> unit
      = "mod_caml_request_set_status_line"
    (** Set [request_rec] [status_line] field. *)
    external status : t -> int = "mod_caml_request_status"
    (** [request_rec] [status] field. *)
    external set_status : t -> int -> unit
      = "mod_caml_request_set_status"
    (** Set [request_rec] [status] field. *)

    external method_name : t -> string = "mod_caml_request_method"
    (** [request_rec] [method] field. *)
    external method_number : t -> method_type
	= "mod_caml_request_method_number"
    (** [request_rec] [method_number] field. *)

    external headers_in : t -> Table.t = "mod_caml_request_headers_in"
    (** [request_rec] [headers_in] field. *)
    external headers_out : t -> Table.t = "mod_caml_request_headers_out"
    (** [request_rec] [headers_out] field. *)
    external err_headers_out : t -> Table.t
	= "mod_caml_request_err_headers_out"
    (** [request_rec] [err_headers_out] field. *)
    external subprocess_env : t -> Table.t = "mod_caml_request_subprocess_env"
    (** [request_rec] [subprocess_env] field. *)
    external notes : t -> Table.t = "mod_caml_request_notes"
    (** [request_rec] [notes] field. *)
    external content_type : t -> string
      = "mod_caml_request_content_type"
    (** [request_rec] [content_type] field. Throws [Not_found] if NULL. *)
    external set_content_type : t -> string -> unit
      = "mod_caml_request_set_content_type"
    (** Set [request_rec] [content_type] field. *)

    external user : t -> string = "mod_caml_request_user"
    (** The authenticated user. In Apache 1.3 this field is actually in
      * the [conn_rec] structure, and was moved here in Apache 2.0. We
      * transparently hide this detail for you.
      * Throws [Not_found] if NULL. *)

    external uri : t -> string = "mod_caml_request_uri"
    (** [request_rec] [uri] field. Throws [Not_found] if NULL. *)
    external set_uri : t -> string -> unit = "mod_caml_request_set_uri"
    (** Set [request_rec] [uri] field. *)
    external filename : t -> string = "mod_caml_request_filename"
    (** [request_rec] [filename] field. Throws [Not_found] if NULL. *)
    external set_filename : t -> string -> unit
      = "mod_caml_request_set_filename"
    (** Set [request_rec] [filename] field. *)
    external path_info : t -> string = "mod_caml_request_path_info"
    (** [request_rec] [path_info] field. Throws [Not_found] if NULL. *)
    external set_path_info : t -> string -> unit
      = "mod_caml_request_set_path_info"
    (** Set [request_rec] [path_info] field. *)
    external args : t -> string = "mod_caml_request_args"
    (** [request_rec] [args] field. Throws [Not_found] if NULL. *)
    external set_args : t -> string -> unit
      = "mod_caml_request_set_args"
    (** Set [request_rec] [args] field. *)
    external finfo : t -> Unix.stats option = "mod_caml_request_finfo"
    (** [request_rec] [finfo] field. *)
    external send_http_header : t -> unit
      = "mod_caml_request_send_http_header"
    (** Send the HTTP headers. *)
    external rflush : t -> int
      = "mod_caml_request_rflush"
    (** Flush any buffered data waiting to be written to the client.
      * Returns [0] on success and [-1] for EOF.
      *)
    external setup_client_block : t -> read_policy -> unit
	= "mod_caml_request_setup_client_block"
    (** Setup for reading client request. *)
    external should_client_block : t -> bool
	= "mod_caml_request_should_client_block"
    (** Returns true if there is any client request data. *)
    external get_client_block : t -> string
	= "mod_caml_request_get_client_block"
    (** Get client request data. *)
    external discard_request_body : t -> unit
	= "mod_caml_request_discard_request_body"
    (** Discard client request body. *)
    external note_auth_failure : t -> unit
	= "mod_caml_request_note_auth_failure"
    (** Set headers to tell browser that authentication failed. *)
    external note_basic_auth_failure : t -> unit
	= "mod_caml_request_note_basic_auth_failure"
    (** Set headers to tell browser that basic authentication failed. *)
    external note_digest_auth_failure : t -> unit
	= "mod_caml_request_note_digest_auth_failure"
    (** Set headers to tell browser that digest authentication failed. *)
    external get_basic_auth_pw : t -> string option
	= "mod_caml_request_get_basic_auth_pw"
    (** Get the password sent in basic authentication. *)
    external internal_redirect : string -> t -> unit
	= "mod_caml_request_internal_redirect"
    (** Internally redirects immediately to [uri]. *)
    external internal_redirect_handler : string -> t -> unit
	= "mod_caml_request_internal_redirect_handler"
    (** Internally redirects immediately to [uri] using handler specified
      by [r]. *)
    external print_char : t -> char -> int = "mod_caml_request_print_char"
    (** Send a character back to the client.  Returns the character
      * written, cast to int, or [-1] on failure.
      *)
    external print_string : t -> string -> int
      = "mod_caml_request_print_string"
    (** Send a string back to the client.  Returns the number of bytes
      * actually written, which is smaller than the number of bytes in
      * the string if there was a failure.
      *)
    val print_int : t -> int -> int
    (** Send a decimal number back to the client. *)
    val print_float : t -> float -> int
    (** Send a floating-point number back to the client. *)
    val print_newline : t -> int
    (** Send a CR LF back to the client. *)
    val print_endline : t -> string -> int
    (** Send a string followed by CR LF back to the client. *)
    external register_cleanup : t -> (unit -> unit) -> unit
        = "mod_caml_request_register_cleanup"
    (** Register a cleanup function which is called when the current request
      * cycle ends.
      *)
  end

type handler_t = Request.t -> result_type
(** The type of handler functions. *)

val print_char : Request.t -> char -> int
(** Send a character back to the client. *)
val print_string : Request.t -> string -> int
(** Send a string back to the client. *)
val print_int : Request.t -> int -> int
(** Send a decimal number back to the client. *)
val print_float : Request.t -> float -> int
(** Send a floating-point number back to the client. *)
val print_newline : Request.t -> int
(** Send a CR LF back to the client. *)
val print_endline : Request.t -> string -> int
(** Send a string followed by CR LF back to the client. *)

module type DbiDriverT = sig
  type connection
  val connect : ?host:string -> ?port:string ->
    ?user:string -> ?password:string -> string ->
    connection
  val close : connection -> unit
  val closed : connection -> bool
  val commit : connection -> unit
  val ping : connection -> bool
  val rollback : connection -> unit
end

module type DbiPoolT = sig
  type connection
  val get : Request.t -> ?host:string -> ?port:string ->
    ?user:string -> ?password:string -> string -> connection
      (** [module MyPool = DbiPool (Dbi_postgres)]
	*
	* [let dbh = MyPool.get request "database_name"]
	*
	* Returns an unused [Dbi.connection] handle from the pool of database
	* handles. Or it may create a new database connection and return that.
	*
	* The parameters uniquely identify the database name (eg. "comments")
	* and optional parameters. Separate pools are maintained for each
	* combination of parameters.
	*
	* The connection is automatically returned to the pool at the end of
	* the current Apache request. After this time the connection may be
	* given away to another user. For this reason, the calling code must
	* NEVER stash the connection across requests (instead, call
	* [get] to get a new handle each time).
	*
	* On returning the handle to the pool, the pool performs a ROLLBACK
	* operation on the handle, thus losing any changes (INSERT, etc.)
	* made to the database. If the program wants to preserve changes, it
	* must perform a COMMIT operation itself, by calling
	* [Dbi.connection.commit].
        *)
end

module DbiPool (Dbi_driver : DbiDriverT) :
  (DbiPoolT with type connection = Dbi_driver.connection)
(** [module MyPool = DbiPool (Dbi_postgres)]
  *
  * creates a pool of PostgreSQL database handles.  To use them:
  *
  * [let dbh = MyPool.get r "database_name"]
  *
  * Gets you a new or recycled database handle [dbh] which is valid until
  * the end of the current Apache request.
  *)
