(** Library for writing CGIs *)
(*
 * Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
 *
 * Copyright (C) 1997 Daniel de Rauglaudre, INRIA
 *
 * Copyright (C) 1998 Jean-Christophe FILLIATRE
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: cgi.mli,v 1.28 2005/01/25 18:34:33 ChriS Exp $
 *)

module Cookie :
sig
  class cookie :
    name:string -> value:string -> domain:string ->
    max_age:int option -> expires:string -> path:string -> secure:bool ->
  object
    method name : string
      (** Return the name of the cookie. *)
    method value : string
      (** Return the value of the cookie. *)
    method max_age : int option
      (** See [#set_expires]. *)
    method expires : string
      (** Return the expires string of the cookie, or "" if not set. *)
    method domain : string
      (** Return the domain of the cookie, or "" if not set. *)
    method path : string
      (** Return the path of the cookie, or "" if not set. *)
    method secure : bool
      (** Return true if the cookie is a secure cookie. *)

    method set_name : string -> unit
      (** Set the name of the cookie. *)
    method set_value : string -> unit
      (** Set the value of the cookie. *)
    method set_max_age : int option -> unit
      (** [#set_max_age (Some s)] set the lifetime of the cookie to
	  [s] seconds [s].  [#set_max_age None] means that the cookie
	  will be discarded when the client browser exits. *)
    method set_expires : string -> unit
      (** @deprecated
	  [#set_expires s] sets the expires string of the cookie. [s]
	  is a date of the type ["Mon, 13 Dec 2004 17:25:30 GMT"]
	  which can be generated by one of the functions of the
	  {!Cgi.Expires} module below.  *)
    method set_domain : string -> unit
      (** Set the domain of the cookie. *)
    method set_path : string -> unit
      (** Set the path of the cookie. *)
    method set_secure : bool -> unit
      (** Set the cookie as a secure cookie.  Secure cookies are only
	  transmitted to HTTPS servers. *)

    method to_string : string
      (** Return the string representation of the cookie. *)
  end

  val cookie : ?max_age:int -> ?expires:string -> ?domain:string ->
    ?path:string -> ?secure:bool -> string -> string -> cookie
    (** [cookie ?expires ?domain ?path name value] creates a cookie
	with name [name] and value [value].

	@param max_age lifetime of the cookie in seconds (default: none).
	@param expires expiration date of the cookie (default: "").
	@param domain  domain of the cookie (default: "").
	@param path    path of the cookie (default: "").
	@param secure  whether the cookie is secure (default: [false]).
    *)

  val parse : string -> cookie list
    (** Parse zero or more cookies, normally from a Cookie: header. *)
end

module Expires :
sig
  val past : unit -> string
    (** Generate an HTTP 'Expires' header string in the past (in
	theory this forces caches along the way to remove content or a
	cookie to be deleted).  *)
  val short : unit -> string
    (** Generate an HTTP 'Expires' header string with a short expiry,
	now + 5 minutes.  This can typically be used for pages
	containing news which is updated frequently.  *)
  val medium : unit -> string
    (** Generate an HTTP 'Expires' header string with a medium expiry,
	now + 24 hours.  This can be used for content generated from
	a database which doesn't change much.  *)
  val long : unit -> string
    (** Generate an HTTP 'Expires' header string with a long expiry,
	now + 2 years.  This should be used for content which really
	will never change.  *)
  val make : int -> string
    (** [make s] generates an a date [s] seconds from now in fixed
	format (RFC 1123).  [s] may be negative.  The date format is
	suitable used for e.g. a HTTP 'Expires' header -- if [s < 0],
	it means the page expires in the past, and should be removed
	from content caches.  *)
end

val random_sessionid : unit -> string
  (** Generates a 128 bit (32 hex digits) random string which can be
      used for session IDs, random cookies and so on.  The string
      returned will be very random and hard to predict, if the OS
      supports this feature.  *)


(** Type of acceptable template objects. *)
class type template =
object
  method output : (string -> unit) -> unit
    (** [#output print] must use the [print] function to output the
	template (with the necessary substitutions done,...). *)
end

type upload_data = {
  upload_value: string;
  upload_filename: string;
  upload_content_type: string
}
(** Type returned by [cgi#upload] method. *)

(** [new cgi ?post_max r] creates a cgi object for the request [r].
    Note that you are advised not to create more than one cgi per
    request unless you know what you are doing.

    @param post_max set the maximum size for POSTed requests in
    bytes. This is a security feature to prevent clients from
    overrunning the server with data. The default is
    [Sys.max_string_length], meaning no limit (besides OCaml ones).
*)
class cgi : ?post_max:int -> Apache.Request.t ->
object
  method header : ?content_type:string -> ?cookie:Cookie.cookie ->
    ?cookies:Cookie.cookie list -> ?cookie_cache:bool -> unit -> unit
    (** Emit the header. The default content type is "text/html". *)

  method template : 'a. ?content_type:string -> ?cookie:Cookie.cookie ->
    ?cookies:Cookie.cookie list -> ?cookie_cache:bool ->
    (#template as 'a) -> unit
    (** Emit the header followed by the given template.  This is
	conceptually equivalent to: [q#header (); Request.print_string
	r template#to_string] if you use the provided {!Template} module.
	@raise Failure if the output is not successful. *)

  method exit : unit -> 'a
    (** Exit the current cgi script.  (You need not to call this at
	the end of your code, just if you want to exit earlier.)  *)

  method redirect : ?cookie:Cookie.cookie -> ?cookies:Cookie.cookie list ->
    ?cookie_cache:bool -> string -> 'a
    (** [#redirect ?cookie ?cookies url] quits the current cgi script
	and send to the client a redirection header to [url]. *)

  method url : unit -> string
    (** Return the URL of the script. *)

  method param : string -> string
    (** [#param name] returns the "first" value of the parameter [name].
	@raise Not_found if [name] does not designate a valid parameter. *)

  method param_all : string -> string list
    (** [#param_all name] returns all the values of the parameter [name].
	@raise Not_found if [name] does not designate a valid parameter. *)

  method param_exists : string -> bool
    (** Return true iff the named parameter exists. *)

  method param_true : string -> bool
    (** This method returns false if the named parameter is missing, is an
	empty string, or is the string ["0"]. Otherwise it returns true. Thus
	the intent of this is to return true in the Perl sense of the word.
	If a parameter appears multiple times, then this uses the first
	definition and ignores the others. *)

  method params : (string * string) list
    (** Return an assoc-list mapping name -> value for all parameters.
	Note that CGI scripts may have multiple values for a single name. *)

  method is_multipart : bool
    (** Returns true iff the request was a [multipart/form-data]
	[POST] request.  Such requests are used when you need to
	upload files to a CGI script.  *)

  method upload : string -> upload_data
    (** For multipart forms only.  Returns the full upload data passed
	by the browser for a particular parameter.
	@raise Not_found is no such parameter exists.  *)

  method upload_all : string -> upload_data list
    (** For multipart forms only.  As for [#upload], but returns all
	uploads.  *)

  method cookie : string -> Cookie.cookie
    (** Return the named cookie, or throw [Not_found]. *)

  method cookies : Cookie.cookie list
    (** Return a list of all cookies passed to the script. *)

  method log : string -> unit
    (** [log s] Log the message [s] into the webserver log. *)

  method request : Apache.Request.t
    (** Returns the original request object (passed in the constructor). *)
end

module Cgi_args :
sig
  val parse : string -> (string * string) list
    (** [parse qs] parses up a standard CGI query string (such as
	["a=1&b=2"]), and returns the list of name, value pairs.
	This is a helper function.  The [cgi] class does this parsing
	for you, so you shouldn't need to use this function unless
	you're doing something out of the ordinary.  *)

  val make : (string * string) list -> string
    (** [make bindings] returns a query string from the list
	[bindings] of name, value pairs.  For example, [make
	[("a", "1"); ("b", "2")]] returns ["a=1&b=2"]. *)
end

module Sendmail :
sig
  exception Failure of string
    (** This exception may be thrown by any of the functions in this
	module, to indicate some problem running sendmail.  *)

  val sendmail : string ref
    (** This contains the path to sendmail (or the binary which acts
	like sendmail, as in the case of exim).  Normally this is set
	to the correct value for your OS, eg. ["/usr/sbin/sendmail"].
    *)
  val sendmail_args : string ref
    (** This contains the arguments passed to sendmail, normally ["-t
	-i"].  You could add the ["-f"] option to specify the sender
	email address, provided that the current user is one of
	sendmail's trusted users.  *)

  val send : unit -> out_channel
    (** Begin sending an email.  This returns a channel on which you
	should write the mail headers, followed by a blank line,
	followed by the message body.  After writing the mail you
	must call {!Cgi.Sendmail.close}.

	[send] does not perform any sort of processing or escaping on
	the message.  *)

  val close : out_channel -> unit
    (** Close the output channel.  You must call this on the channel
	returned from {!Cgi.Sendmail.send}.  *)

  val send_mail : ?subject:string ->
    ?to_addr:string list -> ?cc:string list -> ?bcc:string list ->
    ?from:string ->
    ?content_type:string ->
    ?headers:(string * string) list ->
    string -> unit
    (** This is a less flexible, but simpler interface to sending
	mail.  You specify, optionally, Subject line, To, Cc, Bcc,
	From and Content-Type headers, and a body, and a mail is
	generated.

	[send_mail] will correctly escape headers, provided they
	are in strict 7 bit US-ASCII.  Its behaviour on non-7 bit
	sequences is currently undefined (and probably wrong).
	[send_mail] will not process or escape any part of the body.
    *)
end

