(** Library for building web pages from templates. *)
(*
 * Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: template.mli,v 1.8 2005/01/05 19:26:43 ChriS Exp $
 *)

type var_t =
  | VarString of string				(** ::tag:: *)
  | VarTable of table_row_t list		(** ::table(tag):: *)
  | VarConditional of bool			(** ::if(tag):: *)
  | VarCallback of (string list -> string)	(** ::call(f, x1,...):: *)
and table_row_t = (string * var_t) list

(** Variables are either simple string, tables, conditionals or
    callbacks.

    A simple string is set with [template#set "name" s] where [s] will
    be automatically escaped depending on the declaration in the template:

    - [::name::] does no escaping;
    - [::name_url::] escapes for URL encoding, make it suitable in a
        link [<a href="::name_url::">];
    - [::name_html::] escapes for HTML display of the string;
    - [::name_html_tag::] escapes the string to make it suitable to be
        placed between quotes in an HTML tag, e.g.
        [<input value="::name_html_tag::">];
    - [::name_html_textarea::] escapes the string to make it suitable
        to be placed between [<textarea>...</textarea>].

    See {!Cgi_escape.escape} for more information.

    Tables are declared in the template by [::table(name)::] {i row
    template} [::end::].  The {i row template} can contain other
    variables.  Calling [template#table "name" rows], where [rows] is
    a list [[row1, row2,...,rowN]], will insert [N] {i row templates}
    with each template having its variables set thanks to [row1],...
    each of which is an associative list name -> value (of type
    {!Template.table_row_t}).

    Conditionals are declared in the template by [::if(name)
    .. ::else:: .. ::end::] with the "else" clause being optional.
    Calling [template#conditional] sets up a conditional value.

    Calling [template#callback "fname" f] sets up the callback
    function declared by [::call(fname,arg1,...,argN)::] replacing the
    call by the value of [f] applied to the list [[arg1,...,argN]].
    The string returned by [f] can be escaped by using suffices in the
    template as for simple tags: [::call(fname,arg1,...,argN)_html::],...

    A template may also include other templates with
    [::include(filename)::].
*)


(** [new template ?filename tpl] computes a new template from the
    string [tpl].
    @param filename if set, it is used to determine the base path for
    [::include()::] tags in the template (default: current directory).  *)
class template : ?filename:string -> string ->
object
  method set : string -> string -> unit
    (** Set a variable in the template. *)

  method table : string -> table_row_t list -> unit
    (** Set a table in the template. *)

  method conditional : string -> bool -> unit
    (** Set a conditional in the template. *)

  method callback : string -> (string list -> string) -> unit
    (** Set a callback in the template. *)

  method to_string : string
    (** Return the template as a string. *)

  method to_channel : out_channel -> unit
    (** Write the template to a channel. *)

  method output :(string -> unit) -> unit
  (** [output out] outputs the template, calling [out s] for each
      write of a string [s]. *)

  method source : string
    (** Return the original source code for the template. *)
end

val template : string -> template
  (** Compile the template from a named file. *)

val template_from_string : ?filename:string -> string -> template
  (** Compile the template from a literal string. *)

val template_from_channel : ?filename:string -> in_channel -> template
  (** Compile the template from a channel. *)





module StdPages :
sig
  type button = {
    label : string;
    link : string;
    method_ : string option;
    params : (string * string) list;
  }

  val dialog : ?cookie:Cgi.Cookie.cookie ->
    ?cookies:Cgi.Cookie.cookie list ->
    ?css_url:string ->
    ?template:template ->
    ?title:string ->
    ?icon:string ->
    ?icon_alt:string ->
    ?back_button:bool ->
    ?close_button:bool ->
    ?buttons:button list ->
    Cgi.cgi -> string -> unit
    (** [StdPages.dialog] is used to generate a standard dialog box,
	which can be variously used for asking questions, displaying
	error messages, presenting confirmation interstitial pages,
	etc. (Note that there are specialised versions of this
	function for OK messages and error pages. See
	{!Template.StdPages.ok} and {!Template.StdPages.error} for
	simple functions for these tasks.)

	A dialog looks like this (in left-to-right locales such as US
	english):

	{v
	+--------------------------------------------------+
	|                                                  |
	|             TITLE .............................  |
        | +--------+                                       |
        | | ICON   |                                       |
        | |        |  MESSAGE                              |
        | |        |                                       |
        | +--------+                                       |
	|            +--------+ +--------+ +--------+      |
	|            | BUTTON | | BUTTON | | BUTTON |      |
	|            +--------+ +--------+ +--------+      |
	|                                                  |
	+--------------------------------------------------+
	v}

	Only the message element is required. All other elements are
	optional (even the buttons, although it won't be a very usable
	dialog if there are no buttons at all).

	The optional parameters control which elements are displayed:

	@param title Sets the title element.
	@param icon Sets the path to the icon element.
	@param icon_alt sets the ALT text of the icon.

	@param back_button Controls whether a Javascript back button is
	                   displayed (default: [true]).
	@param close_button Controls whether a Javascript close window
	                    button is displayed (default: [false]).
	@param buttons is a list of buttons to display. (Each button
	               has type {!Template.StdPages.button}).

	@param css_url allows to select a URL for the CSS file (by default
	               the embedded CSS is used).

	@param template can be used to override the template used. The
	{!Cgi} library comes with a suitable default template, but you
	can override it if you want one which is more suitable for the
	style of your website.  *)

  val error : ?cookie:Cgi.Cookie.cookie ->
    ?cookies:Cgi.Cookie.cookie list ->
    ?css_url:string ->
    ?template:template ->
    ?title:string ->
    ?icon:string ->
    ?icon_alt:string ->
    ?back_button:bool ->
    ?close_button:bool ->
    Cgi.cgi -> string -> unit
    (** This is a limited form of {!Template.StdPages.dialog} which
	displays a simple error page.  *)

  val ok : ?cookie:Cgi.Cookie.cookie ->
    ?cookies:Cgi.Cookie.cookie list ->
    ?css_url:string ->
    ?template:template ->
    ?title:string ->
    ?icon:string ->
    ?icon_alt:string ->
    ?back_button:bool ->
    ?close_button:bool ->
    ?buttons:button list ->
    Cgi.cgi -> string -> unit
    (** This is a limited form of {!Template.StdPages.dialog} which
	displays a simple ok/success page.  *)
end
