/*  Moe - My Own Editor
    Copyright (C) 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <algorithm>
#include <string>
#include <vector>

#include "buffer.h"
#include "iso_8859.h"


namespace {

void insert_lines( std::vector< std::string > & data,
                   const int line, const int lines ) throw()
  {
  if( lines > 0 )
    {
    int i = data.size();
    data.resize( i + lines );
    while( --i >= line ) data[i+lines].swap( data[i] );
    }
  }


void delete_lines( std::vector< std::string > & data,
                   const int line, const int lines ) throw()
  {
  if( lines > 0 && line < (int)data.size() )
    {
    for( unsigned int i = line + lines; i < data.size(); ++i )
      data[i-lines].swap( data[i] );
    const int len = std::min( lines, (int)data.size() - line );
    data.erase( data.end() - len, data.end() );
    }
  }

} // end namespace


void Basic_buffer::add_line( const char * buf, const int len ) throw()
  {
  data.back().append( buf, len );
  if( data.back().size() && data.back()[data.back().size()-1] == '\n' )
    data.push_back( std::string() );
  }


     // If all lines end with CR-LF, assume DOS file
bool Basic_buffer::detect_and_remove_cr() throw()
  {
  if( lines() <= 1 ) return false;
  for( int i = 0; i < last_line(); ++i )
    {
    const std::string & line = data[i];
    const int size = line.size();
    if( size < 2 || line[size-2] != '\r' ) return false;
    }
  for( int i = 0; i < last_line(); ++i )
    data[i].erase( data[i].size() - 2, 1 );
  return true;
  }


Basic_buffer::Basic_buffer( const Basic_buffer & b, const Point & p1, const Point & p2 ) throw()
  : data( 1 )
  {
  Point p = bof();
  pputb( p, b, p1, p2 );
  }


int Basic_buffer::characters( const int line ) const throw()
  {
  if( line < 0 || line >= lines() ) return -1;
  return data[line].size();
  }


int Basic_buffer::operator[]( const Point & p ) const throw()
  {
  if( p.col < 0 || p.col >= characters( p.line ) ) return -1;
  unsigned char ch = data[p.line][p.col];
  return ch;
  }


int Basic_buffer::pgetc( Point & p ) const throw()
  {
  Point p1 = p;
  if( !pisvalid( p ) || !pnext( p ) ) return -1;
  unsigned char ch = data[p1.line][p1.col];
  return ch;
  }


int Basic_buffer::pngetc( Point & p ) const throw()
  {
  if( !pnext( p ) || p == eof() ) return -1;
  unsigned char ch = data[p.line][p.col];
  return ch;
  }


int Basic_buffer::ppgetc( Point & p ) const throw()
  {
  if( !pprev( p ) ) return -1;
  unsigned char ch = data[p.line][p.col];
  return ch;
  }


bool Basic_buffer::pnext( Point & p ) const throw()
  {
  if( ++p.col < characters( p.line ) || p == eof() ) return true;
  if( ++p.line < lines() ) { p.col = 0; return true; }
  --p.line; --p.col; return false;
  }


bool Basic_buffer::pprev( Point & p ) const throw()
  {
  int c = characters( p.line );
  if( c >= 0 && p.col > c ) p.col = c;
  if( --p.col >= 0 ) return true;
  int col = characters( --p.line ) - 1;
  if( col >= 0 ) { p.col = col; return true; }
  ++p.line; ++p.col; return false;
  }


bool Basic_buffer::pseek( Point & p, int n ) const throw()
  {
  Point saved_p = p;
  p.col += n;
  if( n > 0 )
    {
    while( p.line < last_line() )
      {
      int c = characters( p.line );
      if( p.col < c || p == eof() ) break;
      p.col -= c; ++p.line;
      }
    if( p.col >= 0 && ( p.line < last_line() || p <= eof() ) ) return true;
    }
  else if( n < 0 )
    {
    while( p.line > 0 && p.col < 0 ) p.col += characters( --p.line );
    if( p.line >= 0 && p.col >= 0 ) return true;
    }
  else return true;	// n == 0
  p = saved_p; return false;
  }


bool Basic_buffer::pvalid( Point & p ) const throw()
  {
  if( p < bof() ) { p = bof(); return false; }
  if( p > eof() ) { p = eof(); return false; }
  if( p.col > 0 && p.col >= (int)data[p.line].size() )
    { p.col = data[p.line].size() - 1; return false; }
  if( p.col < 0 ) { p.col = 0; return false; }
  return true;
  }


// insert block b.[p1,p2) at p and move next
bool Basic_buffer::pputb( Point & p, const Basic_buffer & b, const Point & p1, const Point & p2 ) throw()
  {
  if( !pisvalid( p ) || !b.pisvalid( p1 ) || !b.pisvalid( p2 ) || p1 >= p2 )
    return false;
  const int lines = p2.line - p1.line;
  if( lines == 0 )
    {
    data[p.line].insert( p.col, b.data[p1.line], p1.col, p2.col - p1.col );
    p.col += p2.col - p1.col;
    }
  else
    {
    insert_lines( data, p.line + 1, lines );
    if( p2.col > 0 ) data[p.line+lines].assign( b.data[p2.line], 0, p2.col );
    data[p.line+lines].append( data[p.line], p.col, data[p.line].size() - p.col );
    data[p.line].erase( p.col, data[p.line].size() - p.col );
    data[p.line].append( b.data[p1.line], p1.col, b.data[p1.line].size() - p1.col );
    for( int i = 1; i < lines; ++i )
      data[p.line+i].assign( b.data[p1.line+i] );
    p.line += lines; p.col = p2.col;
    }
  return true;
  }


// insert char and move next
bool Basic_buffer::pputc( Point & p, const unsigned char ch ) throw()
  {
  if( !pisvalid( p ) ) return false;
  data[p.line].insert( data[p.line].begin() + p.col, ch );
  ++p.col;
  if( ch == '\n' )
    {
    if( p == eof() ) data.push_back( std::string() );
    else
      {
      insert_lines( data, p.line + 1, 1 );
      data[p.line+1].assign( data[p.line], p.col, data[p.line].size() - p.col );
      data[p.line].erase( p.col, data[p.line].size() - p.col );
      }
    ++p.line; p.col = 0;
    }
  return true;
  }


// change char and move next
bool Basic_buffer::pchgc( Point & p, const unsigned char ch ) throw()
  {
  int ch1 = (*this)[p];
  if( ch == '\n' || ch1 < 0 || ch1 == '\n' ) return false;
  data[p.line][p.col++] = ch;
  return true;
  }


// delete block at [p1,p2)
bool Basic_buffer::pdelb( const Point & p1, const Point & p2 ) throw()
  {
  if( p1 >= p2 || !pisvalid( p1 ) || !pisvalid( p2 ) ) return false;
  const int lines = p2.line - p1.line;
  if( lines == 0 ) data[p1.line].erase( p1.col, p2.col - p1.col );
  else
    {
    if( p2.col > 0 ) data[p2.line].erase( 0, p2.col );
    if( p1.col == 0 ) delete_lines( data, p1.line, lines );
    else
      {
      data[p1.line].erase( p1.col, data[p1.line].size() - p1.col );
      data[p1.line].append( data[p2.line] );
      delete_lines( data, p1.line + 1, lines );
      }
    }
  return true;
  }


// Is p at beginning of word?
bool Basic_buffer::pisbow( const Point & p ) const throw()
  {
  Point p1 = p;
  int ch = (*this)[p];
  return ( ch >= 0 && ISO_8859::isalnum_( ch ) &&
           ( p == bol( p ) || !ISO_8859::isalnum_( ppgetc( p1 ) ) ) );
  }


// Is p at end of word?
bool Basic_buffer::piseow( const Point & p ) const throw()
  {
  Point p1 = p;
  int ch = ppgetc( p1 );
  return ( ch >= 0 && ISO_8859::isalnum_( ch ) &&
           ( p == eol( p ) || !ISO_8859::isalnum_( (*this)[p] ) ) );
  }


Point Basic_buffer::bol( const Point & p ) const throw()
  {
  if( p.line >= lines() ) return eof();
  else if( p.line < 0 ) return bof();
  else return Point( p.line, 0 );
  }


Point Basic_buffer::eol( const Point & p ) const throw()
  {
  if( p.line >= last_line() ) return eof();
  else if( p.line < 0 ) return eol( bof() );
  else return Point( p.line, data[p.line].size() - 1 );
  }


// returns position of first non whitespace character in line (eol if none)
Point Basic_buffer::bot( const Point & p ) const throw()
  {
  Point p1 = bol( p );
  const std::string & s = data[p1.line];
  while( p1.col < (int)s.size() && std::isspace( s[p1.col] ) && s[p1.col] != '\n' )
    ++p1.col;
  return p1;
  }


// returns position following last non whitespace character in line (bol if none)
Point Basic_buffer::eot( const Point & p ) const throw()
  {
  Point p1 = eol( p );
  const std::string & s = data[p1.line];
  while( p1.col > 0 && std::isspace( s[p1.col-1] ) ) --p1.col;
  return p1;
  }

// begin of paragraph
Point Basic_buffer::bop( const Point & p ) const throw()
  {
  Point p1 = p;
  if( !blank( p1.line ) ) while( !blank( p1.line - 1 ) ) --p1.line;
  return bol( p1 );
  }

// end of paragraph
Point Basic_buffer::eop( const Point & p ) const throw()
  {
  Point p1 = p;
  while( !blank( p1.line ) ) ++p1.line;
  return bol( p1 );
  }


// Move 'p' to the matching delimiter.
// If (force && forward), fail if [p] is closing delimiter.
// If (force && !forward), fail if [p] is opening delimiter.
//
bool Basic_buffer::set_to_matching_delimiter( Point & p, bool forward,
                                              const bool force ) const throw()
  {
  int ch1 = (*this)[p];
  if( ch1 < 0 ) return false;
  unsigned char ch2;
  const bool forward_orig = forward;
  bool comment = false;
  switch( ch1 )
    {
    case '(': ch2 = ')'; forward = true; break;
    case '[': ch2 = ']'; forward = true; break;
    case '{': ch2 = '}'; forward = true; break;
    case '<': ch2 = '>'; forward = true; break;
    case ')': ch2 = '('; forward = false; break;
    case ']': ch2 = '['; forward = false; break;
    case '}': ch2 = '{'; forward = false; break;
    case '>': ch2 = '<'; forward = false; break;
    case '/': ch2 = '*'; forward = true; comment = true; break;
    case '*': ch2 = '/'; forward = false; comment = true; break;
    case '"':
    case '\'':
    case '`': ch2 = ch1; break;
    default: return false;
    }
  if( force && ( forward != forward_orig ) ) return false;

  if( comment )
    {
    Point p1 = p;
    int ch = pngetc( p1 );
    if( ch >= 0 && ch == ch2 )
      {
      if( forward )
        { if( find_text( p1, std::string( "*/" ) ) )
          { pseek( p1, -2 ); p = p1; return true; } }
      else
        { if( rfind_text( p1, std::string( "/*" ) ) )
          { p = p1; return true; } }
      }
    return false;
    }

  Point p1 = p;
  int delta = ( forward ? +1 : -1 );
  int level = 0;	// Nesting level of delimiters
  char quote = 0;	// Ignore quoted delimiters

  while( pseek( p1, delta ) )
    {
    int ch = (*this)[p1]; if( ch < 0 ) break;
    if( !quote )
      {
      if( ch == ch2 ) { if( --level < 0 ) { p = p1; return true; } }
      else if( ch == ch1 ) { ++level; continue; }
      }

    // ignore escaped characters
    if( ch == '\\' ) { if( forward ) { pnext( p1 ); } continue; }

    // ignore single-quoted characters
    if( ch == '\'' )
      {
      Point pt = p1; char t1, t2;
      if( forward ) { t1 = pngetc( pt ) ;t2 = pngetc( pt ); }
      else { t1 = ppgetc( pt ) ;t2 = ppgetc( pt ); }
      if( t2 == '\'' && ( t1 == ch1 || t1 == ch2 ) ) p1 = pt;
      continue;
      }

    // ignore double-quoted text
    if( ch == '"' )
      {
      if( !forward )
        {
        Point pt = p1; int cht = ppgetc( pt );
        if( cht == '\\' ) continue;		// ignore escaped quotes
        }
      if( !quote ) quote = ch; else if( quote == ch ) quote = 0;
      }
    }
  return false;
  }


// Search forward from 'p' for 'pattern' (Boyer-Moore algorithm)
// If found, set 'p' after found substring
//
bool Basic_buffer::find_text( Point & p, const std::string & text, const bool icase ) const throw()
  {
  if( text.size() == 0 ) return true;
  Point p1 = p;
  if( !pseek( p1, text.size() ) ) return false;
  const std::string * textp = &text;
  std::string itext;

  if( icase )
    {
    for( unsigned int i = 0; i < text.size(); ++i )
      itext += ISO_8859::tolower( text[i] );
    textp = &itext;
    }

  std::vector< int > table( 256, textp->size() );
  for( unsigned int i = 0; i < textp->size() - 1; ++i )
    { unsigned char ch = (*textp)[i]; table[ch] = textp->size() - i - 1; }

  while( true )
    {
    Point p2 = p1;
    int i, delta = 0;
    for( i = textp->size() - 1; i >= 0; --i )
      {
      unsigned char ch = (*textp)[i];
      int ch2 = ppgetc( p2 ); if( ch2 < 0 ) return false;
      if( icase ) ch2 = ISO_8859::tolower( ch2 );
      if( !delta ) delta = table[ch2];
      if( ch != ch2 ) break;
      }
    if( i < 0 ) { p = p1; return true; }
    if( !pseek( p1, delta ) ) return false;
    }
  }


// Search backward from 'p-1' for 'pattern' (Boyer-Moore algorithm)
// If found, set 'p' to the first char of found substring
//
bool Basic_buffer::rfind_text( Point & p, const std::string & text, const bool icase ) const throw()
  {
  if( text.size() == 0 ) return true;
  Point p1 = p;
  if( !pseek( p1, -text.size() ) ) return false;
  const std::string * textp = &text;
  std::string itext;

  if( icase )
    {
    for( unsigned int i = 0; i < text.size(); ++i )
      itext += ISO_8859::tolower( text[i] );
    textp = &itext;
    }

  std::vector< int > table( 256, textp->size() );
  for( unsigned int i = textp->size() - 1; i > 0; --i )
    { unsigned char ch = (*textp)[i]; table[ch] = i; }

  while( true )
    {
    Point p2 = p1;
    unsigned int i; int delta = 0;
    for( i = 0; i < textp->size(); ++i )
      {
      unsigned char ch = (*textp)[i];
      int ch2 = pgetc( p2 ); if( ch2 < 0 ) return false;
      if( icase ) ch2 = ISO_8859::tolower( ch2 );
      if( !delta ) delta = table[ch2];
      if( ch != ch2 ) break;
      }
    if( i >= textp->size() ) { p = p1; return true; }
    if( !pseek( p1, -delta ) ) return false;
    }
  }


Point Basic_buffer::to_cursor( const Point & pointer ) const throw()
  {
  Point p = bol( pointer );
  Point cursor = p;
  while( p.col < pointer.col )
    {
    int ch = (*this)[p];
    if( ch < 0 ) break;
    if( ch == '\t' ) cursor.col += 8 - ( cursor.col % 8 );
    else ++cursor.col;
    ++p.col;
    }
  return cursor;
  }


Point Basic_buffer::to_pointer( const Point & cursor, Point & pcursor ) const throw()
  {
  pcursor = bol( cursor );
  Point pointer = pcursor;
  const int peolcol = eol( pointer ).col;
  while( pcursor.col < cursor.col && pointer.col < peolcol )
    {
    int ch = (*this)[pointer];
    if( ch < 0 ) break;
    if( ch == '\t' )
      {
      int col = pcursor.col + 8 - ( pcursor.col % 8 );
      if( col <= cursor.col ) pcursor.col = col; else break;
      }
    else ++pcursor.col;
    ++pointer.col;
    }
  return pointer;
  }


int Basic_buffer::to_string( const Point & p1, const Point & p2,
                             std::string & text ) const throw()
  {
  text.clear();
  if( p1 < p2 && pisvalid( p1 ) && pisvalid( p2 ) )
    {
    if( p1.line == p2.line )
      text.assign( data[p1.line], p1.col, p2.col - p1.col );
    else
      {
      int line = p1.line;
      text.assign( data[line], p1.col, data[line].size() - p1.col );
      while( ++line < p2.line ) text += data[line];
      if( p2.col ) text.append( data[line], 0, p2.col );
      }
    }
  return text.size();
  }
